
from __future__ import absolute_import

import time
import unittest

from stpy.fabric.enums import DataspaceType, SLFileSessionContext, HTTPFabricConnectionState

from stpy.fabric.utils import Utils

import stpy
from stpy import HTTPFabricConnection

class TestHTTPFabricConnection(unittest.TestCase):
    def setUp(self):
        Utils.initLogs()
        pass

    def tearDown(self):
        pass

    def testOpen(self):
        connection = self.openConnection()

        self.assertEqual(connection.isOpened(), True)

        self.assertEqual(connection.ping(), "AVAILABLE")

        connection.close()
        self.assertEqual(connection.isOpened(), False)
        s = connection.ping()
        self.assertEqual(s, "UNAVAILABLE")

    def testImportSemanticType(self):
        connection = self.openConnection()

        connection.importSemanticType("MyComplexObject");

        object = connection.getTypeFactory().instantiateType("MyComplexObject")
        objectMap = connection.getTypeFactory().writeToMap(object)
        object1 = connection.getTypeFactory().readFromMap(objectMap)

        object = connection.getTypeFactory().instantiateType("RowSet")

        connection.close()
        pass

    def testReconnect(self):
        connection = self.openConnection()

        self.assertEqual(connection.isOpened(), True)
        self.assertEqual(connection.ping(), "AVAILABLE")

        # dataspace accessor
        accessor = connection.createDataspaceAccessor(None, DataspaceType.TSPACE, "swagger_tspace")
        accessor.setSessionContext(SLFileSessionContext.SERVER)
        accessor.setRequestTimeout(1234567)
        accessor.setAutoCommit(False)
        accessor.setAutoSwitching(False)

        def checkDataspaceAccesor(self, accessor):
            self.assertEqual(accessor.getSessionContext(), SLFileSessionContext.SERVER)
            self.assertEqual(accessor.getRequestTimeout(), 1234567)
            self.assertEqual(accessor.getAutoCommit(), False)
            self.assertEqual(accessor.autoSwitching(), False)

            self.assertEqual(accessor.isOpened(), True)
            self.assertEqual(accessor.isAvailable(), True)

            rowSet = accessor.executeQuery("list collections")
        checkDataspaceAccesor(self, accessor)

        # slang session
        session = connection.createSLSession()
        def checkSLSession(self, session):
            response = session.slangRequest("list nodes")
            self.assertEqual(response.isOK, True)
        checkSLSession(self, session)

        listener = TestListener()
        connection.addStateListener(listener)
        self.assertEqual(len(listener.events), 0)

        # restart acceptor
        try:
            response = session.slangRequest("restart acceptor http.Default")
        except Exception as e:
            pass
        Utils.sleep(time.time(), 5000)
        self.assertTrue(len(listener.events) > 0)
        self.assertTrue(listener.events[len(listener.events) - 1].state, HTTPFabricConnectionState.SERVER_CONNECTION_REPAIRED)

        self.assertEqual(connection.ping(), "AVAILABLE")

        # check dataspace accessor
        checkDataspaceAccesor(self, accessor)

        # check slang session
        checkSLSession(self, session)

        connection.close()

    def openConnection(self):
        connection = HTTPFabricConnection("localhost:8888", "Admin", "Admin")
        connection.open()
        self.assertEqual(connection.isOpened(), True)
        self.assertEqual(connection.ping(), "AVAILABLE")
        return connection


class TestListener(object):
    def __init__(self):
        self.events = []

    def onStateEvent(self, event):
        self.events.append(event)


if __name__ == '__main__':
    unittest.main()
