from __future__ import absolute_import

import unittest
import decimal
import collections

from datetime import datetime, date, time
import pytz # $ pip install pytz
from tzlocal import get_localzone # $ pip install tzlocal

from stpy.fabric.utils import Utils

import stpy
from stpy import HTTPFabricConnection
from stpy import DataspaceType

class TestHTTPDataspaceAccessor(unittest.TestCase):
    def setUp(self):
        Utils.initLogs()
        pass

    def tearDown(self):
        pass

    def testTspaceTableWithSimpleTypes(self):
        connection = self.openConnection()
        accessor = self.openAccessorTspace(connection)

        #     "  id int, charColumn2 char(3), varcharColumn3 varchar(100), longvarcharColumn4 longvarchar(100)" +
        #     ", nvarcharColumn5 nvarchar(100), stringColumn6 string" +
        #     ", dateColumn7 sqldate, timeColumn8 sqltime, timestampColumn9 sqltimestamp" +
        #     ", tinyintColumn10 tinyint, smallintColumn11 smallint, integerColumn12 integer, bigintColumn13 bigint, realColumn14 real, doubleColumn15 double, floatColumn16 float" +
        #     ", numericColumn17 numeric(11, 6), decimalColumn18 decimal(10, 5), booleanColumn19 boolean" +
        #     ", binaryColumn20 binary(3), varbinaryColumn21 varbinary(100), clobColumn22 clob, blobColumn23 blob(10), bitColumn24 bit" +
        #     ", arrayColumn25 int array[10] default null, otherColumn26 other, eventColumn27 event" +
        #     ", charColumn28 char, varcharColumn29 varchar(1), longmaxColumn30 decimal, varcharwithhexColumn31 varchar(100), clobhexColumn32 clob, varcharColumn33 varchar(100)"+

        rowSet = accessor.executeQuery("delete FROM TestTableWithSimpleTypes")
        rowSet = accessor.executeQuery("delete FROM TestTableWithSimpleTypes")
        self.assertUpdateCount(rowSet, 0)

        # timestamp passed as string should be in dataspace timezone
        # timestamp passed as datetime should be in datetime timezone, default UTC
        # all timestamps returned in UTC timezone

        strdatetimeinlocal = get_localzone().localize(datetime(2017, 1, 18, 15, 16, 17, 123000))

        # insert into TestTableWithSimpleTypes
        rowSet = accessor.executeQuery("insert into TestTableWithSimpleTypes values ( \
                              1, 'c2', 'c3', 'c4',\
                              'c5', 'c6',\
                              '2017-01-18', '15:16:17','2017-01-18 15:16:17.123',\
                              10,11,12,13,14.1,15.2,16.3,\
                              17.1234567,18.12345678,true,\
                              null, null, 'clob22', null, null,\
                              null, null, null,\
                              'c', 's', 30.23456, 'c31', 'c32', 'c33'\
                              )")
        self.assertUpdateCount(rowSet, 1)

        # b = bytes([1,2,3])
        # ba1 = bytearray([1,2,3])
        # ba = bytearray(b'hello world!')

        # insert into TestTableWithSimpleTypes, using arguments
        rowSet = accessor.executeQuery("insert into TestTableWithSimpleTypes values (\
                              ?, ?, ?, ?,\
                              ?, ?,\
                              ?, ?,?,\
                              ?,?,?,?,?,?,?,\
                              ?,?,?,\
                              ?, ?, ?, ?, ?,\
                              ?, ?, ?,\
                              ?, ?, ?, ?, ?, ?)",
                                      [2, 'c22', 'c32', 'c42',
                                       'c52', 'c62',
                                       '2017-01-18', '15:16:17','2017-01-18 15:16:17.123',
                                       20,21,22,23,24.1,25.2,26.3,
                                       27.1234567,28.12345678, True,
                                       bytearray([11,12,13]), bytearray(b'bytesseq'), 'clob22', 'cXdl', 0,
                                       [1,2,3], None, None,
                                       'd', 't', 32.23456, 'c32', 'c33', 'c34']
                                       );
        self.assertUpdateCount(rowSet, 1)

        # insert into TestTableWithSimpleTypes, using date and decimals objects
        dateDate = date(2016, 12, 13)
        dateTime = time(15, 16, 17)
        dateTimestamp = datetime(2016, 12, 13, 15, 16, 17, 123000)
        rowSet = accessor.executeQuery("insert into TestTableWithSimpleTypes values (\
                              ?, ?, ?, ?,\
                              ?, ?,\
                              ?, ?,?,\
                              ?,?,?,?,?,?,?,\
                              ?,?,?,\
                              ?, ?, ?, ?, ?,\
                              ?, ?, ?,\
                              ?, ?, ?, ?, ?, ?)",
                                      [3, 'c22', 'c32', 'c42',
                                       'c52', 'c62',
                                       dateDate, dateTime, dateTimestamp,
                                       20,21,22,23,24.1,25.2,26.3,
                                       decimal.Decimal('27.1234567'),decimal.Decimal('28.12345678'), True,
                                       None, None, None, None, None,
                                       None, None, None,
                                       'd', 't', decimal.Decimal('32.23456'), 'c32', 'c33', 'c34']
                                       );
        self.assertUpdateCount(rowSet, 1)

        # insert into TestTableWithSimpleTypes, using date in locale timezone
        dateTimestampLocal = get_localzone().localize(datetime(2016, 12, 13, 15, 16, 17, 123000))
        rowSet = accessor.executeQuery("insert into TestTableWithSimpleTypes values (\
                              ?, ?, ?, ?,\
                              ?, ?,\
                              ?, ?,?,\
                              ?,?,?,?,?,?,?,\
                              ?,?,?,\
                              ?, ?, ?, ?, ?,\
                              ?, ?, ?,\
                              ?, ?, ?, ?, ?, ?)",
                                       [4, 'c24', 'c32', 'c42',
                                       'c52', 'c62',
                                       dateDate, dateTime, dateTimestampLocal,
                                       20,21,22,23,24.1,25.2,26.3,
                                       decimal.Decimal('27.1234567'),decimal.Decimal('28.12345678'), True,
                                       None, None, None, None, None,
                                       None, None, None,
                                       'd', 't', decimal.Decimal('32.23456'), 'c32', 'c33', 'c34']
                                       );
        self.assertUpdateCount(rowSet, 1)

        # select * from TestTableWithSimpleTypes
        rowSet = accessor.executeQuery("select * from TestTableWithSimpleTypes ORDER BY id")
        self.assertEqual(rowSet.getMetaData().getColumnCount(), 33)
        self.assertEqual(rowSet.getMetaData().getColumnByIndex(0).name(), "id")
        self.assertEqual(rowSet.getMetaData().getColumnByIndex(32).name(), "varcharColumn33")
        self.assertEqual(rowSet.getRowCount(), 4)

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 1)
        self.assertEqual(row.getColumnByName("charColumn2"), 'c2 ')
        self.assertEqual(row.getColumnByName("varcharColumn3"), "c3")
        self.assertEqual(row.getColumnByName("longvarcharColumn4"), "c4")
        self.assertEqual(row.getColumnByName("nvarcharColumn5"), "c5")
        self.assertEqual(row.getColumnByName("stringColumn6"), "c6")
        self.assertEqual(row.getColumnByName("dateColumn7").strftime("%Y-%m-%d"), "2017-01-18")
        self.assertEqual(row.getColumnByName("timeColumn8").strftime("%H:%M:%S"), "15:16:17")
        #self.assertEqual(row.getColumnByName("timestampColumn9").strftime("%Y-%m-%d %H:%M:%S.%f"), "2017-01-18 12:16:17.123000") # here datetime in UTC timezone, works for MSK(+3) timezone only
        self.assertEqual(row.getColumnByName("timestampColumn9").strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.astimezone(pytz.utc).strftime("%Y-%m-%d %H:%M:%S.%f")) # here datetime returned in UTC timezone
        self.assertEqual(row.getColumnByName("timestampColumn9").astimezone(get_localzone()).strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.strftime("%Y-%m-%d %H:%M:%S.%f")) # convert returned datetime to local
        self.assertEqual(row.getColumnByName("tinyintColumn10"), 10)
        self.assertEqual(row.getColumnByName("smallintColumn11"), 11)
        self.assertEqual(row.getColumnByName("integerColumn12"), 12)
        self.assertEqual(row.getColumnByName("bigintColumn13"), 13)
        self.assertEqual(row.getColumnByName("realColumn14"), 14.1)
        self.assertEqual(row.getColumnByName("doubleColumn15"), 15.2)
        self.assertEqual(row.getColumnByName("floatColumn16"), 16.3)
        self.assertEqual(row.getColumnByName("numericColumn17"), decimal.Decimal('17.123457'))
        self.assertEqual(row.getColumnByName("decimalColumn18"), decimal.Decimal('18.12346'))
        self.assertEqual(row.getColumnByName("booleanColumn19"), True)
        self.assertEqual(row.getColumnByName("binaryColumn20"), None)
        self.assertEqual(row.getColumnByName("varbinaryColumn21"), None)
        self.assertEqual(row.getColumnByName("clobColumn22").get_all_string(), "clob22")
        self.assertEqual(row.getColumnByName("blobColumn23"), None)
        self.assertEqual(row.getColumnByName("bitColumn24"), None)
        self.assertEqual(row.getColumnByName("arrayColumn25"), None)
        self.assertEqual(row.getColumnByName("otherColumn26"), None)
        self.assertEqual(row.getColumnByName("eventColumn27"), None)
        self.assertEqual(row.getColumnByName("charColumn28"), "c")
        self.assertEqual(row.getColumnByName("varcharColumn29"), "s")
        self.assertEqual(row.getColumnByName("longmaxColumn30"), decimal.Decimal('30'))  # TODO: check
        self.assertEqual(row.getColumnByName("varcharwithhexColumn31"), "c31")
        self.assertEqual(row.getColumnByName("clobhexColumn32").get_all_string(), "c32")
        self.assertEqual(row.getColumnByName("varcharColumn33"), "c33")

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 2)
        self.assertEqual(row.getColumnByName("charColumn2"), 'c22')
        self.assertEqual(row.getColumnByName("varcharColumn3"), "c32")
        self.assertEqual(row.getColumnByName("longvarcharColumn4"), "c42")
        self.assertEqual(row.getColumnByName("nvarcharColumn5"), "c52")
        self.assertEqual(row.getColumnByName("stringColumn6"), "c62")
        self.assertEqual(row.getColumnByName("dateColumn7").strftime("%Y-%m-%d"), "2017-01-18")
        self.assertEqual(row.getColumnByName("timeColumn8").strftime("%H:%M:%S"), "15:16:17")
        #self.assertEqual(row.getColumnByName("timestampColumn9").strftime("%Y-%m-%d %H:%M:%S.%f"), "2017-01-18 12:16:17.123000") # here datetime in UTC timezone, works for MSK(+3) timezone only
        self.assertEqual(row.getColumnByName("timestampColumn9").strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.astimezone(pytz.utc).strftime("%Y-%m-%d %H:%M:%S.%f")) # here datetime returned in UTC timezone
        self.assertEqual(row.getColumnByName("timestampColumn9").astimezone(get_localzone()).strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.strftime("%Y-%m-%d %H:%M:%S.%f")) # convert returned datetime to local
        self.assertEqual(row.getColumnByName("tinyintColumn10"), 20)
        self.assertEqual(row.getColumnByName("smallintColumn11"), 21)
        self.assertEqual(row.getColumnByName("integerColumn12"), 22)
        self.assertEqual(row.getColumnByName("bigintColumn13"), 23)
        self.assertEqual(row.getColumnByName("realColumn14"), 24.1)
        self.assertEqual(row.getColumnByName("doubleColumn15"), 25.2)
        self.assertEqual(row.getColumnByName("floatColumn16"), 26.3)
        self.assertEqual(row.getColumnByName("numericColumn17"), decimal.Decimal('27.123457'))
        self.assertEqual(row.getColumnByName("decimalColumn18"), decimal.Decimal('28.12346'))
        self.assertEqual(row.getColumnByName("booleanColumn19"), True)
        self.assertEqual(row.getColumnByName("binaryColumn20"), bytearray([11,12,13]))
        self.assertEqual(row.getColumnByName("varbinaryColumn21"), bytearray(b'bytesseq'))
        self.assertEqual(row.getColumnByName("clobColumn22").get_all_string(), 'clob22')
        self.assertEqual(row.getColumnByName("blobColumn23").get_all_bytes(), bytearray(b'qwe'))
        self.assertEqual(row.getColumnByName("bitColumn24"), False)
        self.assertEqual(row.getColumnByName("arrayColumn25"), [1,2,3])
        self.assertEqual(row.getColumnByName("otherColumn26"), None)
        self.assertEqual(row.getColumnByName("eventColumn27"), None)
        self.assertEqual(row.getColumnByName("charColumn28"), "d")
        self.assertEqual(row.getColumnByName("varcharColumn29"), "t")
        self.assertEqual(row.getColumnByName("longmaxColumn30"), decimal.Decimal('32.23456')) # TODO: check
        self.assertEqual(row.getColumnByName("varcharwithhexColumn31"), "c32")
        self.assertEqual(row.getColumnByName("clobhexColumn32").get_all_string(), "c33")
        self.assertEqual(row.getColumnByName("varcharColumn33"), "c34")

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 3)
        self.assertEqual(row.getColumnByName("charColumn2"), 'c22')
        self.assertEqual(row.getColumnByName("dateColumn7").strftime("%Y-%m-%d"), dateDate.strftime("%Y-%m-%d"))
        self.assertEqual(row.getColumnByName("timeColumn8").strftime("%H:%M:%S"), dateTime.strftime("%H:%M:%S"))
        self.assertEqual(row.getColumnByName("timestampColumn9").strftime("%Y-%m-%d %H:%M:%S"), dateTimestamp.strftime("%Y-%m-%d %H:%M:%S")) # TODO: check why milliseconds lost
        self.assertEqual(row.getColumnByName("numericColumn17"), decimal.Decimal('27.123457'))
        self.assertEqual(row.getColumnByName("decimalColumn18"), decimal.Decimal('28.12346'))
        self.assertEqual(row.getColumnByName("longmaxColumn30"), decimal.Decimal('32'))  # TODO: check

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 4)
        self.assertEqual(row.getColumnByName("charColumn2"), 'c24')

        # TODO: check why milliseconds lost
        # datetime returned in UTC
        self.assertEqual(row.getColumnByName("timestampColumn9").astimezone(get_localzone()).strftime("%Y-%m-%d %H:%M:%S"), dateTimestampLocal.strftime("%Y-%m-%d %H:%M:%S"))
        self.assertEqual(row.getColumnByName("numericColumn17"), decimal.Decimal('27.123457'))
        self.assertEqual(row.getColumnByName("decimalColumn18"), decimal.Decimal('28.12346'))

        self.assertFalse(rowSet.next())

        accessor.close()
        self.assertEqual(accessor.isOpened(), False)
        self.assertEqual(accessor.isAvailable(), False)
        self.assertEqual(connection.ping(), "AVAILABLE")

        connection.close()

    def testTestTableWithComplexObject(self):

        # without import
        connection = self.openConnection()
        self._testTestTableWithComplexObject(connection, TestHTTPDataspaceAccessor.createMyComplexObject, self.assertEquals)

        # with import
        connection.importSemanticType("MyComplexObject")
        def createMyComplexObject(id):
            return TestHTTPDataspaceAccessor.createMyComplexObjectWithFactory(id, connection.getTypeFactory())
        def assertEquals(obj1, obj2):
            map1 = connection.getTypeFactory().writeToMap(obj1)
            map2 = connection.getTypeFactory().writeToMap(obj2)
            json1 = Utils.dumpMapToJson(map1)
            json2 = Utils.dumpMapToJson(map2)
            self.assertEquals(json1, json2)

        self._testTestTableWithComplexObject(connection, createMyComplexObject, assertEquals)

        connection.close()

        # with user defined
        connection = self.openConnection()

        class MyComplexObject1(object):

            def __init__(self, id):
                self.ii = id
                self.ss = 's' + str(id)
                self.bb = True

                self.ssubobjectArray = [
                    {
                        'ss' : 'ss' + str(id + 1)
                    }
                ]

        from stpy.fabric.type_converter import TypeConverter
        class MyComplexObject1Converter(TypeConverter):

            def typeClass(self, name = None):
                return MyComplexObject1

            def readFromMap(self, rawValue, typeFactory=None):
                if rawValue is None:
                    return None

                result = MyComplexObject1(-1)
                result.ii = rawValue.get('i', None)
                result.ss = rawValue.get('s', None)
                result.bb = rawValue.get('b', None)
                result.ssubobjectArray = [
                    {
                        'ss' : o['s']
                    } for o in rawValue.get('subobjectArray', None)
                ]

                return result

            def writeToMap(self, value, typeFactory=None):
                if value is None:
                    return None
                return {
                    '@type' : 'MyComplexObject',
                    'i' : value.ii,
                    's' : value.ss,
                    'b' : value.bb,
                    'subobjectArray' : [
                        {
                            '@type' : 'MyComplexSubObject',
                            's' : o['ss']
                        } for o in value.ssubobjectArray
                    ]
                }

        connection.getTypeFactory().registerType('MyComplexObject1', MyComplexObject1Converter())
        connection.getTypeFactory().registerType('MyComplexObject', MyComplexObject1Converter())

        def createMyComplexObject(id):
            return MyComplexObject1(1)

        def assertEquals(obj1, obj2):
            self.assertEquals(type(obj1), MyComplexObject1)
            self.assertEquals(type(obj2), MyComplexObject1)
            map1 = connection.getTypeFactory().writeToMap(obj1)
            map2 = connection.getTypeFactory().writeToMap(obj2)
            json1 = Utils.dumpMapToJson(map1)
            json2 = Utils.dumpMapToJson(map2)
            self.assertEquals(json1, json2)

        self._testTestTableWithComplexObject(connection, createMyComplexObject, assertEquals)


        connection.close()

    def _testTestTableWithComplexObject(self, connection, createMyComplexObject, assertEquals):
        accessor = self.openAccessorTspace(connection)

        rowSet = accessor.executeQuery("delete FROM TestTableWithComplexObject")
        rowSet = accessor.executeQuery("delete FROM TestTableWithComplexObject")
        self.assertUpdateCount(rowSet, 0)

        # create table TestTableWithComplexObject(id int, obj MyComplexObject, primary key (id))
        rowSet = accessor.executeQuery("insert INTO TestTableWithComplexObject VALUES (?,?)", [1, createMyComplexObject(1)])
        self.assertUpdateCount(rowSet, 1)
        rowSet = accessor.executeQuery("insert INTO TestTableWithComplexObject VALUES (?,?)", [2, createMyComplexObject(2)])
        self.assertUpdateCount(rowSet, 1)
        rowSet = accessor.executeQuery("insert INTO TestTableWithComplexObject VALUES (?,?)", [3, createMyComplexObject(3)])
        self.assertUpdateCount(rowSet, 1)

        # select * from TestTableWithComplexObject
        rowSet = accessor.executeQuery("select * from TestTableWithComplexObject ORDER BY id")
        self.assertEqual(rowSet.getMetaData().getColumnCount(), 2)
        self.assertEqual(rowSet.getMetaData().getColumnByIndex(0).name(), "id")
        self.assertEqual(rowSet.getMetaData().getColumnByIndex(1).name(), "obj")
        self.assertEqual(rowSet.getRowCount(), 3)

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 1)
        assertEquals(row.getColumnByName("obj"), createMyComplexObject(1))

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 2)
        assertEquals(row.getColumnByName("obj"), createMyComplexObject(2))

        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByName("id"), 3)
        assertEquals(row.getColumnByName("obj"), createMyComplexObject(3))

        self.assertFalse(rowSet.next())

        accessor.close()
        self.assertEqual(accessor.isOpened(), False)
        self.assertEqual(accessor.isAvailable(), False)
        self.assertEqual(connection.ping(), "AVAILABLE")

    def assertUpdateCount(self, rowSet, updateCount):
        self.assertEqual(rowSet.getMetaData().getColumnCount(), 1)
        self.assertEqual(rowSet.getMetaData().getColumnByIndex(0).name(), "UpdateCount")
        self.assertEqual(rowSet.getRowCount(), 1)
        self.assertTrue(rowSet.next())
        row = rowSet.getCurrentRow()
        self.assertEqual(row.getColumnByIndex(0), updateCount)
        self.assertFalse(rowSet.next())

    def openConnection(self):
        connection = HTTPFabricConnection("localhost:8888", "Admin", "Admin")
        connection.open()
        self.assertEqual(connection.isOpened(), True)
        self.assertEqual(connection.ping(), "AVAILABLE")
        return connection

    def openAccessorTspace(self, connection):
        accessor = connection.createDataspaceAccessor(None, DataspaceType.TSPACE, "swagger_tspace")
        self.assertEqual(accessor.isOpened(), True)
        self.assertEqual(accessor.isAvailable(), True)
        return accessor

    @staticmethod
    def createMyComplexObject(id):
        import pytz
        return {
            '@type' : 'MyComplexObject',
            'i' : id,
            's' : 's' + str(id),
            'b' : True,

            'byteField' : id,
            'byteArrayField' : 'cXdl',
            'shortField' : id,
            'intField' : id,
            'longField' : id,
            'floatField' : float(id),
            'doubleField' : float(id),
            'booleanField' : True,
            'stringField' : "stringField" + str(id),

            'bigIntegerField' : 1000 + id,
            'bigDecimalField' : 12345.223 + id,

            'utilDateField' : int((datetime(2016, 12, 13, 15, 16, 17, 123) - datetime(1970, 1, 1)).total_seconds() * 1000),
            'sqlDateField' : date(2016, 12, 13).strftime("%Y-%m-%d"),
            'sqlTimeField' : time(15, 16, 17).strftime("%H:%M:%S"),
            'sqlTimestampField' : int((datetime(2016, 12, 13, 15, 16, 17, 123) - datetime(1970, 1, 1)).total_seconds() * 1000),

            'integerArray' : [id + 1,id + 2, id + 3],
            'integerList'  : [id + 5, id + 6],
            'subobject' : TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 1),
            'subobjectArray' : [
                TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 2),
                TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 3)
            ],
            'subobjectList' : [
                TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 4),
                TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 5)
            ],

            'subjectMap' : {
                # '@type' : 'map',
                'k6' : TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 6),
                'k7' : TestHTTPDataspaceAccessor.createMyComplexSubObject(id + 7)
            },

            'subobjectNoSemType' : TestHTTPDataspaceAccessor.createMyComplexSubObjectNoSemType(id + 6)
        }

    @staticmethod
    def createMyComplexSubObject(id):
        return {
            '@type' : 'MyComplexSubObject',
            's' : 's' + str(id)
        }

    @staticmethod
    def createMyComplexSubObjectNoSemType(id):
        return {
            '@type' : 'com.streamscape.sef.network.http.MyComplexSubObjectNoSemType',
            's' : 's' + str(id)
        }

    @staticmethod
    def createMyComplexObjectWithFactory(id, typeFactory):
        result = typeFactory.instantiateType("MyComplexObject")
        result.i = id
        result.s = 's' + str(id)
        result.b = True

        result.byteField = id
        result.byteArrayField = bytearray([11,12,13])
        result.shortField = id
        result.intField = id
        result.longField = id
        result.floatField = float(id)
        result.doubleField = float(id)
        result.booleanField = True
        result.stringField = "stringField" + str(id)

        result.bigIntegerField = 1000 + id
        result.bigDecimalField = 12345.223 + id

        result.utilDateField = datetime(2016, 12, 13, 15, 16, 17, 123)
        result.sqlDateField = date(2016, 12, 13)
        result.sqlTimeField = time(15, 16, 17)
        result.sqlTimestampField = datetime(2016, 12, 13, 15, 16, 17, 123)

        result.integerArray = [id + 1,id + 2, id + 3]
        result.integerList = [id + 5, id + 6]
        result.subobject = TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 1, typeFactory)
        result.subobjectArray = [
            TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 2, typeFactory),
            TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 3, typeFactory)
        ]
        result.subobjectList = [
            TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 4, typeFactory),
            TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 5, typeFactory)
        ]

        result.subjectMap = {
            'k6' : TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 6, typeFactory),
            'k7' : TestHTTPDataspaceAccessor.createMyComplexSubObjectWithFactory(id + 7, typeFactory)
        }

        result.subobjectNoSemType = TestHTTPDataspaceAccessor.createMyComplexSubObjectNoSemTypeWithFactory(id + 6, typeFactory)

        return result

    @staticmethod
    def createMyComplexSubObjectWithFactory(id, typeFactory):
        result = typeFactory.instantiateType("MyComplexSubObject")
        result.s = 's' + str(id)
        return result

    @staticmethod
    def createMyComplexSubObjectNoSemTypeWithFactory(id, typeFactory):
        result = typeFactory.instantiateType("com.streamscape.sef.network.http.MyComplexSubObjectNoSemType")
        result.s = 's' + str(id)
        return result