from __future__ import absolute_import

import base64
import decimal
import json
import logging
from datetime import datetime, date, time

from .lobs import ClobImpl, BlobImpl, RowSetClobProxy, RowSetBlobProxy
from .abstract_accessible_object_proxy import AccessibleObjectProxy
from .enums import SQLType
from .http_fabric_exception import HTTPFabricException
from .sl_response import SLResponse

logger = logging.getLogger(__name__)


class Utils(object):
    """
    Auxiliary class with helpfull methods.
    """

    @staticmethod
    def method(methodName, arguments=[]):
        return {
            'methodName': methodName,
            'arguments': arguments
        }

    @staticmethod
    def remote(remoteMethodCalls, postprocessorMethodCall=None):
        return {
            # '@type' : 'HTTPFabricConnectionRemoteCall',
            'consequentMethodCalls': remoteMethodCalls,
            'postprocessorMethodCall': postprocessorMethodCall
        }

    @staticmethod
    def enum(name, value):
        return {
            "@type": name,
            "value": value
        }

    @staticmethod
    def createOpaqueEvent(eventId, data):
        return {
            "@type": "OpaqueEvent",
            # "eventSource":"AA==",
            "eventId": eventId,
            # "eventKey":"",
            # "eventGroupId":"",
            # "durable":False,
            # "timeStamp":0,
            # "coalesced":False,
            "data": data
        }

    @staticmethod
    def parseJson(str):
        try:
            return json.loads(str)
        except ValueError:
            return json

    @staticmethod
    def parseJsonResponse(r):
        return Utils.parseJson(r.data)

    @staticmethod
    def dumpMapToJson(map):
        # we need sort_keys=True to place @type field to the first place
        try:
            if Utils.is_python3():
                class BytesEncoder(json.JSONEncoder):
                    def default(self, obj):
                        if type(obj) == bytes or type(obj) == bytearray:
                            import base64
                            return base64.b64encode(obj).decode('ascii')
                        return json.JSONEncoder.default(self, obj)

                return json.dumps(map, sort_keys=True, cls=BytesEncoder)
            else:
                return json.dumps(map, sort_keys=True) #.encode('utf-8')
        except Exception as e:
            print(e);

    @staticmethod
    def processServerAnswer(answer, expectedType=None):
        if answer == None:
            return answer

        if isinstance(answer, dict):
            if answer['@type'] == 'HTTPClientException':
                Utils.supressStackTrace(answer)
                raise HTTPFabricException(answer['detailMessage'], answer)
            if expectedType != None and answer['@type'] != None and not answer['@type'].endswith(expectedType):
                raise HTTPFabricException(
                    "Unexpected response from server received. Expected '" + expectedType + "', but got '" + answer['@type'] + "'.")
        elif expectedType != None:

            if not Utils.is_python3():
                if expectedType == str and type(answer) == unicode:
                    answer = str(answer)
            if type(answer) != expectedType:
                raise HTTPFabricException(
                    "Unexpected response from server received. Expected '" + str(expectedType) + "', but got '" + str(type(answer)) + "'.")

        return answer

    @staticmethod
    def supressStackTrace(exception):
        if exception is None:
            return
        if exception.get('stackTrace') is not None:
            exception['stackTrace'] = None
        Utils.supressStackTrace(exception.get('cause'))


    @staticmethod
    def parseJsonResponseAndCheck(r, expectedType=None):
        o = Utils.parseJsonResponse(r)
        return Utils.processServerAnswer(o, expectedType)

    @staticmethod
    def sleep(startTime, timeout):
        import time as t
        toSleep = timeout / 1000 - (t.time() - startTime)
        if toSleep > 0:
            t.sleep(toSleep)

    @staticmethod
    def calculateHttpTimeout(timeout, defaultTimeout):
        if timeout == None:
            timeout = defaultTimeout
        if timeout > 0:
            timeout = timeout + 5000
        elif timeout == 0:
            timeout = 0
        elif defaultTimeout > 0:
            timeout = defaultTimeout + 5000
        return timeout

    @staticmethod
    def convertDsToPython(value, sqlType, precision=None, scale=None, typeFactory=None):
        if value == None:
            return None

        if sqlType == SQLType.CHAR or sqlType == SQLType.VARCHAR or sqlType == SQLType.LONGVARCHAR or sqlType == SQLType.NCHAR or sqlType == SQLType.NVARCHAR or sqlType == SQLType.LONGNVARCHAR \
                or sqlType == SQLType.STRING:
            # if type(value) != str:
            value = str(value)
        elif sqlType == SQLType.BINARY or sqlType == SQLType.VARBINARY or sqlType == SQLType.LONGVARBINARY:
            value = Utils.b64decode(value)

        elif sqlType == SQLType.CLOB:
            if type(value) == dict:
                value = typeFactory.readFromMap(value, RowSetClobProxy, True)
            elif value != "[ CLOB ]":
                value = ClobImpl(value)

        elif sqlType == SQLType.BLOB or sqlType == SQLType.FLOB:
            if type(value) == dict:
                value = typeFactory.readFromMap(value, RowSetBlobProxy, True)
            elif value != "[ BLOB ]" and value != "[ FLOB ]":
                value = BlobImpl(Utils.b64decode(value))

        elif sqlType == SQLType.TINYINT or sqlType == SQLType.SMALLINT or sqlType == SQLType.INTEGER \
                or sqlType == SQLType.UTINYINT or sqlType == SQLType.USMALLINT:
            value = int(value)
        elif sqlType == SQLType.LONG or sqlType == SQLType.BIGINT \
                or sqlType == SQLType.UINTEGER or sqlType == SQLType.UBIGINT:
            if Utils.is_python3():
                value = int(value)
            else:
                value = long(value)
        elif sqlType == SQLType.REAL or sqlType == SQLType.FLOAT or sqlType == SQLType.DOUBLE:
            value = float(value)
        elif sqlType == SQLType.NUMERIC or sqlType == SQLType.DECIMAL:
            value = decimal.Decimal(str(value))
        elif sqlType == SQLType.BOOLEAN or sqlType == SQLType.BIT:
            value = bool(value)
        elif sqlType == SQLType.DATE:
            try:
                from dateutil.parser import parse
                value = parse(value).date()
            except ImportError:
                return value
            except ValueError:
                raise HTTPFabricException("Failed to parse `{0}` into a date object".format(value))
        elif sqlType == SQLType.TIME:
            try:
                from dateutil.parser import parse
                value = parse(value).time()
            except ImportError:
                return value
            except ValueError:
                raise HTTPFabricException("Failed to parse `{0}` into a date object".format(value))
        elif sqlType == SQLType.TIMESTAMP:
            # millis = float(value['millis'])
            value = Utils.fromMillisInUtc(float(value))
        elif sqlType == SQLType.EVENT or sqlType == SQLType.ARRAY or sqlType == SQLType.URL or sqlType == SQLType.OTHER:
            value = typeFactory.readFromMap(value, None, False)
        else:
            logger.warning("unknown type '{0}' for value '{1}'.".format(sqlType, value))

        return value

    @staticmethod
    def convertPythonToDs(value, typeFactory):
        if type(value) == date:
            value = value.strftime("%Y-%m-%d")
        elif type(value) == time:
            value = value.strftime("%H:%M:%S")
        elif type(value) == datetime:
            value = Utils.toMillisInUtc(value)

            # value = value.strftime("%Y-%m-%d %H:%M:%S.%f")
            # value = {
            #     '@type:' : 'SqlTimestamp',
            #     'millis': datetime.strftime("%s") * 1000
            # }
        elif type(value) == decimal.Decimal:
            value = str(value)
        elif type(value) == bytearray:
            value = base64.b64encode(value)
            if Utils.is_python3():
                value = value.decode('ascii');
        else:
            value = typeFactory.writeToMap(value, raiseExceptionIfConverterNotFound=False)

        return value

    @staticmethod
    def toMillisInUtc(d):
        import pytz # $ pip install pytz
        if d.tzinfo is not None and d.tzinfo.utcoffset(d) is not None:
            d = d.astimezone(pytz.utc)
            d = d.replace(tzinfo=None)

        return int((d - datetime.utcfromtimestamp(0)).total_seconds() * 1000)

    @staticmethod
    def fromMillisInUtc(millis):
        import pytz # $ pip install pytz
        return pytz.utc.localize(datetime.utcfromtimestamp(millis / 1000.0))

    @staticmethod
    def b64decode(value):
        value = base64.b64decode(value)
        bytes = bytearray()
        bytes.extend(value)
        return bytes

    @staticmethod
    def processAccessibleObjects(response, accessor=None):
        if type(response) == SLResponse:
            if response.rowSet and issubclass(type(response.rowSet), AccessibleObjectProxy):
                response.rowSet.setAccessor(accessor)
            if response.object and issubclass(type(response.rowSet), AccessibleObjectProxy):
                response.object.setAccessor(accessor)
                # TODO:
                # if response.object and response.object is array:
                # Object[] array = (Object[])slResponse.getObject();
                # for (Object o : array)
                # if (o instanceof AccessibleObjectProxy)
                # ((AccessibleObjectProxy)o).setAccessor(accessor);
                # }
                # }
        elif issubclass(type(response), AccessibleObjectProxy):
            response.setAccessor(accessor)
        return response

    @staticmethod
    def log(func, message, args):
        if args and len(args) > 0:
            func(message, *args)
        else:
            func(message)

    loggerconf = None
    @staticmethod
    def initLogs():
        if Utils.loggerconf:
            return

        # create logger with 'stpy'
        Utils.loggerconf = logging.getLogger('stpy')
        Utils.loggerconf.setLevel(logging.DEBUG)
        # create file handler which logs even debug messages
        fh = logging.FileHandler('target/stpy.log')
        fh.setLevel(logging.DEBUG)
        # create console handler with a higher log level
        ch = logging.StreamHandler()
        ch.setLevel(logging.ERROR)
        # create formatter and add it to the handlers
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        fh.setFormatter(formatter)
        ch.setFormatter(formatter)
        # add the handlers to the logger
        Utils.loggerconf.addHandler(fh)
        Utils.loggerconf.addHandler(ch)

    @staticmethod
    def is_python3():
        import sys
        return (sys.version_info[0] >= 3)

class RemoteMethodCallPostprocessor(object):
    IS_NULL = Utils.enum("RemoteMethodCallPostprocessor", "IS_NULL")
    GET_NAMES_FROM_LIST = Utils.enum("RemoteMethodCallPostprocessor", "GET_NAMES_FROM_LIST")
    GET_NAME_IF_NOT_NULL = Utils.enum("RemoteMethodCallPostprocessor", "GET_NAME_IF_NOT_NULL")
