"""
This module contains objects that represent returned ``RowSet``
"""

from __future__ import absolute_import

from .type_converter import TypeConverter
from .http_fabric_exception import HTTPFabricException
from .abstract_accessible_object_proxy import AbstractAccessibleObjectProxy, AccessibleObjectProxy

class ColumnDescriptor(object):

    def __init__(self):
        self._name = None
        self._type = None
        self._nullable = True

        self._precision = 128
        self._scale = 0

    @staticmethod
    def readFromMap(map, descriptor=None, typeFactory=None):
        if map == None:
            return None

        if descriptor is None:
            descriptor = ColumnDescriptor()
        descriptor._name = map.get('name', None)
        descriptor._type = map.get('type', None)
        if descriptor._type:
            descriptor._type = descriptor._type.get('value', None)
        descriptor._nullable = map.get('nullable', True)
        descriptor._precision = map.get('precision', 128)
        descriptor._scale = map.get('scale', 0)

        return descriptor

    def name(self):
        return self._name

    def type(self):
        return self._type

    def nullable(self):
        return self._nullable

    def precision(self):
        return self._precision

    def scale(self):
        return self._scale

class RowMetaData(object):

    def __init__(self):
        self._capacity = -1
        self._hasReturnCode = False
        self._keyIndices = []
        self._columns = []
        self._outParameters = []

    @staticmethod
    def readFromMap(map, meta = None, typeFactory=None):
        if map == None:
            return None

        if meta is None:
            meta = RowMetaData()
        meta._capacity = map.get('capacity', -1)
        meta._hasReturnCode = map.get('hasReturnCode', False)
        meta._keyIndices = map.get('keyIndices', [])

        columns = map.get('columns', None)
        if columns:
            meta._columns = [ColumnDescriptor.readFromMap(column, typeFactory=typeFactory) for column in columns]

        outParameters = map.get('outParameters', None)
        if outParameters:
            meta._outParameters = [ColumnDescriptor.readFromMap(outParameter, typeFactory=typeFactory) for outParameter in outParameters]

        return meta

    def getColumnByName(self, columnName):
        for column in self._columns:
            if column._name == columnName:
                return column
        return None

    def getColumnIndex(self, columnName):
        for i in range(0, len(self._columns)):
            if self._columns[i].name() == columnName:
                return i
        return None

    def getColumnByIndex(self, columnIndex):
        if columnIndex < 0 or columnIndex >= len(self._columns):
            return None
        return self._columns[columnIndex]

    def getColumnCount(self):
        return len(self._columns)

    def getColumns(self):
        return self._columns

class Row(object):

    def __init__(self, meta = None):
        self._data = []
        self._hasRID = False
        self._meta = meta

    @staticmethod
    def readFromMap(map, meta = None, typeFactory=None):
        if map == None:
            return None
        row = Row.readFromDataArray(map.get('data', []), meta, typeFactory)
        row._hasRID = map.get('hasRID', False)
        return row

    @staticmethod
    def readFromDataArray(data, meta = None, typeFactory=None):
        if data == None:
            return None
        row = Row()

        row._data = data
        row._meta = meta

        from .utils import Utils
        for i in range(0, len(row._data)):
            descriptor = meta.getColumnByIndex(i)
            row._data[i] = Utils.convertDsToPython(row._data[i], descriptor._type, descriptor._precision, descriptor._scale, typeFactory)

        return row

    def getColumnsCount(self):
        return len(self._data)

    def getColumnByIndex(self, index):
        if index < 0 or index >= len(self._data):
            raise IndexError("Column index is out of range.")
        return self._data[index]

    def getColumnByName(self, name):
        index = self._meta.getColumnIndex(name)
        return self.getColumnByIndex(index)

    def getColumns(self, ):
        return self._data

class RowSet(object):

    def __init__(self, meta = None):
        self._meta = meta

        self._returnCodeValue = None
        self._outParameterValues = None
        self._rows = []

        self._readOnly = False
        self._encryptedPassword = None

        self._currentRow = -1
        self._pageSize = 0

    @staticmethod
    def readFromMap(map, rowSet = None, typeFactory=None):
        if map == None:
            return None

        if rowSet is None:
            rowSet = RowSet()
        rowSet._meta = RowMetaData.readFromMap(map.get('meta', None), typeFactory=typeFactory)
        rowSet._returnCodeValue = map.get('returnCodeValue', None)
        rowSet._outParameterValues = map.get('outParameterValues', None)

        rows = map.get('rows', None)
        if rows:
            rowSet._rows = [Row.readFromMap(row, rowSet._meta, typeFactory) for row in rows]

        rowSet._readOnly = map.get('readOnly', False)
        rowSet._encryptedPassword  = map.get('encryptedPassword', None)

        return rowSet

    def getMetaData(self):
        return self._meta

    def _addAll(self, other):
        for row in other._rows:
            row._meta = self._meta
            if self._rows == None:
                self._rows = []
            self._rows.append(row)

    # NAVIGATION METHODS

    def absolute(self, row):
        """
        Moves the cursor to the given row number in this RowSet object.

        :param row:
            the number of the row to which the cursor should move. A
            positive number indicates the row number counting from the
            beginning of the result set; a negative number indicates the row
            number counting from the end of the result set
        :return: True if the cursor is on the result set; False otherwise
        """
        row = len(self._rows) - 1 + row if row < 0 else row
        if row < 0 or row >= len(self._rows):
            return False
        self._currentRow = row
        return True

    def afterLast(self):
        self._currentRow = len(self._rows)
        pass

    def beforeFirst(self):
        self._currentRow = -1

    def first(self):
        if len(self._rows) > 0:
            self._currentRow = 0
            return True
        return False

    def isAfterLast(self):
        return self._currentRow >= len(self._rows)

    def isBeforeFirst(self):
        return self._currentRow < 0

    def isFirst(self):
        return self._currentRow == 0

    def isLast(self):
        return self._currentRow == len(self._rows) - 1

    def last(self):
        self._currentRow = len(self._rows) - 1
        return True

    def next(self):
        self._currentRow += 1
        return not self.isAfterLast()

    def previous(self):
        self._currentRow -= 1
        return not self.isBeforeFirst()

    def relative(self, rowsCount):
        curRow = self._currentRow
        if self.isBeforeFirst():
            curRow = -1
        elif self.isAfterLast():
            curRow = len(self._rows)

        row = curRow + rowsCount
        if row < 0 or row >= len(self._rows):
            return False

        self._currentRow = row
        return True

    def getRow(self):
        return self._currentRow + 1

    def close(self):
        pass

    def isClosed(self):
        return False

    def setPageSize(self, pageSize):
        self.setFetchSize(pageSize)

    def getPageSize(self):
        return self.getFetchSize()

    def getFetchSize(self):
        return self._pageSize

    def setFetchSize(self, fetchSize):
        self._pageSize = fetchSize

    def nextPage(self):
        if self.getPageSize() <= 0:
            return False

        if self._currentRow + self.getPageSize() < len(self._rows):
            self._currentRow = (self._currentRow // self.getPageSize() + 1) * self.getPageSize()
            return True
        return False

    def previousPage(self):
        if self.getPageSize() <= 0:
            return False

        if self._currentRow - self.getPageSize() > 0:
            self._currentRow = (self._currentRow // self.getPageSize() - 1) * self.getPageSize()
            return True
        return False

    def getRowAt(self, index):
        if index < 1 or index > len(self._rows):
            raise HTTPFabricException("Row index is out of range.")
        return self._rows[int(index - 1)]

    def getCurrentRow(self):
        return self.getRowAt(self._currentRow + 1)

    def getRowCount(self):
        return len(self._rows)

    def asRowSet(self):
        # nothing needs to be done
        return self


class RowSetProxy(RowSet, AccessibleObjectProxy):

    def __init__(self, meta = None):
        super(RowSetProxy, self).__init__(meta)
        self._fetchSize = 0
        self._offset = 0

        self._isScrollable = False
        self._isClosed = False

        self._helper = AbstractAccessibleObjectProxy()

    @staticmethod
    def readFromMap(map, typeFactory=None):
        if map == None:
            return None

        rowSet = RowSet.readFromMap(map, RowSetProxy(), typeFactory=typeFactory)
        rowSet._helper._oid = map.get("proxyHelper").get('oid', 0)
        rowSet._fetchSize = map.get('fetchSize', 0)
        rowSet._isScrollable = map.get('isScrollable', 0)

        rowSet._typeFactory = typeFactory
        rowSet._setRowProxies()

        return rowSet

    def getId(self):
        return self._helper.getId()

    def setId(self, id):
        self._helper.setId(id)

    def setAccessor(self, accessor):
        self._helper.setAccessor(accessor)

    def _setRowProxies(self):
        from .lobs import AbstractRowSetLobProxy
        for row in self._rows:
            for value in row._data:
                if issubclass(type(value), AbstractRowSetLobProxy):
                    value.setAccessor(self);

    def _populateData(self, rowsData):
        self._rows = []
        if not rowsData:
            return

        for rowData in rowsData:
            row = Row.readFromDataArray(rowData, self._meta, self._typeFactory)
            self._rows.append(row)
        self._setRowProxies()

    def _fetchRows(self, offset):
        self._offset = offset
        result = self._helper.invokeMethodWithReturn("fetchRows", [offset, self._fetchSize])
        self._populateData(result)

    def isScrollable(self):
        return self._isScrollable

    def setIsScrollable(self, isScrollable):
        self._isScrollable = isScrollable

    ###################### ResultSet interface implementation ######################

    ######################  Navigation methods ######################

    ######################  scrollable methods ######################

    #### row : 0,1..

    def absolute(self, row):
        self._checkNotClosed()
        self._checkIsScrollable()

        if row < 0:
            absoluteResult = self._helper.invokeMethodWithReturn("absolute", [row])
            if not absoluteResult:
                return False

            realRowNumber = self._helper.invokeMethodWithReturn("getRow")
            row = realRowNumber - 1

        if self._offset >= 0 and row >= self._offset and row < self._offset + len(self._rows):
            self._currentRow = row - self._offset
            if len(self._rows) == 0:
                self._fetchRows(self._offset)
            return True

        try:
            result = self._helper.invokeMethodWithReturn("absolute", [row])
            if result == False:
                self._currentRow = -1
                return False

            self._fetchRows(row)
            self._currentRow = 0
            return True
        except Exception as exception:
            self._currentRow = -1
            raise exception

    def beforeFirst(self):
        self._checkNotClosed()
        if self._offset == 0:
            self._currentRow = -1
            return

        self._checkIsScrollable()
        self._currentRow = -1
        
        if self._offset != 0:
            self._currentRow = -1
            self._offset = -1
            self._rows = []

    def afterLast(self):
        self._checkNotClosed()
        self._checkIsScrollable()

        if not self.last():
            raise HTTPFabricException("Failed to navigate to the last.")
        self.next()

    def first(self):
        self._checkNotClosed()
        if self._offset == 0:
            self._currentRow = 0
            return True

        self._checkIsScrollable()
        return self.absolute(0)

    def last(self):
        self._checkNotClosed()
        self._checkIsScrollable()

        last = self._helper.invokeMethodWithReturn("last")
        if not last:
            return False

        row = self._helper.invokeMethodWithReturn("getRow")
        return self.absolute(row - 1)

    def relative(self, rowsCount):
        self._checkNotClosed()
        self._checkIsScrollable()

        if self._offset < 0 or self._offset + self._currentRow + rowsCount < 0:
            self._currentRow = -1
            return False

        result = False
        if rowsCount < 0 and self._currentRow <= 0 and self._offset > 0:
            # fetch previous page
            if self._fetchSize > 0 and self._fetchSize < self._offset:
                result = self.absolute(self._offset - self._fetchSize)
                if result:
                    result = self.absolute(self._offset + self._fetchSize + rowsCount)
            else:
                tmp = self._offset + rowsCount
                result = self.absolute(0)
                if result:
                    self.absolute(tmp)
        else:
            result = self.absolute(self._offset + self._currentRow + rowsCount)
        return result

    def previous(self):
        self._checkNotClosed()
        self._checkIsScrollable()

        return self.relative(-1)

    ##################### forward methods #################

    def isAfterLast(self):
        self._checkNotClosed()
        if self._currentRow == len(self._rows):
            return self._helper.invokeMethodWithReturn("isAfterLast")

        return False

    def isBeforeFirst(self):
        return self._currentRow < 0 and self._offset <= 0

    def isFirst(self):
        return self._currentRow == 0 and self._offset == 0

    def isLast(self):
        self._checkNotClosed()
        if len(self._rows) > 0 and self._currentRow == len(self._rows) - 1:
            return self._helper.invokeMethodWithReturn("isLast")
        return False

    def next(self):
        self._checkNotClosed()

        if self._isScrollable:
            result = self.relative(1)
            if not result and self._offset >= 0 and self._currentRow == -1:
                self._currentRow = len(self._rows)
            return result

        if self._offset >= 0 and self._currentRow >= - 1 and self._currentRow < len(self._rows) - 1:
            self._currentRow+=1
            return True

        try:
            self._fetchRows(self._offset + self._currentRow + 1)
            self._currentRow = 0
            if len(self._rows) == 0:
                return False
            return True
        except Exception as exception:
            self._currentRow = -1
            raise exception

    def getRow(self):
        if not self.isBeforeFirst() and not self.isAfterLast():
            return self._offset + self._currentRow + 1
        return 0

    def close(self):
        try:
            self._helper.invokeMethod("close", [])
        except Exception as exception:
            pass

        # call = self._slDataspaceCall(self.TOP_ACCESSOR_ID, "eraseFromObjectCache", [self._helpder.getId()])
        # self._helper.invokeRequest(self.SLDataspaceCallEventId, call)

        call = self._slDataspaceCall(AccessibleObjectProxy.TOP_ACCESSOR_ID, "eraseFromObjectCache", [self._helper.getId()])
        self._helper.invokeRequest(AccessibleObjectProxy.SLDataspaceCallEventId, call)
        self._isClosed = True
        self._helper._accessor = None


    def isClosed(self):
        return self._isClosed

    def getFetchSize(self):
        return self._fetchSize

    def setFetchSize(self, fetchSize):
        self._fetchSize = fetchSize

    def setPageSize(self, pageSize):
        self.setFetchSize(pageSize)

    def getPageSize(self):
        return self.getFetchSize()

    def getCurrentRow(self):
        if self._currentRow < 0 or self._currentRow >= len(self._rows):
            raise HTTPFabricException("Row not set.")

        return self._rows[self._currentRow]

    def asRowSet(self):
        self.beforeFirst()
        result = RowSet(self._meta)

        result._addAll(self)
        self._fetchAllRowSetLobs()

        while self.nextPage():
            result._addAll(self)
            self._fetchAllRowSetLobs()

        self.close()
        return result

    def _fetchAllRowSetLobs(self):
        lobIndexes = []
        from stpy import SQLType
        for i in range(0, self._meta.getColumnCount()):
            sqlType = self._meta.getColumnByIndex(i).type()
            if sqlType == SQLType.BLOB or sqlType == SQLType.CLOB or sqlType == SQLType.FLOB:
                lobIndexes.append(i);

        from .lobs import AbstractRowSetLobProxy, RowSetBlobProxy, RowSetClobProxy, BlobImpl, ClobImpl
        for row in self._rows:
            for index in lobIndexes:
                value = row._data[index];
                if issubclass(type(value), AbstractRowSetLobProxy):
                    value.setAccessor(self)
                if type(value) == RowSetBlobProxy:
                    value = BlobImpl(value.get_all_bytes())
                elif type(value) == RowSetClobProxy:
                    value = ClobImpl(value.get_all_string())
                row._data[index] = value

    # def getRowCount(self):
    #     return None

    def nextPage(self):
        self._checkNotClosed()

        backup = self._backupRowsAndOffset()
        try:
            self._fetchRows(self._offset + len(self._rows))
            if len(self._rows) == 0:
                self._restoreRowsAndOffset(backup)
                return False
            else:
                self._currentRow = 0
                return True
        except Exception as exception:
            self._restoreRowsAndOffset(backup)

            if hasattr(exception, 'message') and exception.message is not None and exception.message.find("Failed to set cursor position to ") == -1:
                raise exception
            else:
                return False

    def previousPage(self):
        if self._offset == 0:
            return False

        backup = self._backupRowsAndOffset()
        try:
            pageOffset = -self._currentRow
            if self._fetchSize > 0 and self._fetchSize < self._offset:
                pageOffset -= self._fetchSize
            else:
                pageOffset -= self._offset

            result = self.relative(pageOffset)
            if not result:
                self._restoreRowsAndOffset(backup)
            return result
        except Exception as exception:
            self._restoreRowsAndOffset(backup)
            raise exception

    def _backupRowsAndOffset(self):
        return [list(self._rows), self._currentRow, self._offset]

    def _restoreRowsAndOffset(self, backup):
        self._rows = backup[0]
        self._currentRow = backup[1]
        self._offset     = backup[2]


    def _checkIsScrollable(self):
        if not self._isScrollable:
            raise HTTPFabricException("RowSet is forward only.")

    def _checkNotClosed(self):
        if self._isClosed:
            raise HTTPFabricException("RowSet is closed.")

    ############################

    def _invokeRowSetMethodWithReturn(self, id, method):
        return self._helper.invokeMethodWithReturn(method, [id])

    def _invokeRowSetBlobMethodWithReturn(self, id, method, params):
        return self._helper.invokeMethodWithReturn("invokeRowSetBlobMethod", self._makeRowSetBlobMethodParmas(id, method, params))

    def _invokeRowSetBlobMethod(self, id, method, params):
        self._helper.invokeMethod("invokeRowSetBlobMethod", self._makeRowSetBlobMethodParmas(id, method, params))

    def _makeRowSetBlobMethodParmas(self, id, method, params):
        return [id, method, params]


class RowSetConverter(TypeConverter):

    @staticmethod
    def typeClass(name):
        if name == 'RowSet':
            return RowSet
        elif name == 'RowSetProxy':
            return RowSetProxy
        else:
            raise HTTPFabricException("Unknown RowSet type {}".format(name))

    @staticmethod
    def readFromMap(map, typeFactory=None):
        if map == None:
            return None

        t = map.get('@type', None)
        if t == None or t == 'RowSet':
            return RowSet.readFromMap(map, typeFactory=typeFactory)
        elif t == 'RowSetProxy':
            return RowSetProxy.readFromMap(map, typeFactory=typeFactory)
        else:
            raise HTTPFabricException("Unknown RowSet type {}".format(t))

