from __future__ import absolute_import

import logging
import time

try:
    import urllib3
except ImportError:
    raise ImportError('python http client requires urllib3.')

from .utils import Utils
from .enums import HTTPFabricConnectionState
from .http_fabric_exception import HTTPFabricException
from .http_client import HTTPClient

logger = logging.getLogger(__name__)

class HTTPReverseRunnable(object):

    def __init__(self, connection):
        self._connection = connection
        self._running = True
        self._reverseWrapper = None

        self._reverseConnectionErrorsCount = 0
        self._reverseConnectionErrorsCountLast = 0
        self._reverseConnectionTimeoutedFirstTime = 0

        from stpy import HTTPRequestsStatistics
        self._reverseRequests = HTTPRequestsStatistics()

    def __call__(self):
        self._log(logger.info, "Starting reverse connection thread.")

        reverseResponse = None
        postTime = 0
        try:

            while self._running == True and self._connection.isOpened():

                if not self._reconnect():
                    continue

                try:
                    if self._reverseWrapper == None:
                        self._log(logger.debug, "Opening reverse connection...")
                        self._reverseWrapper = self._connection._createHttpClientWrapper(
                                timeout = self._connection._reverseConnectionReplyTimeoutMs)
                        reverseResponse = None

                    if reverseResponse == None:
                        self._log(logger.debug, "Reverse connection...")
                        postTime = time.time()
                        self._reverseRequests.requestsCount += 1
                        self._reverseRequests.bytesSent += 10 # TODO: calculate accurate length
                        reverseResponse = self._reverseWrapper.httpClient.post(query="reverse", data="")

                    self._log(logger.debug, "Reverse response got.")
                    getStatusTime = time.time()

                    if reverseResponse != None and reverseResponse.data != None:
                        self._reverseRequests.bytesReceived += len(reverseResponse.data)

                    self._log(logger.debug, "Deserializing reverse response data...")
                    answer = Utils.parseJsonResponse(reverseResponse)
                    self._log(logger.debug, "Reverse response data deseriaized.")

                    if reverseResponse.status == 401:
                        self._reverseWrapper.closeQuiet()
                        self._reverseWrapper = None
                        self._reverseRequests.unauthorizedResponsesCount += 1
                        self._log(logger.debug, "Reverse request returned status 401(unauthorized). It means that session is not valid or expired. It can be caused by acceptor or node restart. Trying to reopen...")
                        self._connection._reopen()
                        continue

                    self._reverseConnectionErrorsCountLast = self._reverseConnectionErrorsCount
                    self._reverseConnectionErrorsCount = 0
                    self._reverseConnectionTimeoutedFirstTime = 0

                    if reverseResponse.status == 200:
                        self._reverseRequests.successfulResponsesCount += 1
                    else:
                        self._reverseRequests.errorResponsesCount +=1

                    try:
                        if isinstance(answer, str) and len(answer) == 0:
                            #  to avoid infinite loop, sleep one second if received empty response
                            if reverseResponse.status != 200:
                                self._log(logger.debug, "HTTP server replied with empty response and status %s.", reverseResponse.getStatusCode())
                                Utils.sleep(getStatusTime, 1000)
                            else:
                                self._log(logger.debug, "HTTP server replied with empty response. Reopening reverse connection...")
                        elif answer == None:
                            #  to avoid infinite loop, sleep one second if received empty response
                            self._log(logger.debug, "HTTP server replied with null response.")
                            Utils.sleep(getStatusTime, 1000)
                        else:
                            self._processEvents(answer)
                    finally:
                        reverseResponse = None
                except urllib3.exceptions.TimeoutError as exception:
                    self._log(logger.debug, "Timeout exception when getting reverse data.")
                    self._reverseRequests.timeoutResponsesCount += 1
                    if time.time() - postTime >= (self._connection._reverseConnectionHoldTimeoutMs + 1000)/1000:
                        # connection timeouted
                        if self._reverseConnectionTimeoutedFirstTime == 0:
                            self._reverseConnectionTimeoutedFirstTime = time.time()
                            self._reverseConnectionErrorsCount += 1
                            self._reverseConnectionErrorsCountLast = 0
                except Exception as e:
                    self._log(logger.error, "Exception when executing reverse operation.")
                    self._log(logger.exception, e)

                    self._reverseRequests.exceptionResponsesCount += 1
                    self._reverseConnectionErrorsCount += 1
                    self._reverseConnectionErrorsCountLast = 0
                    self._reverseConnectionTimeoutedFirstTime = 0

                    if self._reverseWrapper:
                        self._reverseWrapper.closeQuiet()
                        self._reverseWrapper = None

        except Exception as exception:
            self._log(logger.error, "Reverse thread was interrupted. Close connection.")
            self._log(logger.exception, exception)
            self._connection.closeQuiet()

        if self._reverseWrapper:
            self._reverseWrapper.closeQuiet()
            self._reverseWrapper = None

        self._log(logger.info, "Reverse connection thread finished.")


    def stop(self):
        self._running = False

    def _isOpenedInternal(self):
        wrapper = None
        try:
            wrapper = self._connection._createHttpClientWrapper(timeout=5000)
            result = self._connection._invokeMethod(wrapper, HTTPClient.IS_OPENED_INTERNAL, [])
            return Utils.parseJsonResponseAndCheck(result, bool)
        finally:
            if wrapper != None:
                wrapper.closeQuiet()

    def _isNotOpenedException(self, exception):
        if type(exception) == HTTPFabricException and hasattr(exception, 'message') and exception.message is not None and " is not opened." in exception.message:
            self._log(logger.info, "WARNING: Connection not opened exception received, close fabric connection and stop reverse thread.")
            self._connection._closeInternalQuiet()
            return True
        return False

    def _reconnect(self):
        if self._reverseConnectionErrorsCount == 0:
            return True

        reconnectAttempts = self._connection.getReconnectAttempts()
        if reconnectAttempts == 0 or reconnectAttempts > 0 and self._reverseConnectionErrorsCount > reconnectAttempts:
            self._log(logger.info, "Reconnection attempts %s limit reached. Closing connection.", reconnectAttempts)
            self._connection.fabricConnectionLost()
            return False

        if self._reverseConnectionErrorsCount > 1:
            timeout = self._connection.getReconnectInterval()
            if self._connection._reopenFailedTime > self._connection._reopenTime:
                timeout = self._connection.getConnectionReopenInterval()
            Utils.sleep(time.time(), timeout * 1000)

        if not self._connection.isOpened():
            return False

        try:
            self._log(logger.info, "Reconnection attempt %s/%s ...", self._reverseConnectionErrorsCount, self._connection.getReconnectAttempts())
            isOpenedInternal = self._isOpenedInternal()
            if isOpenedInternal:
                self._log(logger.info, "Reconnection successful!")
                self._reverseConnectionErrorsCountLast = self._reverseConnectionErrorsCount
                self._reverseConnectionErrorsCount = 0
                self._connection._serverConnectionRepaired()
                return True
            else:
                self._reverseConnectionTimeoutedFirstTime = 0
                if self._connection.isOpened():
                    self._log(logging.info, "Connection on server closed. Closing fabric connection.")
                    self._connection._fabricConnectionLost()
                return False
        except Exception as exception:
            self._reverseConnectionTimeoutedFirstTime = 0

            self._log(logging.error, exception)

            if self._isNotOpenedException(exception):
                self._log(logging.info, "Connection on server closed. Closing fabric connection.")
                self._connection._fabricConnectionLost()
            else:
                self._connection._serverConnectionLost()
                self._connection._onStateChange(HTTPFabricConnectionState.RECONNECT_FAILED, exception)
            self._reverseConnectionErrorsCount += 1
            self._reverseConnectionErrorsCountLast = 0
            return False

    def _log(self, func, message, *args):
        if self._connection:
            self._connection._log(func, message, *args)
        else:
            Utils.log(func, message, args)

    def _processEvents(self, answer):
        logger.debug(answer)

