from __future__ import absolute_import

import sys
import io
import ssl
import logging
import threading
import time

from .utils import Utils
from .http_fabric_exception import HTTPFabricException

try:
    import urllib3
except ImportError:
    raise ImportError('python http client requires urllib3.')

logger = logging.getLogger(__name__)

class HTTPClient(object):
    unique = 0
    uniqueLock = threading.Lock()

    def __init__(self, connection, url, timeout = None, ssl = None):
        self._connection = connection
        self._url = url
        self._timeout = timeout

        self._username = None
        self._password = None

        self._ssl = {
            'cert_reqs' : None,
            'ca_certs' : None,
            'ca_cert_dir' : None,
            'ssl_version' : None,
            'key_file' : None,
            'cert_file' : None,
            'disable_warnings' : True
        }

        if ssl is not None:
            self._ssl['cert_reqs'] = ssl.get('cert_reqs')
            self._ssl['ca_certs'] = ssl.get('ca_certs')
            self._ssl['ca_cert_dir'] = ssl.get('ca_cert_dir')
            self._ssl['ssl_version'] = ssl.get('ssl_version')
            self._ssl['key_file'] = ssl.get('key_file')
            self._ssl['cert_file'] = ssl.get('cert_file')
            self._ssl['disable_warnings'] = ssl.get('disable_warnings', True)

        if self._ssl['disable_warnings']:
            urllib3.disable_warnings()

        self._pool_manager = None
        self._openHttpClient()

    def makeFabricUrlQuery(self, query = None, sessionId = None):
        urlQuery = ""
        if query != None:
            urlQuery += query + "&"

        if sessionId:
            urlQuery += "FabricSession=" + sessionId + "&"
        elif self._connection.isOpened() or query == "logout":
            urlQuery += "FabricSession=" + \
                        (self._connection._sessionId if self._connection._sessionId != None else "" ) + \
                        "&"

        with HTTPClient.uniqueLock:
            HTTPClient.unique = HTTPClient.unique + 1
            urlQuery += "unique=" + str(HTTPClient.unique)

        return urlQuery

    def makeFabricUrl(self, query):
        return self._url + self.makeFabricUrlQuery(query)

    def applyAuthentication(self, headers):
        if not self._username == None:
            headers['authorization'] = urllib3.util.make_headers(basic_auth=self._username + ':' + self._password).get('authorization')

    def _isOpenCall(self, call):
        return call and call['consequentMethodCalls'] and len(call['consequentMethodCalls']) == 1 \
               and call['consequentMethodCalls'][0]['methodName'] == "open"

    def _isIsOpenedCall(self, call):
        return call and call['consequentMethodCalls'] and len(call['consequentMethodCalls']) == 1 \
            and call['consequentMethodCalls'][0]['methodName'] == "isOpened"

    IS_OPENED_INTERNAL = "isOpenedInternal"
    def _isIsOpenedInternalCall(self, call):
        if call and call['consequentMethodCalls'] and len(call['consequentMethodCalls']) == 1 \
            and call['consequentMethodCalls'][0]['methodName'] == HTTPClient.IS_OPENED_INTERNAL:
            call['consequentMethodCalls'][0]['methodName'] = "isOpened"
            return True
        return False

    def invokeCall(self, remoteCall, query = None, timeout = None):
        params = None
        try:
            isIsOpenedInternalCall = self._isIsOpenedInternalCall(remoteCall)
            isIsOpenedCall = self._isIsOpenedCall(remoteCall)
            # we need sort_keys=True to place @type field to the first place
            data = Utils.dumpMapToJson(remoteCall)
            params = self._makePostParams(query, data, timeout)

            retry = 0
            retriesCount = self._connection._maxRetriesCount
            httpException = None
            response = None
            while response == None and retry < retriesCount:
                retry += 1
                self._log(logger.debug, "Executing POST request to %s, retry %s/%s, request data %s.", params['url'], retry, retriesCount, self._cutForLog(params['body']))

                startRequestTime = time.time()
                self._connection._allRequests.requestsCount += 1
                self._connection._allRequests.bytesSent += len(params['url']) + len(params['body'])
                if isIsOpenedCall:
                    self._connection._isOpenedRequests.requestsCount += 1
                    self._connection._isOpenedRequests.bytesSent += len(params['url']) + len(params['body'])

                try:
                    response = self._postInternal(params)
                except urllib3.exceptions.TimeoutError as exception:
                    self._log(logger.debug, "Socket timeout exception caught on %s, retry %s/%s after %s ms, on %s. Reopen http client and rethrow exception.",
                        params['url'], retry, retriesCount, (time.time() - startRequestTime) * 1000, "main client" if self._connection._httpClientWrapper and self == self._connection._httpClientWrapper._httpCLient else "not main client")

                    self._connection._allRequests.timeoutResponsesCount += 1
                    if isIsOpenedCall:
                        self._connection._isOpenedRequests.timeoutResponsesCount += 1

                    # reopen http client
                    try:
                        self._openHttpClient()
                    except Exception as e:
                        pass

                    raise exception
                # except urllib3.exceptions.RequestError as exception:
                except urllib3.exceptions.ProtocolError as exception:
                    httpException = exception

                    self._log(logger.debug, "IO exception caught on %s, retry %s/%s after %s ms, on %s. Reopen connection and %.",
                            params['url'], retry, retriesCount, (time.time() - startRequestTime)*1000,
                              "main client" if self._connection._httpClientWrapper and self == self._connection._httpClientWrapper._httpClient else "not main client",
                              "throw exception " if retry >= retriesCount else "resend request")

                    self._connection._allRequests.exceptionResponsesCount += 1
                    if isIsOpenedCall:
                        self._connection._isOpenedRequests.exceptionResponsesCount += 1

                    # Server seems to close the connection. Try to reopen it and resend the request.
                    # no reopen, so use URL from wrapper
                    self._openHttpClient()

                    if retry >= retriesCount:
                        raise exception

                    # retry on the next iteration
                    self._connection._allRequests.retriesCount += 1
                    if isIsOpenedCall:
                        self._connection._isOpenedRequests.retriesCount += 1

                except Exception as exception:
                    self._connection._allRequests.exceptionResponsesCount += 1
                    if isIsOpenedCall:
                        self._connection._isOpenedRequests.exceptionResponsesCount += 1
                    raise exception

                self._log(logger.debug, "Response on POST request received.")

            if response.status == 200:
                self._connection._allRequests.successfulResponsesCount += 1
                if isIsOpenedCall:
                    self._connection._isOpenedRequests.successfulResponsesCount += 1
            elif response.status == 401:
                self._connection._allRequests.unauthorizedResponsesCount += 1
                if isIsOpenedCall:
                    self._connection._isOpenedRequests.unauthorizedResponsesCount += 1
            else:
                self._connection._allRequests.errorResponsesCount += 1
                if isIsOpenedCall:
                    self._connection._isOpenedRequests.errorResponsesCount += 1

            if response.data:
                self._connection._allRequests.bytesReceived += len(response.data)
                if isIsOpenedCall:
                    self._connection._isOpenedRequests.bytesReceived += len(response.data)

            if response.data == "!@#$request with specifed unique id already processed$#@!":
                self._connection._allRequests.duplicateResponsesCount += 1
                if isIsOpenedCall:
                    self._connection._isOpenedRequests.duplicateResponsesCount += 1

                if httpException == None:
                    httpException = HTTPFabricException("Request already processed.")
                raise httpException

            if response.status != 200:
                self._log(logger.debug, "HTTP request url: %s", params['url'])
                self._log(logger.debug, "HTTP request userName: %s", self._connection._username)
                self._log(logger.debug, "HTTP request data: %s", self._cutForLog(params['body']))
                self._log(logger.debug, "HTTP response status: %s", response.status)
                self._log(logger.debug, "HTTP response content length: %s", len(response.data) if response.data else None)
                self._log(logger.debug, "HTTP response content: %s", self._cutForLog(response.data) if response.data else None)

                if response.status == 401:
                    # if connection not yet opened it means that user calls open method, throw exception to the user
                    if not self._connection.isOpened():
                        raise HTTPFabricException("Authentication failed.")

                    if self._connection._isReopening:
                        if self._isOpenCall(remoteCall):
                            raise HTTPFabricException("Authentication failed.")
                        else:
                            raise HTTPFabricException("Fabric connection lost. Reopening is in progress.")

                    self._log(logger.error, "Server responded with status 401(unauthorized). It means that session is not valid or expired. It can be caused by acceptor or node restart.")
                    # reopening connection only if it is isOpenedInternal call, that called from reverse thread
                    if self._connection.isReopenOnAnyCall() or isIsOpenedInternalCall:
                        self._connection._reopen()
                        # reopen was performed and if we are here, it was success
                        # if reopen succeeded, this.url is set and it can be set the new URL, so replace client with new URL
                        self._openHttpClient()
                        return self.invokeCall(remoteCall, query, timeout)
                    else:
                        raise HTTPFabricException("Fabric connection lost. Reconnecting is in progress.")
                else:
                    raise HTTPFabricException("Error http response received. Status %s. Data: %s.".
                                              format(response.getStatusCode(), self._cutForLog(response.data) if response.data else None))

            return response
        except Exception as exception:
            self._log(logger.error, "Failed to execute http request to %s. Error: %s", params['url'] if params else self.makeFabricUrlQuery(query), repr(exception))
            if type(exception) == HTTPFabricException:
                raise exception
            raise HTTPFabricException(exception.message if hasattr(exception, 'message') else "Failed to execute http request.", exception)

    def post(self, query, data, timeout = None, sessionId = None):
        params = self._makePostParams(query, data, timeout, sessionId)
        return self._postInternal(params)

    def _makePostParams(self, query, data, timeout = None, sessionId = None):
        headers = {
            "User-Agent" : "stpyclient",
            "Keep-Alive" : "timeout=600",
            "Connection" : "Keep-Alive",
            "Accept" : "application/json;notation=type;level=root_element,root_enum,complex_objects,enums"
        }
        self.applyAuthentication(headers)

        if timeout == None or timeout < 0:
            timeout = self._timeout
        if timeout == None:
            timeout = self._connection._httpTimeoutMs

        query = self.makeFabricUrlQuery(query, sessionId)
        url = self._url + '/fabric?' + query
        body = "5eeve2ec" + data

        return {
            'headers' : headers,
            'timeout' : timeout,
            'url' : url,
            'body' : body
        }

    def _postInternal(self, params):
        try:
            r = self._pool_manager.request("POST", params['url'], body=params['body'], headers=params['headers'], timeout=params['timeout'] / 1000)
        except urllib3.exceptions.SSLError as e:
            # msg = "{0}\n{1}".format(type(e).__name__, str(e))
            # raise ApiException(status=0, reason=msg)
            raise e
        except Exception as e:
            raise e

        r = RESTResponse(r)
        if Utils.is_python3():
            r.data = r.data.decode('utf8')

        # log response body
        #self._log(logger.debug, "response body: %s", r.data)

        return r

    def destroy(self):
        if self._pool_manager:
            self._pool_manager.clear()

    def _openHttpClient(self):
        if self._pool_manager:
            self._pool_manager.clear()
        self._pool_manager = urllib3.PoolManager(
                num_pools = 1,
                cert_reqs = self._ssl['cert_reqs'],
                ca_certs  = self._ssl['ca_certs'],
                ca_cert_dir = self._ssl['ca_cert_dir'],
                ssl_version = self._ssl['ssl_version'],
                key_file  = self._ssl['key_file'],
                cert_file = self._ssl['cert_file']
        )

    def _cutForLog(self, data):
        if data and len(data) > 1150:
            data = data[0:1000] + "\n.....\n.....\n" + data[len(data) - 100:]
        return data

    def _log(self, func, message, *args):
        if self._connection:
            self._connection._log(func, message, *args)
        else:
            Utils.log(func, message, args)

class RESTResponse(io.IOBase):

    def __init__(self, resp):
        self.urllib3_response = resp
        self.status = resp.status
        self.reason = resp.reason
        self.data = resp.data

    def getheaders(self):
        return self.urllib3_response.getheaders()

    def getheader(self, name, default=None):
        return self.urllib3_response.getheader(name, default)



