from __future__ import absolute_import

try:
    from exceptions import Exception, StandardError, Warning
except ImportError:
    # Python 3
    StandardError = Exception

def _build_message_causes(message, cause):
    if cause and hasattr(cause, "message"):
        message = message + u' Caused by ' + repr(cause) + ": " + cause.message
    elif cause and isinstance(cause, map):
        message = message + u' Caused by ' + repr(cause) + ": " + cause.get('detailMessage', None)

    return message


class DSError(StandardError):

    """Exception related to operation with Dataspace."""

    def __init__(self, message, cause=None):
        super(DSError, self).__init__(message, cause, _build_message_causes(message, cause))
        self.cause = cause
        self.message = _build_message_causes(message, cause)


class Warning(Warning, DSError):

    """Exception raised for important warnings like data truncations
    while inserting, etc."""

    def __init__(self, message, cause = None):
        super(Warning, self).__init__(message, cause)

class Error(DSError):

    """Exception that is the base class of all other error exceptions
    (not Warning)."""

    def __init__(self, message, cause = None):
        super(Error, self).__init__(message, cause)


class InterfaceError(Error):

    """Exception raised for errors that are related to the database
    interface rather than the database itself."""

    def __init__(self, message, cause = None):
        super(InterfaceError, self).__init__(message, cause)


class DatabaseError(Error):

    """Exception raised for errors that are related to the
    database."""

    def __init__(self, message, cause = None):
        super(DatabaseError, self).__init__(message, cause)


class DataError(DatabaseError):

    """Exception raised for errors that are due to problems with the
    processed data like division by zero, numeric value out of range,
    etc."""

    def __init__(self, message, cause = None):
        super(DataError, self).__init__(message, cause)


class OperationalError(DatabaseError):

    """Exception raised for errors that are related to the database's
    operation and not necessarily under the control of the programmer,
    e.g. an unexpected disconnect occurs, the data source name is not
    found, a transaction could not be processed, a memory allocation
    error occurred during processing, etc."""

    def __init__(self, message, cause = None):
        super(OperationalError, self).__init__(message, cause)


class IntegrityError(DatabaseError):

    """Exception raised when the relational integrity of the database
    is affected, e.g. a foreign key check fails, duplicate key,
    etc."""

    def __init__(self, message, cause = None):
        super(IntegrityError, self).__init__(message, cause)


class InternalError(DatabaseError):

    """Exception raised when the database encounters an internal
    error, e.g. the cursor is not valid anymore, the transaction is
    out of sync, etc."""

    def __init__(self, message, cause = None):
        super(InternalError, self).__init__(message, cause)


class ProgrammingError(DatabaseError):

    """Exception raised for programming errors, e.g. table not found
    or already exists, syntax error in the SQL statement, wrong number
    of parameters specified, etc."""

    def __init__(self, message, cause = None):
        super(ProgrammingError, self).__init__(message, cause)


class NotSupportedError(DatabaseError):

    """Exception raised in case a method or database API was used
    which is not supported by the database, e.g. requesting a
    .rollback() on a connection that does not support transaction or
    has transactions turned off."""

    def __init__(self, message, cause = None):
        super(NotSupportedError, self).__init__(message, cause)

