from __future__ import absolute_import

from .connection import Connection
from stpy import SQLType

apilevel = "2.0"

"""
Read only attribute.

0	Threads may not share the module.
1	Threads may share the module, but not connections.
2	Threads may share the module and connections.
3	Threads may share the module, connections and cursors.
"""
threadsafety = 1


"""
Read only attribute.

qmark	Question mark style, e.g. ...WHERE name=?
numeric	Numeric, positional style, e.g. ...WHERE name=:1
named	Named style, e.g. ...WHERE name=:name
format	ANSI C printf format codes, e.g. ...WHERE name=%s
pyformat	Python extended format codes, e.g. ...WHERE name=%(name)s
"""
paramstyle = "qmark"

try:
    frozenset
except NameError:
    from sets import ImmutableSet as frozenset

class DBAPISet(frozenset):
    """A special type of set for which A == x is true if A is a
    DBAPISet and x is a member of that set."""
    def __eq__(self, other):
        if isinstance(other, DBAPISet):
            return not self.difference(other)
        return other in self

# TODO:
# EVENT = "EVENT"
# ARRAY = "ARRAY"
# OTHER = "OTHER"

# TODO:
# conv - conversion map of dataspace objects to python objects

STRING    = DBAPISet([SQLType.CHAR, SQLType.VARCHAR, SQLType.LONGVARCHAR, SQLType.NCHAR, SQLType.NVARCHAR, SQLType.LONGNVARCHAR,
                      SQLType.STRING, SQLType.CLOB, SQLType.URL])
BINARY    = DBAPISet([SQLType.BINARY, SQLType.VARBINARY, SQLType.LONGVARBINARY, SQLType.BLOB, SQLType.FLOB])
NUMBER    = DBAPISet([SQLType.TINYINT, SQLType.SMALLINT, SQLType.INTEGER, SQLType.LONG, SQLType.BIGINT,
                      SQLType.REAL, SQLType.DOUBLE, SQLType.FLOAT, SQLType.NUMERIC, SQLType.DECIMAL,
                      SQLType.UTINYINT, SQLType.USMALLINT, SQLType.UINTEGER, SQLType.UBIGINT, SQLType.BOOLEAN, SQLType.BIT])
DATE      = DBAPISet([SQLType.DATE])
TIME      = DBAPISet([SQLType.TIME])
TIMESTAMP = DBAPISet([SQLType.TIMESTAMP])
DATETIME  = TIMESTAMP

ROWID     = DBAPISet()

def define_type(type):
    if STRING == type:
        return STRING
    elif BINARY == type:
        return BINARY
    elif NUMBER == type:
        return NUMBER
    elif DATE == type:
        return DATE
    elif TIME == type:
        return TIME
    elif TIMESTAMP == type:
        return TIMESTAMP
    elif DATETIME == type:
        return DATETIME
    else:
        return STRING

def Binary(x):
    return str(x)

def connect(url, dataspace, username, password, autocommit = True, timeout = 30, ssl = None):
    """
    Creates a connection to the dataspace.

    url
      string, URL of the node to connect to in format http[s]://<host>:<port>

    dataspace
      string, dataspace to connect to in format <DataspaceType>.<DataspaceName>

    username
      string, user to connect as

    password
      string, password to use

    autocommit
      If True (default), autocommit is enabled.
      If False, autocommit is disabled.
      If None, autocommit isn't set and server default is used.

    timeout
      http timeout in seconds.

    ssl
      dictionary with SSL connection parameters:
        - cert_reqs - specifies whether a certificate is required from the other side of the connection, and whether it will be validated if provided.
                      Should be one of the following values:
                        * CERT_NONE or None - certificates ignored
                        * CERT_OPTIONAL - not required, but validated if provided
                        * CERT_REQUIRED - required and validated
                      If the value of this parameter is not CERT_NONE, then the ca_certs parameter must point to a file of CA certificates.
        - ca_certs - file contains a set of concatenated 'certification authority' certificates, which are used to validate
                   certificates passed from the other end of the connection.
        - ca_cert_dir - a directory containing CA certificates in multiple separate files, as supported by OpenSSL's -CApath flag or
                        the capath argument to SSLContext.load_verify_locations().
        - ssl_version - specifies which version of the SSL protocol to use. Optional.
        - key_file and cert_file - optional files which contain a certificate to be used to identify the local side of the connection.
        - disable_warnings - specifies disable or not InsecureRequestWarning warning, by default True.

    """

    return Connection(url, dataspace, username, password, autocommit, timeout, ssl)

