package com.streamscape.sample.ws.events.servlet;

import java.io.IOException;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.WebSocketException;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketClose;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketConnect;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketError;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketMessage;
import org.eclipse.jetty.websocket.api.annotations.WebSocket;
import org.eclipse.jetty.websocket.servlet.ServletUpgradeRequest;

import com.streamscape.Trace;
import com.streamscape.sdo.EventDatagram;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sef.EventAsyncConsumer;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.network.http.server.authentication.AuthenticationHelper;
import com.streamscape.sef.network.http.server.fabric.HTTPServerFabricConnection;
import com.streamscape.sef.network.http.server.jetty.fabric.FabricConnectionsManager;
import com.streamscape.sef.network.http.server.utils.HTTPUtils;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Sample Web Socket implementation for raise/consume events.
 *
 * <p>Copyright: Copyright (c) 2015</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.5
 */
 @WebSocket
 public class EventsWebSocket
 {
    private EventAsyncConsumer consumer;
    
    @OnWebSocketConnect
    public void onConnect(final Session session)
    {
       Trace.logInfo(this, "Web socket from " + session.getRemoteAddress().getHostString() + " connected, creating consumer...");

       HTTPServerFabricConnection fabricConnection = getHttpFabricConnectionOrThrowException(session);
       HttpServletRequest request = getHttpServletRequest(session);
       String eventId = request.getParameter("eventId");
       if (eventId == null)
          sendErrorAndThrowException(session, "Error: eventId is not specified.");
       
       try
       {
          dropConsumer(session);
          consumer = fabricConnection.createEventAsyncConsumer("AsyncEventConsumerForEventsWebSocket" + System.currentTimeMillis(), new FabricEventListener()
          {
             public void onEvent(ImmutableEventDatagram event) throws FabricEventException
             {
                if (session.isOpen())
                {
                   byte [] serializedEvent = null;
                   try
                   {
                      serializedEvent = HTTPUtils.getJsonSerializerForRest().serialize(event);
                   }
                   catch(Exception exception)
                   {
                      serializedEvent = ("Failed to serialize event [" + event.getEventId() + "]. Cause: " + exception.getMessage()).getBytes();
                   }
                   try
                   {
                      session.getRemote().sendString(new String(serializedEvent));
                   }
                   catch (IOException exception)
                   {
                      throw new WebSocketException("I/O exception.", exception);
                   }
                }
             }
          }, eventId, null, EventScope.GLOBAL, true);
          consumer.start();
       }
       catch (Exception exception)
       {
          sendErrorAndThrowException(session, "Failed to create event consumer on event [" + eventId + "]. Cause: " + exception.getMessage(), exception);
       }

       Trace.logInfo(this, "Web socket from " + session.getRemoteAddress().getHostString() + " connected, consumer on event [" + eventId + "] created.");

       sendMessage(session, "Consumer on [" + eventId + "] created.");
    }
    
    @OnWebSocketMessage
    public void onTextMessage(Session session, String message)
    {
       Trace.logDebug(this, "Web socket message " + message + " received from " + session.getRemoteAddress().getHostString() + ", raising event...");
       if (message.equals("Ping"))
       {
          getHttpFabricConnectionOrThrowException(session).touch();
          AuthenticationHelper.touchSession(getHttpServletRequest(session).getSession());
          return;
       }
       
       EventDatagram event = null;
       try
       {
          event = (EventDatagram) HTTPUtils.getJsonSerializerForRest().deserialize(message);
       }
       catch(Exception exception)
       {
          Trace.logError(this, "Failed to deserialize event.");
          Trace.logException(this, exception, false);
          sendMessage(session, "Failed to deserialize event.");
          return;
       }
       
       HTTPServerFabricConnection fabricConnection = getHttpFabricConnectionOrThrowException(session);
       try
       {
          if (!fabricConnection.isBoundEventId(event.getEventId()))
             fabricConnection.bindProducerFor(event.getEventId());
          fabricConnection.raiseEvent(event, EventScope.GLOBAL, -1);
       }
       catch (Exception exception)
       {
          Trace.logError(this, "Failed to send event.");
          Trace.logException(this, exception, true);
          sendMessage(session, "Failed to send event.");
          return;
       }
    }

    @OnWebSocketClose
    public void onClose(Session session, int status, String reason)
    {
       Trace.logInfo(this, "Web socket from " + session.getRemoteAddress().getHostString() + " closed. Status code: " + status + ", reason: " + reason + ".");
       dropConsumer(session);
    }

    @OnWebSocketError
    public void onError(Session session, Throwable error)
    {
       Trace.logError(this, "Web socket from " + session.getRemoteAddress().getHostString() + " error. Cause: " + error);
       Trace.logException(this, error, true);
    }
    
    private void sendMessage(Session session, String message)
    {
       try
       {
          session.getRemote().sendString(message);
       }
       catch (IOException exception)
       {
          Trace.logError(this, "Send to ws failed. Cause: " + exception);
       }
    }
    
    private void dropConsumer(Session session)
    {
       if (consumer != null)
       {
          consumer.stop();
          HTTPServerFabricConnection fabricConnection = getHttpFabricConnectionOrThrowException(session);
          if (fabricConnection != null)
          {
             try
             {
                fabricConnection.dropConsumer(consumer.getName());
             }
             catch (Exception exception)
             {
             }
          }
       }
    }
    
    ///////////////////////////////////////

    private void sendErrorAndThrowException(Session session, String error)
    {
       sendErrorAndThrowException(session, error, null);
    }

    private void sendErrorAndThrowException(Session session, String error, Throwable cause)
    {
       sendMessage(session, error);
       throw new WebSocketException(error, cause);
    }
    
    private static HTTPServerFabricConnection getHttpFabricConnectionOrThrowException(Session session)
    {
       HttpSession httpSession = getHttpServletRequest(session).getSession(false);
       if (httpSession == null)
          throw new WebSocketException("Http session not created for ws session " + session);
       
       HTTPServerFabricConnection connection = getHttpFabricConnection(session);
       if (connection == null || connection.isRemoved())
       {
          Trace.logError(EventsWebSocket.class, "Fabric connection " + (connection == null ? "doesn't exist" : "removed") + " in http session " + httpSession.getId());
          throw new WebSocketException("Can't get access to HTTP fabric connection. Not logged in session " + httpSession.getId() + " or session is expired.");
       }
       return connection;
    }

    private static HTTPServerFabricConnection getHttpFabricConnection(Session session)
    {
       HttpServletRequest request = getHttpServletRequest(session);
       HttpSession httpSession = getHttpServletRequest(session).getSession(false);
       if (httpSession == null)
          throw new WebSocketException("Http session not created for ws session " + session);
       return ((FabricConnectionsManager) request.getServletContext().getAttribute(FabricConnectionsManager.ATTRIBUTE)).getFabricConnection(httpSession.getId());
    }

    private static HttpServletRequest getHttpServletRequest(Session session)
    {
       return ((ServletUpgradeRequest)session.getUpgradeRequest()).getHttpServletRequest();
    }
 }
