package streaming;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.concurrent.FabricThreadManager;
import com.streamscape.lib.utils.Utils;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.sdo.EventDatagram;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.event.ChunkEvent;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.StreamEvent;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.enums.EventScope;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: First <i>Streaming</i> sample.
 *
 * <p>This sample shows how to receive mixed chunks (binary or text) from two different streams.</p>
 *
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.6
 */
 public class MixedStreamSample
 {
    static RuntimeContext   context;
    static FabricConnection consumerConnection ; // It is used for consumers.
    static FabricConnection producerConnection1; // It is used for event sending.
    static FabricConnection producerConnection2; // It is used for event sending.

    static final String     streamEventId1 = "event.mixed.stream1";
    static final String     chunkEventId1  = ChunkEvent.getEventId(streamEventId1);
    static final String     streamEventId2 = "event.mixed.stream2";
    static final String     chunkEventId2  = ChunkEvent.getEventId(streamEventId2);
    static final String     streamEventId3 = "event.mixed.stream3";
    static final String     chunkEventId3  = ChunkEvent.getEventId(streamEventId3);
    static final String     textEventId    = "event.mixed.text";

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("streaming.*",               Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node1");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node1");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("\nAdd event prototypes...");
          addEventPrototypes();

          System.out.println("Create consumers...");
          createConsumers();

          System.out.println("Create producers...");
          createProducers();

          System.out.println("\nRaise Stream event and Text events...\n");
          raiseEvents1();

          Utils.sleep(3000);
          
          System.out.println("\nRaise two Stream events...\n");
          raiseEvents2();

          Utils.sleep(3000);
          
          System.out.println("\nShutting down...\n");
          shutdown();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void addEventPrototypes() throws Exception
    {
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();

       // Stream Event 1
       if (!prototypeFactory.existsPrototype(streamEventId1))
          prototypeFactory.addEventPrototype("StreamEvent", streamEventId1);
       // Note that user does not need to directly add ChunkEvent prototype for corresponding stream.

       // Stream Event 2
       if (!prototypeFactory.existsPrototype(streamEventId2))
          prototypeFactory.addEventPrototype("StreamEvent", streamEventId2);
       // Note that user does not need to directly add ChunkEvent prototype for corresponding stream.

       // Stream Event 3
       if (!prototypeFactory.existsPrototype(streamEventId3))
          prototypeFactory.addEventPrototype("StreamEvent", streamEventId3);
       // Note that user does not need to directly add ChunkEvent prototype for corresponding stream.
       
       // Text Event
       if (!prototypeFactory.existsPrototype(textEventId))
       {
          EventDatagram event = context.getEventDatagramFactory().newEventInstance("TextEvent");
          event.setEventIntProperty(ChunkEvent.CHUNK_NUMBER_PROPERTY, 0); // Just for convenience.
          prototypeFactory.addEventPrototype(textEventId, event);
       }
    }

    static void createConsumers() throws Exception
    {
       consumerConnection = new FabricConnectionFactory().createConnection();
       consumerConnection.open();

       // Creates the asynchronous consumer for processing of ChunkEvents and TextEvents for the first sample.
       consumerConnection.createEventAsyncConsumer("Consumer1", new Listener1(), chunkEventId1 + "&" + textEventId,
                                                    null, EventScope.OBSERVABLE, true).start();

       // Creates the asynchronous consumer for processing of ChunkEvents for the second sample.
       consumerConnection.createEventAsyncConsumer("Consumer2", new Listener2(), chunkEventId2 + "&" + chunkEventId3,
                                                    null, EventScope.OBSERVABLE, true).start();
    }

    static class Stream1Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             Utils.sleep(500);
             System.out.println("Confirmation: " + new String(((StreamEvent)event).getBytes()) + "\n\n");
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class Listener1 implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             if (event instanceof ChunkEvent)
             {
                ChunkEvent chunkEvent = (ChunkEvent)event;
                System.out.print("Binary #" + chunkEvent.getChunkNumber() + ": " + new String(chunkEvent.getBytes()));
             }
             if (event instanceof TextEvent)
             {
                TextEvent textEvent = (TextEvent)event;
                System.out.println(" ... Text #" + textEvent.getEventStringProperty(ChunkEvent.CHUNK_NUMBER_PROPERTY) + ": " + textEvent.getText());
             }
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class Listener2 implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             ChunkEvent chunkEvent = (ChunkEvent)event;
             if (event.getEventId().equalsIgnoreCase(chunkEventId2))
                System.out.print("Binary #" + chunkEvent.getChunkNumber() + ": " + new String(chunkEvent.getBytes()));
             if (event.getEventId().equalsIgnoreCase(chunkEventId3))
                System.out.println(" ... Text #" + chunkEvent.getChunkNumber() + ": " + new String(chunkEvent.getBytes()));
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static void createProducers() throws Exception
    {
       producerConnection1 = new FabricConnectionFactory().createConnection();
       producerConnection1.open();
       producerConnection1.bindProducerFor(streamEventId1);
       producerConnection1.bindProducerFor(streamEventId2);

       producerConnection2 = new FabricConnectionFactory().createConnection();
       producerConnection2.open();
       producerConnection2.bindProducerFor(textEventId);
       producerConnection2.bindProducerFor(streamEventId3);
    }

    static void raiseEvents1() throws Exception
    {
       FabricThreadManager.getInstance().createThread("ProducerThread1", "Raises Stream Event.", () ->
       {
          try
          {
             byte[] bytes = "BinaryChunk1BinaryChunk2BinaryChunk3".getBytes();

             StreamEvent event = (StreamEvent)EventDatagramFactory.getInstance().createEvent(streamEventId1);
             event.writeBytes(bytes);
             event.setChunkSize(bytes.length / 3);
             event.setChunkDelay(500); // 500 milliseconds
             
             producerConnection1.raiseEvent(event, EventScope.INHERITED, -1);
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }).start();

       FabricThreadManager.getInstance().createThread("ProducerThread2", "Raises Text Events.", () ->
       {
          try
          {
             for (int i = 1; i<= 3; ++i)
             {
                TextEvent event = (TextEvent)EventDatagramFactory.getInstance().createEvent(textEventId);
                event.setText("TextChunk" + i);
                event.setEventIntProperty(ChunkEvent.CHUNK_NUMBER_PROPERTY, i);

                Utils.sleep(500);
                producerConnection2.raiseEvent(event, EventScope.INHERITED, -1);
             }
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }).start();
    }

    static void raiseEvents2() throws Exception
    {
       FabricThreadManager.getInstance().createThread("ProducerThread3", "Raises Stream Event 1.", () ->
       {
          try
          {
             byte[] bytes = "BinaryChunk1BinaryChunk2BinaryChunk3".getBytes();

             StreamEvent event = (StreamEvent)EventDatagramFactory.getInstance().createEvent(streamEventId1);
             event.writeBytes(bytes);
             event.setChunkSize(bytes.length / 3);
             event.setChunkDelay(500); // 500 milliseconds

             producerConnection1.raiseEvent(event, EventScope.INHERITED, -1);
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }).start();

       FabricThreadManager.getInstance().createThread("ProducerThread4", "Raises Stream Event 2.", () ->
       {
          try
          {
             byte[] bytes = "TextChunk1TextChunk2TextChunk3".getBytes();

             StreamEvent event = (StreamEvent)EventDatagramFactory.getInstance().createEvent(streamEventId3);
             event.writeBytes(bytes);
             event.setChunkSize(bytes.length / 3);
             event.setChunkDelay(600); // 600 milliseconds

             producerConnection2.raiseEvent(event, EventScope.INHERITED, -1);
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }).start();
    }

    static void shutdown() throws Exception
    {
       consumerConnection.close();
       producerConnection1.close();
       producerConnection2.close();

       context.shutdown();
    }
 }
