package service.triggers;

import com.streamscape.Trace;
import com.streamscape.cli.service.ServiceAccessor;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationFactory;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationObject;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.advisory.ServiceMetricAdvisory;
import com.streamscape.sdo.advisory.ServiceStateAdvisory;
import com.streamscape.sdo.enums.StateAdvisoryType;
import com.streamscape.sdo.enums.Unit;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sdo.utils.SDOUtils;
import com.streamscape.sef.ConfigurationProperty;
import com.streamscape.sef.FabricComponentProperties;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.advisories.StateAdvisoryDefinition;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.metrics.MetricDefinition;
import com.streamscape.sef.service.ServiceDescriptor;
import com.streamscape.sef.service.ServiceManager;
import com.streamscape.sef.service.ServiceManagerException;
import com.streamscape.service.osf.eim.EventIdentityProcessStart;
import com.streamscape.service.osf.enums.InvokeMode;
import com.streamscape.service.osf.evh.EventHandler;

import java.io.BufferedReader;
import java.io.InputStreamReader;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Event triggers and advisories samples.
 * 
 * <p>Copyright: Copyright (c) 2012</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.3
 */
 public class ServiceSample
 {
    protected static final Class<?> SAMPLE_SERVICE_CLASS     = SampleServiceObject.class;
    protected static final String   SAMPLE_SERVICE_TYPE      = SAMPLE_SERVICE_CLASS.getSimpleName();
    protected static final String   SAMPLE_SERVICE_NAME      = SAMPLE_SERVICE_TYPE + "Name";
    protected static final String   SAMPLE_SERVICE_FULL_NAME = SAMPLE_SERVICE_TYPE + "." + SAMPLE_SERVICE_NAME;

    protected static final String   SAMPLE_SERVICE_METRIC_NAME       = "SampleServiceCounterMetric";
    protected static final String   SAMPLE_SERVICE_NOTIFICATION_NAME = "SampleServiceNotification"; 
    
    protected static final String   EVENT_ID_TO_SERVICE_HANDLER    = "event.to.service.handler";
    protected static final String   EVENT_ID_FROM_SERVICE_HANDLER  = "event.from.service.handler";
    protected static final String   EVENT_REQUEST_ID_TO_SERVICE_HANDLER = "event.request.to.service.handler";
    
    protected static FabricConnection connection; 
    
    public static void main(String[] args) throws Exception
    {
       // Enables some traces.
       Trace.enable("com.streamscape.sef.service.*", Trace.Level.INFO);
       Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
       Trace.enable("*",                  Trace.Level.ERROR);

       System.setProperty(RuntimeContext.STARTUP_DIR, "./");
       System.setProperty(RuntimeContext.DEPLOYMENT, "./");

       try
       {
          // Initializes the Runtime Context.
          RuntimeContext.getInstance();

          // create event prototypes
          createEventPrototypes();

          // create event consumers
          createEventConsumers();

          // create service sco, register and start service
          setUpSampleService();

          // create triggers in slang
          createTriggersInSlang();

          // raise events
          raiseEvents();
          
          // stop service
          stopService();

          // clean up service
          cleanUpSampleService();
       }
       catch (Exception e)
       {
          e.printStackTrace();
       }
       System.exit(0);
    }
    
    protected static void createEventPrototypes() throws Exception
    {
       connection = new FabricConnectionFactory().createConnection("admin", "Admin");
       connection.open();

       SDOUtils.addEventPrototype("TextEvent", EVENT_ID_TO_SERVICE_HANDLER);
       SDOUtils.addEventPrototype("TextEvent", EVENT_ID_FROM_SERVICE_HANDLER);
       SDOUtils.addEventPrototype("TextEvent", EVENT_REQUEST_ID_TO_SERVICE_HANDLER);
    }

    protected static void createEventConsumers() throws Exception
    {
       connection.createEventAsyncConsumer("AsyncConsumer", new SampleFabricEventListener(), EVENT_ID_FROM_SERVICE_HANDLER, null, EventScope.OBSERVABLE, true).start();
    }
    
    protected static void createTriggersInSlang() throws Exception
    {
       ServiceAccessor serviceAccessor = connection.createServiceAccessor(SAMPLE_SERVICE_TYPE, SAMPLE_SERVICE_NAME);
       SLResponse response;
       response = serviceAccessor.invokeLanguageRequest(
             "create event trigger ServiceStateAdvisoryLogger\n" +
             " event scope inherited\n" +
             " after event [" + ServiceStateAdvisory.EVENT_ID + "]\n" +
             " as\n" +
             " {\n" +
             "   log.info('Service state advisory header text', this.getCurrentEvent(), 'xml');\n" +
             " } enabled\n");
       if (!response.isOK())
          System.out.println("Failed to create service state advisory logger trigger: " + response.getException());

       response = serviceAccessor.invokeLanguageRequest(
             "create event trigger ServiceMetricAdvisoryLogger\n" +
             " event scope inherited\n" +
             " after event [" + ServiceMetricAdvisory.EVENT_ID + "]\n" +
             " as\n" +
             " {\n" +
             "   log.info('Service metric advisory header text', this.getCurrentEvent(), 'xml');\n" +
             " } enabled\n");
       if (!response.isOK())
          System.out.println("Failed to create service metric advisory  logger trigger: " + response.getException());

       response = serviceAccessor.invokeLanguageRequest(
             "create event trigger EventResponsePublisherTrigger\n" +
             " event scope inherited\n" +
             " after event [" + EVENT_ID_FROM_SERVICE_HANDLER + "]\n" +
             " as\n" +
             " {\n" +
             "   raise event this.getCurrentEvent();\n" +
             " } enabled\n");
       if (!response.isOK())
          System.out.println("Failed to publish trigger: " + response.getException());
       serviceAccessor.close();
    }
    
    protected static void setUpSampleService() throws Exception
    {
       // create service prototype sco
       ServiceConfigurationObject servicePrototypeSco = ServiceConfigurationFactory.createDefaultServiceConfiguration
             (
                RuntimeContext.getInstance(), 
                SAMPLE_SERVICE_TYPE
             );
       servicePrototypeSco.setServiceClassName(SAMPLE_SERVICE_CLASS.getName());
       servicePrototypeSco.setEIMPluginName(EventIdentityProcessStart.class.getSimpleName());
       ServiceConfigurationFactory.saveConfigurationObject(servicePrototypeSco);
       
       // create service sco
       ServiceConfigurationObject sco = ServiceConfigurationFactory.createServiceConfiguration
             (
                RuntimeContext.getInstance(), 
                SAMPLE_SERVICE_NAME,
                SAMPLE_SERVICE_TYPE,
                false
             );
             
       sco.setServiceClassName(SAMPLE_SERVICE_CLASS.getName());
       sco.setEIMPluginName(EventIdentityProcessStart.class.getSimpleName());
       sco.setInterruptableService(false);
       sco.setInvokeMode(InvokeMode.ASYNC);

       ConfigurationProperty property = sco.createAdvancedProperty(EventIdentityProcessStart.EVENT_GROUP_IDENTITY,
             EventIdentityProcessStart.EVENT_GROUP_IDENTITY, 
             EventIdentityProcessStart.EVENT_GROUP_IDENTITY);
       property.setTextValue("Sample Service");
       sco.addAdvancedProperty(property);

       // create event handlers
       EventHandler handler = sco.createEventHandler("handleEventMethod", "HandleEventHandler");
       ImmutableEventDatagram prototype = EventDatagramFactory.getInstance().createEvent(EVENT_ID_TO_SERVICE_HANDLER);
       handler.bindRequestEvent(prototype);
       prototype = EventDatagramFactory.getInstance().createEvent(EVENT_ID_FROM_SERVICE_HANDLER);
       handler.bindResponseEvent(prototype);
       handler.setInvokeMode(InvokeMode.ASYNC);
       handler.setTimeout(30000);
       sco.addEventHandler(handler);

       // create request handler
       handler = sco.createEventHandler("onRequest", "RequestHandler");
       prototype = EventDatagramFactory.getInstance().createEvent(EVENT_REQUEST_ID_TO_SERVICE_HANDLER);
       handler.bindRequestEvent(prototype);
       prototype = EventDatagramFactory.getInstance().createEvent(EVENT_ID_FROM_SERVICE_HANDLER);
       handler.bindResponseEvent(prototype);
       handler.setInvokeMode(InvokeMode.ASYNC);
       handler.setTimeout(30000);
       sco.addEventHandler(handler);

       // create metric
       MetricDefinition metric = new MetricDefinition(SAMPLE_SERVICE_METRIC_NAME);
       metric.setDescription("Sample counter metric");
       metric.setIncrementUnits(1);
       metric.setMaxThreshold(5);
       metric.setMinThreshold(0);
       metric.setNotificationEnabled(true);
       metric.setUnits(Unit.Times);
       sco.addMetric(metric);
       
       // add advisories to actionable events
       sco.addActionableEvent(ServiceMetricAdvisory.EVENT_ID);
       sco.addActionableEvent(ServiceStateAdvisory.EVENT_ID);
       
       //notifications
       StateAdvisoryDefinition destateAdvisoryDefinition = new StateAdvisoryDefinition(SAMPLE_SERVICE_NOTIFICATION_NAME);
       destateAdvisoryDefinition.setType(StateAdvisoryType.INFORMATION);
       sco.addStateAdvisory(destateAdvisoryDefinition);

       // set suppress stack trace property
       property = sco.createAdvancedProperty(FabricComponentProperties.EXCEPTION_SUPPRESS_STACKTRACE,
                                             FabricComponentProperties.EXCEPTION_SUPPRESS_STACKTRACE,
                                             FabricComponentProperties.EXCEPTION_SUPPRESS_STACKTRACE);
       property.setBooleanValue(true);
       sco.addAdvancedProperty(property);
       
       ServiceConfigurationFactory.saveConfigurationObject(sco);
       
       // register and start service 
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       if (!svcManager.isServiceRegistered(SAMPLE_SERVICE_FULL_NAME))
       {
          ServiceDescriptor service = new ServiceDescriptor(SAMPLE_SERVICE_TYPE, SAMPLE_SERVICE_NAME);
          svcManager.registerService(service);
          svcManager.runServiceAs(SAMPLE_SERVICE_FULL_NAME, "admin", "Admin");
       }
       
       svcManager.startService(SAMPLE_SERVICE_FULL_NAME);
    }
    
    protected static void cleanUpSampleService() throws Exception
    {
       ServiceConfigurationFactory.removeConfigurationObject(SAMPLE_SERVICE_NAME, SAMPLE_SERVICE_TYPE);
    }
    
    protected static void stopService() throws ServiceManagerException
    {
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       svcManager.stopService(SAMPLE_SERVICE_FULL_NAME);
       svcManager.unregisterService(SAMPLE_SERVICE_FULL_NAME);
    }
 
    protected static class SampleFabricEventListener implements  FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          System.out.println("SampleFabricEventListener: event received: " + event.getEventId());
       }
    }
    
    protected static void raiseEvents() throws Exception
    {
       connection.bindProducerFor(EVENT_ID_TO_SERVICE_HANDLER);
       connection.bindProducerFor(EVENT_REQUEST_ID_TO_SERVICE_HANDLER);

       InputStreamReader converter = new InputStreamReader(System.in);
       BufferedReader in = new BufferedReader(converter);
       
       System.out.println("\nSample service demonstrates event handlers, event triggers and advisories:");
       System.out.println(" - service handler receives events " + EVENT_ID_TO_SERVICE_HANDLER + " and raises received event with id " + EVENT_ID_FROM_SERVICE_HANDLER);
       System.out.println("\tif event contains word 'metric' then metric value increased");
       System.out.println("\tif event contains word 'notification' then notification raised");
       System.out.println(" - service handler onRequest receives events " + EVENT_REQUEST_ID_TO_SERVICE_HANDLER + " and throws exception");
       System.out.println(" - EventLogger triggers are registered for service advisory events");
       System.out.println(" - EventPublisher trigger is registered for " + EVENT_ID_FROM_SERVICE_HANDLER + " event");
       System.out.println(" - metric has range 0-5, so there should be 6 events to get metric advisory");
       
       String line = "";
       while(!line.equalsIgnoreCase("quit"))
       {
          
          do
          {
             System.out.println("\nActions:\n");
             System.out.println("\tmetric - raises " + EVENT_ID_TO_SERVICE_HANDLER + " event to service handler, and service handler increase metric");
             System.out.println("\tnotification - raises " + EVENT_ID_TO_SERVICE_HANDLER + " event to service handler, and service handler raises notifcation");
             System.out.println("\trequest - raises " + EVENT_REQUEST_ID_TO_SERVICE_HANDLER + " request to service request handler that throws exception");
             System.out.println("\tany word - raises " + EVENT_ID_TO_SERVICE_HANDLER + " event to service handler");
             System.out.print(">");
             line = in.readLine();
          }
          while (line.trim().equals(""));

          if (line.trim().equals("request"))
          {
             TextEvent request = (TextEvent) EventDatagramFactory.getInstance().createEvent(EVENT_REQUEST_ID_TO_SERVICE_HANDLER);
             request.setText(line);
             
             ServiceAccessor accessor = connection.createServiceAccessor("SampleServiceObject", "SampleServiceObjectName");
             try
             {
                accessor.invokeServiceRequest(request);
             }
             catch (Exception exception)
             {
                System.out.println(exception.getMessage());
                exception.printStackTrace();
             }
          }
          else
          {
             TextEvent textEvent = (TextEvent) EventDatagramFactory.getInstance().createEvent(EVENT_ID_TO_SERVICE_HANDLER);
             textEvent.setText(line);
             connection.raiseEvent(textEvent, EventScope.INHERITED, -1);
          }
          
          Thread.sleep(1000);
       }
    }
 }
