package service.dsl;


import com.streamscape.Trace;
import com.streamscape.cli.service.ServiceAccessor;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationFactory;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationObject;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sef.service.ServiceDescriptor;
import com.streamscape.sef.service.ServiceManager;
import com.streamscape.sef.service.ServiceManagerException;
import com.streamscape.sef.service.dsl.DSLSyntaxFactory;
import com.streamscape.sef.service.dsl.DSLSyntaxFactoryManager;
import com.streamscape.service.osf.eim.EventIdentityProcessStart;
import com.streamscape.service.osf.enums.InvokeMode;
import com.streamscape.slex.lang.DSLStatementSyntax;
import com.streamscape.slex.lang.parameter.StringParameter;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Demonstrates using of custom service DSL operations.
 * 
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.3
 */
 public class Sample
 {
    protected static final Class<?> SERVICE_CLASS     = DSLProcessorSampleService.class;
    protected static final String   SERVICE_TYPE      = SERVICE_CLASS.getSimpleName();
    protected static final String   SERVICE_NAME      = "Default";
    protected static final String   SERVICE_FULL_NAME = SERVICE_TYPE + "." + SERVICE_NAME;

    public static void main(String[] args) throws Exception
    {
       // Enables some traces.
       Trace.enable("com.streamscape.sef.service.*", Trace.Level.INFO);
       Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
       Trace.enable("*",                  Trace.Level.ERROR);

       System.setProperty(RuntimeContext.STARTUP_DIR, "./");
       System.setProperty(RuntimeContext.DEPLOYMENT, "./");

       try
       {
          // Initializes the Runtime Context.
          RuntimeContext.getInstance();

          // create service sco, register and start service
          createServiceSco();

          // create and register syntaxes
          createDSLSyntaxes();
          
          // start service
          startService();
          
          useCustomDSLOperations();

          Thread.sleep(1000000);
          // stop service
          stopService();
       }
       catch (Exception e)
       {
          e.printStackTrace();
       }
       System.exit(0);
    }
    
    protected static void createServiceSco() throws Exception
    {
       // create service sco
       ServiceConfigurationObject sco = ServiceConfigurationFactory.createServiceConfiguration
             (
                RuntimeContext.getInstance(), 
                SERVICE_NAME,
                SERVICE_TYPE,
                false
             );
             
       sco.setServiceClassName(SERVICE_CLASS.getName());
       sco.setEIMPluginName(EventIdentityProcessStart.class.getSimpleName());
       sco.setInterruptableService(false);
       sco.setInvokeMode(InvokeMode.ASYNC);
       
       ServiceConfigurationFactory.saveConfigurationObject(sco);
    }
    
    protected static void createDSLSyntaxes() throws Exception
    {
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       
       DSLSyntaxFactoryManager dslSyntaxFactoryManager = svcManager.getDSLSyntaxFactoryManager();
       dslSyntaxFactoryManager.removeDSLSyntaxFactory(SERVICE_TYPE);
       
       DSLSyntaxFactory dslSyntaxFactory = dslSyntaxFactoryManager.createDSLSyntaxFactory(SERVICE_TYPE);

       {
          DSLStatementSyntax syntax = new DSLStatementSyntax("helloworld");
          syntax.setAction("hello world");
          syntax.setDescription("Sample DSL custom service operation. Should return a string \"Hello World!\".");

          dslSyntaxFactory.addDSLSyntax(syntax);
       }

       {
          DSLStatementSyntax syntax = new DSLStatementSyntax("getcurrenttime");
          syntax.setAction("get current time");
          syntax.setDescription("Sample DSL custom service operation. Should return current time.");

          dslSyntaxFactory.addDSLSyntax(syntax);
       }

       {
          DSLStatementSyntax syntax = new DSLStatementSyntax("print");
          syntax.setAction("print");
          syntax.addActionParameter(new StringParameter("text"));
          syntax.setDescription("Sample DSL custom service operation. Should print text from parameter <text>.");

          dslSyntaxFactory.addDSLSyntax(syntax);
       }

       {
          DSLStatementSyntax syntax = new DSLStatementSyntax("getauthors");
          syntax.setAction("get authors");
          syntax.setDescription("Sample DSL custom service operation. Should return table of authrs/books.");

          dslSyntaxFactory.addDSLSyntax(syntax);
       }

       {
          DSLStatementSyntax syntax = new DSLStatementSyntax("someoperation");
          syntax.setAction("some operation");
          syntax.setDescription("Sample DSL custom service operation. Should return error \"usupported operation\".");

          dslSyntaxFactory.addDSLSyntax(syntax);
       }
       
       dslSyntaxFactory.save();
    }
    
    protected static void  useCustomDSLOperations() throws Exception
    {
       FabricConnection connection = new FabricConnectionFactory().createConnection();
       connection.open();
       
       ServiceAccessor accessor = connection.createServiceAccessor(SERVICE_TYPE, SERVICE_NAME);

       {
          SLResponse response = accessor.invokeLanguageRequest("hello world");
          System.out.println("response.isOK(): " + response.isOK());
          System.out.println("response.getText(): " + response.getText());
       }

       {
          SLResponse response = accessor.invokeLanguageRequest("print '123'");
          System.out.println("response.isOK(): " + response.isOK());
          System.out.println("response.getText(): " + response.getText());
       }
    }

    protected static void  startService() throws ServiceManagerException
    {
       // register and start service 
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       if (!svcManager.isServiceRegistered(SERVICE_FULL_NAME))
       {
          ServiceDescriptor service = new ServiceDescriptor(SERVICE_TYPE, SERVICE_NAME);
          svcManager.registerService(service);
          svcManager.runServiceAs(SERVICE_FULL_NAME, "user1", "123");
       }
       
       svcManager.startService(SERVICE_FULL_NAME);
    }
    
    protected static void stopService() throws ServiceManagerException
    {
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       svcManager.stopService(SERVICE_FULL_NAME);
       svcManager.unregisterService(SERVICE_FULL_NAME);
    }
 }
