package rowset;

import com.streamscape.Trace;
import com.streamscape.cli.ds.DataspaceType;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.omf.serializer.TextSerializer;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.RuntimeContextException;
import com.streamscape.runtime.mf.admin.dfo.JDBCConfigurationFactory;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.event.RowSetEvent;
import com.streamscape.sdo.excp.SQLQueryException;
import com.streamscape.sdo.excp.SQLQueryParseException;
import com.streamscape.sdo.excp.SQLQueryValidationException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sdo.rowset.RowSet;
import com.streamscape.sdo.rowset.RowSetFactory;
import com.streamscape.sdo.rowset.RowSetPrinter;
import com.streamscape.sdo.sql.SQLQuery;
import com.streamscape.sdo.sql.SQLQueryFactory;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.dataspace.DataspaceManager;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.service.osf.jdbc.DatabaseConnection;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: RowSet samples. 
 * 
 * <p>Copyright: Copyright (c) 2012</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.4
 * @since 3.3
 */
 public class RowSetSamples
 {

    private static RuntimeContext context;
    private static Connection     connection;

    private static final String   EVENT_ROW_SET_ID = "event.rowset";
    
    public static void main(String[] args) throws Exception
    {
       // Enables some traces.
       Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
       Trace.enable("*",                         Trace.Level.ERROR);

       System.setProperty(RuntimeContext.STARTUP_DIR, "./");
       System.setProperty(RuntimeContext.DEPLOYMENT, "./");

       // Initializes the Runtime Context.
       context = RuntimeContext.getInstance();

       // create SQLQuery objects and serialize it 
       generateSQLQuery("create table TestTable (id int, varcharColumn varchar(100), doubleColumn double, clobColumn clob, blobColumn blob(100))", "CreateTestTable");
       generateSQLQuery("insert into TestTable values(1, 'stringvalue', 2.3, 'clobvalue', null)", "InsertIntoTestTable");
       generateSQLQuery("select * from TestTable", "TestSelect1");
       generateSQLQuery("update TestTable set varcharColumn = 'stringvalue1' where clobColumn = 'clobvalue'", "TestUpdate1");
       
       // create connection to database
       initializeDataspaceConnection();
       //initializeMysqlConnection();
       
       // execute queries and print results
       executeSQLQuery("CreateTestTable");
       executeSQLQuery("InsertIntoTestTable");
       executeSQLQuery("TestSelect1");
       executeSQLQuery("TestUpdate1");
       
       // shows how to use update query 
       updateQueries();
       
       // initialize row set and navigate thru it
       navigateRowSet();

       updateRowSet();

       // raise row set event
       raiseRowSetEvent();
       
       System.exit(0);
    }

    private static void generateSQLQuery(String sqlScript, String name) throws SQLQueryParseException, SQLQueryValidationException, RuntimeContextException, SQLQueryException
    {
       SQLQuery sqlQuery = SQLQueryFactory.createQuery(sqlScript);
       sqlQuery.setName(name);
       SQLQueryFactory.saveQueryObject("./", sqlQuery);
    }

    private static void initializeDataspaceConnection() throws Exception
    {
       DataspaceManager dsmgr = context.getDataspaceManager();
       try
       {
          dsmgr.dropDataspace("TEST_TSPACE", true, false);
       }
       catch (Exception ignored) {}

       dsmgr.createDataspace(DataspaceType.TSPACE, "TEST_TSPACE", EventScope.OBSERVABLE);

       connection = dsmgr.getJDBCConnection("User1", "123");
       Statement stat = connection.createStatement();
       stat.execute("set dataspace TEST_TSPACE");
    }

    private static void initializeMysqlConnection() throws Exception
    {
       DatabaseConnection databaseConnection = JDBCConfigurationFactory.createConnection("MySQLServer", "Prototype");
       databaseConnection.connect();
       Thread.sleep(1000);
       connection = databaseConnection;
       
       Statement stat = connection.createStatement();
       stat.execute("drop database if exists TEST_TSPACE");
       stat.execute("create database TEST_TSPACE");
       stat.execute("use TEST_TSPACE");
    }

    private static List<RowSet> executeSQLQuery(String queryName) throws Exception
    {
       SQLQuery sqlQuery = SQLQueryFactory.loadQueryObject("./", queryName); 
       List<RowSet> rowSets = SQLQueryFactory.execute(sqlQuery, 5, connection);
       
       System.out.println("Query '" + queryName + "': " + sqlQuery.getSqlScript());
       System.out.println("Results:");

       if (rowSets.isEmpty())
          System.out.println("\t empty result");

       for (RowSet rowSet : rowSets)
          new RowSetPrinter().print(rowSet);
       System.out.println();
       
       return rowSets;
    }
    
    private static void navigateRowSet() throws Exception
    {
       PreparedStatement preparedStatement = connection.prepareStatement("select * from TestTable");
       RowSet rowSet = new RowSetFactory().createRowSet(preparedStatement.getResultSet());
       rowSet.beforeFirst();
       
       while(rowSet.next())
       {
          System.out.println("Row " + rowSet.getRow());
          System.out.println("\tid \t\t= " + rowSet.getInt("id"));
          System.out.println("\tvarcharColumn \t= " + rowSet.getString("varcharColumn"));
          System.out.println("\tdoubleColumn \t= " + rowSet.getDouble("doubleColumn"));
          System.out.println("\tclobColumn \t= " + rowSet.getClob("clobColumn"));
          System.out.println("\tblobColumn \t= " + rowSet.getBlob("blobColumn"));
          System.out.println();
       }
       
       TextSerializer xSerializer = context.getXSerializer();
       String text = xSerializer.serialize(rowSet);
       System.out.println("Serialization OK: " + xSerializer.serialize(xSerializer.deserialize(text)).equals(text));
    }
    
    private static void updateRowSet() throws Exception
    {
       PreparedStatement preparedStatement = connection.prepareStatement("select * from TestTable");
       RowSet rowSet = new RowSetFactory().createRowSet(preparedStatement.getResultSet());
       rowSet.beforeFirst();
       rowSet.next();

       RowSetPrinter printer = new RowSetPrinter();
       printer.setPrintStream(System.out);

       rowSet.updateString("varcharColumn", "stringvalue1");

       System.out.println("\nRowSet after update:");
       printer.print(rowSet.clone());
       
       // cannot change read only row set
       rowSet.setReadOnly(true);
       rowSet.protect("password");
       try
       {
          rowSet.updateString("varcharColumn", "stringvalue");
       }
       catch(SQLException ignored) {}
       
       // cannot change protected RowSet
       try
       {
          rowSet.setReadOnly(false);
       }
       catch(SQLException ignored) {}
       
       // cannot unprotect with wrong password
       try
       {
          rowSet.unprotect("wrongpassword");
       }
       catch(SQLException ignored) {}

       rowSet.unprotect("password");
       rowSet.setReadOnly(false);
       rowSet.updateString("varcharColumn", "stringvalue");
       
       // add new row
       rowSet.addToRowSet(new Object [] {2, "str", 3.3, "str1", new byte[] {1,2,3}});
       System.out.println("\nRowSet after adding new row:");
       printer.print(rowSet.clone());
       
       // delete row
       rowSet.deleteFromRowSet(1);
       System.out.println("\nRowSet after first row deletion:");
       printer.print(rowSet.clone());
    }

    private static void updateQueries() throws Exception
    {
       // create update query for all columns
       SQLQuery query = SQLQueryFactory.createForUpdate(connection, "TestTable", new String[] {"varcharColumn", "doubleColumn"}, new String[] {"id"});
       query.setParameter("varcharColumn", "123");
       query.setParameter("doubleColumn", 2.3);
       query.setParameter("id", 1);
       query.setName("TestUpdateAllColumns");
       SQLQueryFactory.saveQueryObject("./", query);
       executeSQLQuery("TestUpdateAllColumns");
       executeSQLQuery("TestSelect1");
       
       
       // create update query for specified columns from dynamic sql
       query = SQLQueryFactory.createForDynamicSQL(connection, "update TestTable set varcharColumn = ? where doubleColumn = ?");
       query.setParameter("varcharColumn", "111");
       query.setParameter("doubleColumn", 2.3);
       query.setName("TestUpdateColumnsFromDynamicSQL");
       SQLQueryFactory.saveQueryObject("./", query);
       executeSQLQuery("TestUpdateColumnsFromDynamicSQL");
       executeSQLQuery("TestSelect1");
    }
    
    private static void raiseRowSetEvent() throws Exception
    {
       // create fabric connections
       FabricConnectionFactory connectionFactory = new FabricConnectionFactory();

       FabricConnection fabricConnection1 = connectionFactory.createConnection();
       fabricConnection1.open(); 

       FabricConnection fabricConnection2 = connectionFactory.createConnection();
       fabricConnection2.open();

       // create prototypes
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();

       if (!prototypeFactory.existsPrototype(EVENT_ROW_SET_ID))
          prototypeFactory.addEventPrototype("RowSetEvent", EVENT_ROW_SET_ID);

       // create consumer
       fabricConnection2.createEventAsyncConsumer("RowSetEventConsumer", new RowSetEventListener(), EVENT_ROW_SET_ID, null, EventScope.GLOBAL, true).start();

       // bind producer
       fabricConnection1.bindProducerFor(EVENT_ROW_SET_ID);
       
       RowSetEvent rowSetEvent = (RowSetEvent)context.getEventDatagramFactory().createEvent(EVENT_ROW_SET_ID);
       rowSetEvent.init(connection.createStatement().executeQuery("select * from TestTable"));

       fabricConnection1.raiseEvent(rowSetEvent, EventScope.INHERITED, -1);
    }

    static class RowSetEventListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("\nRowSetEventListener: received event: " + event.getEventId());
             RowSetPrinter printer = new RowSetPrinter();
             printer.setPrintStream(System.out);
             printer.print(((RowSetEvent)event).getRowSet());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }
 }
