package http;

import com.streamscape.Trace;
import com.streamscape.cli.http.*;
import com.streamscape.omf.json.JSONSerializerFactory;
import com.streamscape.omf.json.jackson.JSONSerializer;
import com.streamscape.omf.json.jackson.JsonNotation;
import com.streamscape.omf.mf.admin.ObjectMediationAliasManager;
import com.streamscape.sdo.EventDatagram;
import com.streamscape.sdo.IAbstractExceptionEvent;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sef.EventAsyncConsumer;
import com.streamscape.sef.EventReceiver;
import com.streamscape.sef.enums.EventScope;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: HTTP client sample.</p>
 * 
 * <p>Copyright: Copyright (c) 2011</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Batsyn
 * @version 3.2
 */
 public class HTTPClient
 {
    public static void main(String[] arguments)
    {
       if (arguments.length < 2)
       {
          System.out.println("Usage: client <event-id> <'raise' | 'receive' | 'subscribe'>\n");
          return;
       }

       // Enables traces.
       Trace.enable("http.*",                    Trace.Level.INFO);
       Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
       Trace.enable("com.streamscape.sef.*",     Trace.Level.INFO);

       String eventId = arguments[0];
       String action = arguments[1];

       HTTPFabricConnection connection = null;

       try
       {
          connection = new HTTPFabricConnectionFactory("http://localhost:80").createConnection("Admin", "Admin");
          connection.open();
          connection.bindProducerFor(eventId);

          // This serializer is needed only for printing events
          ObjectMediationAliasManager aliasManager = new HTTPClientAliasManager();
          HTTPClientUtils.addJavaAliases(aliasManager);
          HTTPClientUtils.addSysTypeAliases(aliasManager);
          HTTPClientUtils.realiasHttpAliases(aliasManager);

          JSONSerializerFactory factory = JSONSerializerFactory.getInstance();
          final JSONSerializer serializer = factory.createSerializerDefaultBuilderLazy("JsonSerializer")
                                                  .apply(builder ->
                                                         {
                                                            builder.setAliasManager(aliasManager);
                                                            builder.setJsonNotation(JsonNotation.TOP_ELEMENT);
                                                            builder.setPrettyPrint(true);
                                                         })
                                                  .getOrBuild();

          EventDatagram event = (EventDatagram)connection.getEventFactory().createEvent(eventId);
          if (action.equals("raise"))
          {
             connection.raiseEvent(event, EventScope.GLOBAL, 0);
             Trace.logInfo(HTTPClient.class, "Raised: " + serializer.serialize(event));
          }
          else if (action.equals("receive"))
          {
             EventReceiver receiver = connection.createEventReceiver("Receiver", eventId, null, EventScope.GLOBAL, false);

             Trace.logInfo(HTTPClient.class, "Receiving...");
             ImmutableEventDatagram received = receiver.receive();
             Trace.logInfo(HTTPClient.class, "Received: " + serializer.serialize(received));
          }
          else if (action.equals("subscribe"))
          {
             HTTPListener listener = new HTTPListener()
             {
                public void onEvent(ImmutableEventDatagram consumedEvent)
                {
                   try
                   {
                      Trace.logInfo(HTTPClient.class, "Consumed: " + serializer.serialize(consumedEvent));
                   }
                   catch (IAbstractExceptionEvent exception)
                   {
                      Trace.logException(HTTPClient.class, exception, true);
                   }
                }
             };

             EventAsyncConsumer consumer = connection.createEventAsyncConsumer("Consumer", listener, eventId, null, EventScope.GLOBAL, false);
             Trace.logInfo(HTTPClient.class, "Consuming...");
             consumer.start();
          }
          else
          {
             System.out.println("Usage: client <event-id> <'raise' | 'receive' | 'subscribe'>\n");
          }
       }
       catch (Throwable exception)
       {
          Trace.logException(HTTPClient.class, exception, true);
       }
       finally
       {
          try
          {
             if (!action.equals("subscribe"))
                connection.close();
          }
          catch (HTTPFabricException exception)
          {
             Trace.logException(HTTPClient.class, exception, true);
          }
       }
    }
 }
