package events;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.concurrent.FabricThreadManager;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.operation.security.ListUsersOperation;
import com.streamscape.sdo.EventDatagram;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.enums.AcknowledgeAction;
import com.streamscape.sdo.enums.ExceptionStrategy;
import com.streamscape.sdo.enums.ReplyMatchStrategy;
import com.streamscape.sdo.enums.RequestDistributionStrategy;
import com.streamscape.sdo.event.AcknowledgementEvent;
import com.streamscape.sdo.event.DataEvent;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sef.FabricRequestException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sdo.mf.admin.SemanticTypeFactory;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sdo.rowset.RowSetPrinter;
import com.streamscape.sef.*;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.moderator.*;
import com.streamscape.sef.utils.Utils;
import com.streamscape.slex.slang.SLSession;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: First <i>Runtime</i> sample.
 *
 * <p>This sample initializes the <code>Sample.Node1</code> node and makes some typical operations within this node.
 * Also this node can process events received from the <code>Sample.Node2</code> node and sample clients.
 * <br>The sample uses two embedded <code>FabricConnections</code>: one for consumers creation and other for event sending.
 *
 * <p>Copyright: Copyright (c) 2011</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.2
 */
 public class RuntimeSample1
 {
    static RuntimeContext   context;
    static FabricConnection connection1; // It is used for consumers and receivers.
    static FabricConnection connection2; // It is used for event sending.

    static final String     textEventId1       = "event.text1";
    static final String     textEventId2       = "event.text2";
    static final String     dataEventId        = "event.data";
    static final String     ackEventId         = "event.ack";
    static final String     directErrorEventId = "event.error.direct";
    static final String     asyncErrorEventId  = "event.error.async";
    static final String     receiverEventId    = "event.receiver";

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("events.*",                  Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node1");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node1");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("\nCreate connections...\n");
          createConnections();

          System.out.println("\nAdd prototypes...\n");
          addPrototypes();

          System.out.println("\nCreate consumers...\n");
          createConsumers();

          System.out.println("\nUse moderator...\n");
          useModerator();

          System.out.println("\nRaise events...\n");
          raiseEvents();

          System.out.println("\nRaise requests...\n");
          raiseRequests();

          System.out.println("\nInvoke SLANG requests...\n");
          invokeSlangRequests();

          System.out.println("\nUse receiver...\n");
          useReceiver();

          System.out.println();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void createConnections() throws Exception
    {
       FabricConnectionFactory connectionFactory = new FabricConnectionFactory();

       // This connection will work under the Runtime user (specified in the Deployment Descriptor).
       connection1 = connectionFactory.createConnection(); // The connection has  GLOBAL event scope by default (can act within the whole Fabric).
       connection1.setName("NodeConnection");
       connection1.open(); // Now the connection can perform various operations.

       // This connection will work under the user 'user1'.
       connection2 = connectionFactory.createConnection("user1", "123");
       connection2.open(); // The connection has an auto-generated name.
    }

    static void addPrototypes() throws Exception
    {
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();

       // TextEvent prototypes.
       if (!prototypeFactory.existsPrototype(textEventId1))
          prototypeFactory.addEventPrototype("TextEvent", textEventId1);
       if (!prototypeFactory.existsPrototype(textEventId2))
          prototypeFactory.addEventPrototype("TextEvent", textEventId2);

       // DataEvent prototype with three properties.
       if (!prototypeFactory.existsPrototype(dataEventId))
       {
          // Creates a semantic type for data class of the prototype.
          SemanticTypeFactory typeFactory = context.getSemanticTypeFactory();
          if (!typeFactory.existsSemanticType("SampleData"))
             context.getSemanticTypeFactory().addSemanticType(typeFactory.createSemanticType("SampleData", SampleData.class));

          DataEvent prototype = (DataEvent)context.getEventDatagramFactory().newEventInstance("DataEvent");
          prototype.setData(new SampleData());

          prototype.addAnnotation("stringProperty", "//data//stringField"); // Annotated property for {@link SampleData.stringField}.
          prototype.addAnnotation("intProperty",    "//data//intField");    // Annotated property for {@link SampleData.intField}.
          prototype.setEventDoubleProperty("doubleProperty", 0.0);

          prototypeFactory.addDataEventPrototype(dataEventId, prototype);
       }

       // AcknowledgementEvent prototype.
       if (!prototypeFactory.existsPrototype(ackEventId))
          prototypeFactory.addEventPrototype("AcknowledgementEvent", ackEventId);

       // TextEvent prototypes for error listeners.
       if (!prototypeFactory.existsPrototype(directErrorEventId))
          prototypeFactory.addEventPrototype("TextEvent", directErrorEventId);
       if (!prototypeFactory.existsPrototype(asyncErrorEventId))
          prototypeFactory.addEventPrototype("TextEvent", asyncErrorEventId);

       // TextEvent prototype for receiver.
       if (!prototypeFactory.existsPrototype(receiverEventId))
          prototypeFactory.addEventPrototype("TextEvent", receiverEventId);
    }

    static void createConsumers() throws Exception
    {
       // Creates the direct consumer for processing of FabricModeratorAdvisory events.
       // FabricModeratorAdvisory events are raised by the Exchange and always have OBSERVABLE event scope.
       // Direct consumers should be used very carefully (the best way is to use them for quick stable operations).
       connection1.createEventConsumer("ModeratorAdvisoryConsumer", new ModeratorAdvisoryListener(), FabricModeratorAdvisory.EVENT_ID,
                                       null, EventScope.OBSERVABLE, true);

       // Creates the direct consumer for processing of 'event.text' events.
       connection1.createEventConsumer("TextEventConsumer", new TextEventListener(), textEventId1, null, EventScope.OBSERVABLE, true);

       // Creates the first asynchronous consumer for processing of 'event.data' events.
       EventAsyncConsumer consumer = connection1.createEventAsyncConsumer("DataEventConsumer1", new DataEventListener1(), dataEventId,
                                                                          "stringProperty == 'sss' AND intProperty == 1", EventScope.GLOBAL, true);
       consumer.start(); // Do not forget to start async consumer!

       // Creates the second asynchronous consumer for processing of 'event.data' events.
       connection1.createEventAsyncConsumer("DataEventConsumer2", new DataEventListener2(), dataEventId,
                                            "doubleProperty == 1.1 AND intProperty != 1", EventScope.GLOBAL, true).start();

       // Creates the asynchronous consumer for processing of 'event.text2' events and raising of acknowledgements on these events.
       connection1.createEventAsyncConsumer("EventListenerWithAckConsumer", new EventListenerWithAck(), textEventId2, null,
                                            EventScope.GLOBAL, true).start();

       // Creates the direct consumer for processing of 'event.error.direct' events.
       connection1.createEventConsumer("DirectConsumerWithException", new EventListenerWithException(), directErrorEventId, null,
                                       EventScope.GLOBAL, true);

       // Creates the asynchronous consumer for processing of 'event.error.async' events.
       connection1.createEventAsyncConsumer("AsyncConsumerWithException", new EventListenerWithException(), asyncErrorEventId, null,
                                            EventScope.GLOBAL, true).start();

       // Creates the direct consumer for processing of ALL events raised in this sample.
       // Try not to abuse a use of of generic event filters because they match multiple events and can slow performance.
       connection1.createEventConsumer("AllEventsConsumer", new AllEventsListener(), "event.#", null, EventScope.GLOBAL, true);

       // Creates the request consumer for processing of any event directly raised to him.
       connection1.createRequestConsumer("RequestConsumer1", new RequestListener(), EventScope.OBSERVABLE);

       // Creates the request consumer for processing of any event directly raised to him.
       connection1.createRequestConsumer("RequestConsumer2", new RequestListener(), EventScope.GLOBAL);
    }

    static void useModerator()
    {
       Moderator moderator = context.getModerator();

       // In the Runtime each opened Fabric connection is represented by Fabric component.
       // Fabric components can be accessed through the Moderator using their read-only references:
       ComponentReference componentReference = moderator.lookupComponent(ModeratorUtils.makeComponentFullName(connection1));
       System.out.println("   Component: " + componentReference);

       // Component reference contains references for all consumers created by the corresponding Fabric connection:
       for (ConsumerReference consumerReference : componentReference.getConsumers(EventScope.INHERITED))
          System.out.println("      " + consumerReference);

       // Consumer references can also be accessed through the Moderator itself:
       System.out.println("   " + moderator.lookupRequestConsumer(ModeratorUtils.makeConsumerFullName(connection1, "RequestConsumer1")));
    }

    static void raiseEvents() throws Exception
    {
       // To have possibility of raising an event, connection should bind the corresponding event id to itself.
       // After a connection will bind any event id, this connection becomes event producer.
       // Pay attention that an event id must be bound to a connection only once.
       connection2.bindProducerFor(textEventId1);
       connection2.bindProducerFor(textEventId2);
       connection2.bindProducerFor(dataEventId);
       connection2.bindProducerFor(ackEventId);
       connection2.bindProducerFor(directErrorEventId);
       connection2.bindProducerFor(asyncErrorEventId);

       System.out.println("   Raising text event...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId1);
       textEvent.setText("event");
       // This event will get to TextEventListener and AllEventsListener.
       connection2.raiseEvent(textEvent, EventScope.INHERITED, -1);

       // Note that an event instance can be raised only once.

       System.out.println("   Raising data event 1...");

       DataEvent dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("sss", 1));
       // This event will get to DataEventListener1 (but not to DataEventListener2 due to selector) and AllEventsListener.
       connection2.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising data event 2...");

       dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("sss", 2));
       dataEvent.setEventDoubleProperty("doubleProperty", 1.1);
       // This event will get to DataEventListener2 (but not to DataEventListener1 due to selector) and AllEventsListener.
       connection2.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising data event 3...");

       dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("zzz", 1));
       dataEvent.setEventDoubleProperty("doubleProperty", 1.1);
       // This event will get to AllEventsListener only.
       connection2.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising direct error event 1...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(directErrorEventId);
       textEvent.setText("error");

       try
       {
          // This event will get to EventListenerWithException and perhaps AllEventsListener (it is order-dependent).
          connection2.raiseEvent(textEvent, EventScope.INHERITED, -1);
       }
       catch (FabricEventException exception)
       {
          Trace.logException(RuntimeSample2.class, exception, false);
       }

       // All FabricEventExceptions thrown by direct consumers will be ignored.
       connection2.setExceptionStrategy(ExceptionStrategy.IGNORE);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising direct error event 2...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(directErrorEventId);
       textEvent.setText("error");
       // This event will get to EventListenerWithException and AllEventsListener. Exception will be ignored.
       connection2.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising async error event...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(asyncErrorEventId);
       textEvent.setText("error");
       // This event will get to EventListenerWithException and AllEventsListener. Exception will be ignored.
       connection2.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
    }

    static void raiseRequests() throws Exception
    {
       System.out.println("   Raising first non-direct request...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request1");
       textEvent.setReplyTo(ackEventId); // Acknowledgement event will have the specified event id.

       // This event will get to EventListenerWithAck (which will raise acknowledgement) and AllEventsListener.
       // The current thread will wait for acknowledgement for 1 second.
       AcknowledgementEvent ack = connection2.raiseRequest(textEvent, EventScope.INHERITED, RequestDistributionStrategy.AUCTION,
                                                           ReplyMatchStrategy.REPLY_TO, 1000);
       System.out.println("      Acknowledgement: " + ack.getEventId());

       System.out.println("   Raising second non-direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request2");

       // Since 'ReplyTo' is not set, an acknowledgement event will have an auto-generated event id.
       ack = connection2.raiseRequest(textEvent, EventScope.INHERITED, RequestDistributionStrategy.AUCTION, ReplyMatchStrategy.REPLY_TO, 2000);
       System.out.println("      Acknowledgement: " + ack.getEventId());

       Moderator moderator = connection2.getModerator();

       System.out.println("   Raising first direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId1);
       textEvent.setText("request3");

       // This event will get directly to RequestConsumer1.
       // The current thread will wait for reply for 1 seconds.
       ImmutableEventDatagram reply = connection2.raiseRequest(moderator.lookupRequestConsumer("Client_TLP.NodeConnection:RequestConsumer1"),
                                                               textEvent, 1000);
       System.out.println("      Reply: " + reply.getEventId());

       System.out.println("   Raising second direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request4");

       // This event will get directly to RequestConsumer2.
       // The current thread will wait for reply for infinite time.
       // Use of an infinite waiting time can significantly increase performance within the local node,
       // but use it very carefully because this way can block the current thread forever.
       reply = connection2.raiseRequest(moderator.lookupRequestConsumer("Client_TLP.NodeConnection:RequestConsumer2"), textEvent, -1);
       System.out.println("      Reply: " + reply.getEventId());
    }

    static void invokeSlangRequests() throws Exception
    {
       // Creates Semantic Language session to the current node.
       SLSession session = connection1.createSLSession();

       System.out.println("\n   First SLANG request...\n");

       // Invokes a request using a plain text command.
       SLResponse response = session.slangRequest("list users", 1000);
       // Result data is contained in the RowSet.
       new RowSetPrinter().print(response.getRowSet());

       System.out.println("\n   Second SLANG request...\n");

       // Invokes a request using a special SLStatement object.
       response = session.slangRequest(new ListUsersOperation.Definition(), 1000);
       new RowSetPrinter().print(response.getRowSet());
    }

    static void useReceiver() throws Exception
    {
       // Creates the receiver consumer for a obtaining of 'event.receiver' events.
       final EventReceiver receiver = connection1.createEventReceiver("Receiver", receiverEventId, null, EventScope.GLOBAL, true);

       connection2.bindProducerFor(receiverEventId);

       // Waits an event for 1 second. No event will be obtained.
       ImmutableEventDatagram event = receiver.receive(1000);
       if (event == null)
          System.out.println("\n   No event received.\n");

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising receiver event 1...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(receiverEventId);
       textEvent.setText("event1");
       // This event will get to AllEventsListener and receiver.
       connection2.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising receiver event 2...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(receiverEventId);
       textEvent.setText("event2");
       // This event will get to AllEventsListener and receiver.
       connection2.raiseEvent(textEvent, EventScope.INHERITED, -1);

       // Waits an event for 1 second. First event will be obtained.
       event = receiver.receive(1000);
       System.out.println("\n   Received " + event.getEventId() + " : " + ((TextEvent)event).getText());

       // Waits indefinitely until event will be available. Second event will be obtained.
       // Use such method very carefully because it can block the current thread forever.
       event = receiver.receive();
       System.out.println("   Received " + event.getEventId() + " : " + ((TextEvent)event).getText());

       // Tries to obtain an event without a waiting. No event will be obtained.
       event = receiver.receiveNoWait();
       if (event == null)
          System.out.println("\n   No event received.\n");

       // Creates and starts thread for a obtaining of 'event.receiver' events from other node and clients.
       FabricThreadManager.getInstance().createThread("ReceiverThread", "Receives remote events.",
                                                      new Runnable()
                                                      {
                                                         @Override
                                                         public void run()
                                                         {
                                                            while (!Thread.interrupted())
                                                            {
                                                               try
                                                               {
                                                                  // Waits indefinitely until next event will be available.
                                                                  TextEvent event = (TextEvent)receiver.receive();
                                                                  System.out.println("      Receiver: " + event.getEventId() + " : " + event.getText());
                                                               }
                                                               catch (Exception exception)
                                                               {
                                                                  exception.printStackTrace();
                                                               }
                                                            }
                                                         }
                                                      }).start();
    }

    // Event listener for FabricModeratorAdvisory events.
    static class ModeratorAdvisoryListener  implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          FabricModeratorAdvisory advisory = (FabricModeratorAdvisory)event;
          if (advisory.getType() == ModeratorAdvisoryType.NODE_CONNECTED)
             System.out.println();
          System.out.println("      FabricModeratorAdvisoryListener: " + advisory.getType() + ", " + advisory.getEntity());
          if (advisory.getType() == ModeratorAdvisoryType.NODE_CONNECTED)
             System.out.println();
       }
    }

    // Event listener for 'event.text' events.
    static class TextEventListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      TextEventListener: " + event.getEventId() + " : " + ((TextEvent)event).getText());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // First event listener for 'event.data' events.
    static class DataEventListener1 implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      DataEventListener1: " + event.getEventId() + " : " + ((DataEvent)event).getData());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // Second event listener for 'event.data' events.
    static class DataEventListener2 implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      DataEventListener2: " + event.getEventId() + " : " + ((DataEvent)event).getData());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // Event listener for 'event.text' events with acknowledgement.
    static class EventListenerWithAck implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      EventListenerWithAck: " + event.getEventId() + " : " + ((TextEvent)event).getText());
             connection2.acknowledgeEvent((EventDatagram)event, true, AcknowledgeAction.ACKNOWLEDGE, null, EventScope.INHERITED, 0);
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // Event listener for 'event.error.*' events. It is used in direct and async consumers.
    static class EventListenerWithException implements FabricEventListener
    {
       @Override
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             System.out.println("      EventListenerWithException: " + event.getEventId() + " : " + ((TextEvent)event).getText());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }

          throw new FabricEventException("EventListenerWithException.");
       }
    }

    // Event listener for all events used in the sample.
    static class AllEventsListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          System.out.println("      AllEventsListener: " + event.getEventId());
       }
    }

    // Request listener for any event.
    static class RequestListener implements FabricRequestListener
    {
       public ImmutableEventDatagram onRequest(ImmutableEventDatagram event) throws FabricRequestException
       {
          System.out.println("      RequestListener: " + event.getEventId());
          return event;
       }
    }
 }
