package streaming;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.omf.xml.XSerializer;
import com.streamscape.omf.xml.XSerializerFactory;
import com.streamscape.repository.types.SemanticType;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.event.ChunkEvent;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.StreamEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.network.tlp.acceptor.TLPAcceptor;
import com.streamscape.sef.utils.Utils;

import java.io.File;
import java.io.FileOutputStream;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Second <i>Streaming</i> sample.
 *
 * <p>This sample initializes the <code>Sample.Node2</code> node and starts second data stream to the <code>Sample.Node1</code> node.
 *
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class Stream2Sample
 {
    static RuntimeContext   context;
    static FabricConnection connection1; // It is used for consumers.
    static FabricConnection connection2; // It is used for event sending.

    static final String     streamEventId2 = "event.stream2";
    static final String     chunkEventId2  = ChunkEvent.getEventId(streamEventId2);

    public static void main(String[] args)
    {
       try
       {
          // This auxiliary method is required only for automatic creation of TLP acceptor with non-default port (to avoid a manual change).
          prepareRepository();

          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("streaming.*",               Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node2");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node2");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("Create consumers...\n");
          createConsumers();

          System.out.println("Raise stream event...\n\n");
          raiseStreamEvent();

          System.out.println();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void createConsumers() throws Exception
    {
       connection1 = new FabricConnectionFactory().createConnection();
       connection1.open();

       // Creates the asynchronous consumer for processing of StreamEvent for a second stream.
       connection1.createEventAsyncConsumer("Stream2Consumer", new Stream2Listener(), streamEventId2, null, EventScope.GLOBAL, true).start();

       // Creates the asynchronous consumer for processing of ChunkEvent for a second stream.
       connection1.createEventAsyncConsumer("Chunk2Consumer", new Chunk2Listener(), chunkEventId2, null, EventScope.GLOBAL, true).start();
    }

    static class Stream2Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             Utils.sleep(500);
             System.out.println("Confirmation: " + new String(((StreamEvent)event).getBytes()) + "\n\n");
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class Chunk2Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             System.out.print(new String(((ChunkEvent)event).getBytes()));
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static void raiseStreamEvent() throws Exception
    {
       connection2 = new FabricConnectionFactory().createConnection();
       connection2.open();

       connection2.bindProducerFor(streamEventId2);
       // Note that user does not need to directly bind producer for event id of corresponding ChunkEvent.

       StreamEvent event = (StreamEvent)EventDatagramFactory.getInstance().createEvent(streamEventId2);
       event.writeBytes("This is the data of a second stream.".getBytes());
       event.setChunkSize(3);    // 3 bytes
       event.setChunkDelay(500); // 500 milliseconds

       // User should see that the text of event is displayed on the screen step-by-step (for both nodes).
       // Then a confirmation of this text will be displayed.
       connection2.raiseEvent(event, EventScope.INHERITED, -1);
    }

    static void prepareRepository() throws Exception
    {
       File nodeDir = new File("Sample.Node2");
       if (!(new File(nodeDir, ".tfcache").exists()))
       {
          XSerializerFactory.getInstance().alias(new SemanticType("TLPAcceptor", TLPAcceptor.class.getName()));
          XSerializer serializer = XSerializerFactory.getInstance().getDefaultSerializer();

          Utils.createTFCache(nodeDir);
          createAcceptor(nodeDir, serializer);

          XSerializerFactory.getInstance().destroy();
       }
    }

    static void createAcceptor(File nodeDir, XSerializer serializer) throws Exception
    {
       TLPAcceptor acceptor = new TLPAcceptor("Default");
       acceptor.setURL("tlp://127.0.0.1:5001");

       byte[] bytes = serializer.serialize(acceptor).getBytes();
       FileOutputStream stream = new FileOutputStream(new File(Utils.createDirectory(new File(nodeDir, ".tfcache/objects/sys/network/acceptors/tlp")),
                                                               "TLPAcceptor.Default.xdo"));
       stream.write(bytes);
       stream.close();
    }
 }
