package service.mail;

import java.io.File;
import java.io.FileInputStream;
import java.util.Arrays;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationFactory;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationObject;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.MailEvent;
import com.streamscape.sdo.utils.SDOUtils;
import com.streamscape.sef.ConfigurationProperty;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.service.ServiceDescriptor;
import com.streamscape.sef.service.ServiceManager;
import com.streamscape.sef.service.ServiceManagerException;
import com.streamscape.service.mail.sender.MailSender;
import com.streamscape.service.osf.config.PasswordPropertyValue;
import com.streamscape.service.osf.config.ServiceConfigurationProperty;
import com.streamscape.service.osf.config.ServicePropertyType;
import com.streamscape.service.osf.eim.EventIdentityProcessStart;
import com.streamscape.service.osf.enums.InvokeMode;
import com.streamscape.service.osf.evh.EventHandler;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: MailSender service sample.
 *
 * <p>Copyright: Copyright (c) 2012</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.3
 */
 public class MailSenderServiceSample
 {
    private static final Class<?> SERVICE_CLASS     = MailSender.class;
    private static final String   SERVICE_TYPE      = SERVICE_CLASS.getSimpleName();
    private static final String   SERVICE_NAME      = SERVICE_TYPE + "Name";
    private static final String   SERVICE_FULL_NAME = SERVICE_TYPE + "." + SERVICE_NAME;

    private static final String   MAIL_EVENT_ID         = "event.mail";

    private static final String   SMTP_SERVER_NAME      = "smtp.gmail.com";
    private static final String   SMTP_SERVER_PORT      = "587";
    private static final String   SMTP_SERVER_USER      = "neeve.team@gmail.com";
    private static final String   SMTP_SERVER_PASSWORD  = "";
    private static final String   START_TLS_REQUIRED    = "true";
    
    private static final String[] TO_ADDRESSES          = {"neeve.team@gmail.com"};
    private static final String[] CC_ADDRESSES          = {"neeve.team@gmail.com"};
    
    public static void main(String[] args) throws Exception
    {
       // Enables some traces.
       Trace.enable("com.streamscape.sef.service.*", Trace.Level.INFO);
       Trace.enable("com.streamscape.runtime.*",     Trace.Level.INFO);
       Trace.enable("*",                             Trace.Level.ERROR);

       System.setProperty(RuntimeContext.STARTUP_DIR, ".");
       System.setProperty(RuntimeContext.DEPLOYMENT,  ".");
       
       try
       {
          // Initializes the Runtime Context.
          RuntimeContext.getInstance();

          // create event prototypes
          createEventPrototypes();

          // create service sco, register and start service
          setUpMailSenderService();

          // send mail events
          sendMailEvents();
          
          // stop service
          stopService();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
       }
       System.exit(0);
    }
    
    protected static void createEventPrototypes() throws Exception
    {
       SDOUtils.addEventPrototype("MailEvent", MAIL_EVENT_ID);
    }

    protected static void setUpMailSenderService() throws Exception
    {
       // create service sco
       ServiceConfigurationObject sco = ServiceConfigurationFactory.createServiceConfiguration
             (
                RuntimeContext.getInstance(), 
                SERVICE_NAME,
                SERVICE_TYPE,
                false
             );
             
       sco.setServiceClassName(SERVICE_CLASS.getName());
       sco.setEIMPluginName(EventIdentityProcessStart.class.getSimpleName());
       sco.setInterruptableService(false);
       sco.setInvokeMode(InvokeMode.ASYNC);

       ConfigurationProperty property = sco.createAdvancedProperty(EventIdentityProcessStart.EVENT_GROUP_IDENTITY,
             EventIdentityProcessStart.EVENT_GROUP_IDENTITY, 
             EventIdentityProcessStart.EVENT_GROUP_IDENTITY);
       property.setTextValue("MailSender Service sample");
       sco.addAdvancedProperty(property);

       EventHandler handler = new EventHandler(sco, "sendMailFromEvent", "SendMailHandler");
       ImmutableEventDatagram prototype = EventDatagramFactory.getInstance().createEvent(MAIL_EVENT_ID);
       handler.bindRequestEvent(prototype);
       sco.addEventHandler(handler);

       ServiceConfigurationProperty hostProp = sco.createProperty(com.streamscape.service.mail.sender.SCOProperties.SMTP_SERVER_NAME,
             ServicePropertyType.STRING, null);
       hostProp.setValue(SMTP_SERVER_NAME);
       sco.addProperty(hostProp);

       ServiceConfigurationProperty portProp = sco.createProperty(com.streamscape.service.mail.sender.SCOProperties.SMTP_SERVER_PORT,
             ServicePropertyType.STRING, null);
       portProp.setValue(SMTP_SERVER_PORT);
       sco.addProperty(portProp);

       ServiceConfigurationProperty userProp = sco.createProperty(com.streamscape.service.mail.sender.SCOProperties.SMTP_SERVER_USER,
             ServicePropertyType.STRING, null);
       userProp.setValue(SMTP_SERVER_USER);
       sco.addProperty(userProp);

       ServiceConfigurationProperty startTLS = sco.createProperty(
             com.streamscape.service.mail.sender.SCOProperties.START_TLS_REQUIRED, ServicePropertyType.BOOLEAN, null);
       startTLS.setValue(START_TLS_REQUIRED);
       sco.addProperty(startTLS);

       ServiceConfigurationProperty passwdProp = sco.createProperty(
             com.streamscape.service.mail.sender.SCOProperties.SMTP_SERVER_PASSWD, ServicePropertyType.PASSWORD, null);
       passwdProp.setValue(new PasswordPropertyValue(SMTP_SERVER_PASSWORD));
       sco.addProperty(passwdProp);

       ServiceConfigurationProperty replyTo = sco.createProperty(com.streamscape.service.mail.sender.SCOProperties.REPLY_TO_ADDRESS,
             ServicePropertyType.STRING, null);
       replyTo.setValue(SMTP_SERVER_USER);
       sco.addProperty(replyTo);

       ServiceConfigurationProperty subject = sco.createProperty(com.streamscape.service.mail.sender.SCOProperties.MESSAGE_SUBJECT,
             ServicePropertyType.STRING, null);
       subject.setValue("default message subject");
       sco.addProperty(subject);

       ServiceConfigurationProperty recipients = sco.createProperty(com.streamscape.service.mail.sender.SCOProperties.RECIPIENTS_LIST,
             ServicePropertyType.STRING, null);
       recipients.setValue("");
       sco.addProperty(recipients);

       ServiceConfigurationFactory.saveConfigurationObject(sco);
       
       // register and start service 
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       if (!svcManager.isServiceRegistered(SERVICE_FULL_NAME))
       {
          ServiceDescriptor service = new ServiceDescriptor(SERVICE_TYPE, SERVICE_NAME);
          svcManager.registerService(service);
          svcManager.runServiceAs(SERVICE_FULL_NAME, "user1", "123");
       }
       
       svcManager.startService(SERVICE_FULL_NAME);
    }
    
    protected static void stopService() throws ServiceManagerException
    {
       ServiceManager svcManager = RuntimeContext.getInstance().getServiceManager();
       svcManager.stopService(SERVICE_FULL_NAME);
       svcManager.unregisterService(SERVICE_FULL_NAME);
    }

    protected static void sendMailEvents() throws Exception
    {
       FabricConnection connection = new FabricConnectionFactory().createConnection("user1", "123");
       connection.open();
       connection.bindProducerFor(MAIL_EVENT_ID);

       System.out.println("sending emails...");
       
       // mail event with html content
       {
          MailEvent mailEvent = (MailEvent)EventDatagramFactory.getInstance().createEvent(MAIL_EVENT_ID);
          mailEvent.setSubject("html email sent by MailSender service.");
          mailEvent.setBody("<b><i>html content</i></b><br><img src=\"cid:imagecid\">");
          mailEvent.setContentType("text/html");
          mailEvent.setTo(Arrays.asList(TO_ADDRESSES));
          mailEvent.setCc(Arrays.asList(CC_ADDRESSES));

          {
             String attachmentName = "streamscape.jpg";
             File attachmentFile = new File("../../streamscape.jpg");
             byte[] attachmentData = new byte[(int) attachmentFile.length()];
             new FileInputStream(attachmentFile).read(attachmentData);
             MailEvent.MailAttachment attachment = new MailEvent.MailAttachment(attachmentName, attachmentData, null, "<imagecid>");  
             mailEvent.addAttachment(attachment);
          }
          
          {
             String attachmentName = "sampleattachmenttextfile.txt";
             byte[] attachmentData = "text data".getBytes();
             MailEvent.MailAttachment attachment = new MailEvent.MailAttachment(attachmentName, attachmentData, "text/plain");  
             mailEvent.addAttachment(attachment);
          }

          connection.raiseEvent(mailEvent, EventScope.INHERITED, -1);
       }
    }
 }
