package jdbc;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import com.streamscape.sdo.rowset.RowSet;
import com.streamscape.sdo.rowset.RowSetFactory;
import com.streamscape.sdo.rowset.RowSetPrinter;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: JDBC Driver FLOB sample.
 * 
 * <p>This sample shows how to operate with FLOB columns using JDBC driver.
 * 
 * <p>Copyright: Copyright (c) 2016</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.5
 */
 public class JDBCDriverFlobSample
 {
    public static void main(String[] args) throws Exception
    {
       Connection jdbcConnection = null;
       try
       {
          System.out.println("Assume you have node srarted and available on tlp://localhost:5501 and dataspace TSPACE.FlobSpace created.");
          System.out.println("We will connect to dataspace TSPACE.FlobSpace create table with FLOB columns and execute some queries on this table.");

          System.out.println("Please make sure that node started. Press enter.");
          System.in.read();
          System.out.println("If dataspace is not created, please create it with command: 'create dataspace FlobSpace type TSPACE'");
          System.out.println("Press enter.");
          System.in.read();

          String url = "jdbc:streamscape:dataspace://localhost:11608/TSPACE.FlobSpace";
          System.out.println("Connecting to TSPACE.FlobSpace with url");
          System.out.println("   " + url);

          Class.forName("com.streamscape.ds.jdbc.JDBCDriver");
          jdbcConnection = DriverManager.getConnection(url, "admin", "admin");

          System.out.println("  OK\n");

          System.out.println("Creating table...");
          executeQuery(jdbcConnection, "drop table FlobFilesStorage if exists");
          executeQuery(jdbcConnection, "create table FlobFilesStorage (id int, flobFile flob, flobFileAutotaged flob(autotag), primary key(id))");
          System.out.println("  OK\n");
          
          System.out.println("Creating test files...");
          createFile("flobsamplefile1.txt", "FLOB sample text file 1.", "UTF-8");
          createFile("flobsamplefile2.txt", "FLOB sample text file 2.", "UTF-8" );
          createFile("flobsamplefile3utf16.txt", "FLOB sample text file 3, UTF-16.", "UTF-16");
          System.out.println("  OK\n");
          
          System.out.println("Inserting test FLOB files...");
          executeQuery(jdbcConnection, "insert into FlobFilesStorage values(1, readFile('flobsamplefile1.txt'), readFile('flobsamplefile1.txt'))");
          executeQuery(jdbcConnection, "insert into FlobFilesStorage values(2, readFile('flobsamplefile2.txt'), readFile('flobsamplefile2.txt'))");
          executeQuery(jdbcConnection, "insert into FlobFilesStorage values(3, readFile('flobsamplefile3utf16.txt', 0, -1, 'UTF-16'), readFile('flobsamplefile3utf16.txt', 0, -1, 'UTF-16'))");
          System.out.println("  OK\n");
          
          System.out.println("Selecting all...");
          Statement statement = jdbcConnection.createStatement();
          statement.execute("select * from FlobFilesStorage");
          ResultSet resultSet = statement.getResultSet();
          
          System.out.println("Read columns into BLOB...");
          {
             resultSet.next();
             int id = resultSet.getInt("id");
             Blob flobFile = resultSet.getBlob("flobFile");
             Blob flobFileAutotaged = resultSet.getBlob("flobFileAutotaged");
             
             System.out.printf("id: %d, flobFile: %s, flobFileAutotaged: %s\n", 
                   id, new String(flobFile.getBytes(1, (int) flobFile.length())), new String(flobFileAutotaged.getBytes(1, (int) flobFileAutotaged.length())));
          }

          System.out.println("Read columns into CLOB...");
          {
             resultSet.next();
             int id = resultSet.getInt("id");
             Clob flobFile = resultSet.getClob("flobFile");
             Clob flobFileAutotaged = resultSet.getClob("flobFileAutotaged");
             
             System.out.printf("id: %d, flobFile: %s, flobFileAutotaged: %s\n", 
                   id, flobFile.getSubString(1, (int)flobFile.length()), flobFileAutotaged.getSubString(1, (int)flobFileAutotaged.length()));
          }

          System.out.println("Read columns into string...");
          {
             resultSet.next();
             int id = resultSet.getInt("id");
             String flobFile = resultSet.getString("flobFile");
             String flobFileAutotaged = resultSet.getString("flobFileAutotaged");
             
             System.out.printf("id: %d, flobFile: %s, flobFileAutotaged: %s\n", id, flobFile, flobFileAutotaged);
          }
          resultSet.close();
          statement.close();
          System.out.println("  OK\n");

          System.out.println("Removing test files...");
          new File("flobsamplefile1.txt").delete();
          new File("flobsamplefile2.txt").delete();
          new File("flobsamplefile3utf16.txt").delete();
          System.out.println("  OK\n");
          
          jdbcConnection.close();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
       }
       finally
       {
          if(jdbcConnection != null)
             jdbcConnection.close();
       }
       System.out.println("Exiting...");
       System.out.println("  OK\n");
       System.exit(0);
    }

    private static void executeQuery(Connection jdbcConnection, String query) throws SQLException
    {
       System.out.print("Executing query: " + query);
       Statement statement = null;
       try
       {
          statement = jdbcConnection.createStatement();
          if (statement.execute(query))
          {
             System.out.println();
             ResultSet resultSet = statement.getResultSet();
             RowSet rowSet = new RowSetFactory().createRowSet(resultSet);
             new RowSetPrinter().print(rowSet);
          }
          else
             System.out.println("  OK");
       } 
       catch (SQLException exception)
       {
          System.out.println("\n Error: " + exception.getMessage());

       }
       finally
       {
          if (statement != null)
             statement.close();
       }
    }
    
    public static void createFile(String filename, String content, String charset) throws Exception
    {
       Path parent = Paths.get(filename).getParent();
       if (parent != null)
          Files.createDirectories(parent);
       
       new File(filename).delete();
       try (Writer writer = charset != null ? new OutputStreamWriter(new FileOutputStream(filename), charset) : new OutputStreamWriter(new FileOutputStream(filename)))
       {
          writer.write(content);
       }
    }

 }
