package group;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.timer.AbstractFabricTimerTask;
import com.streamscape.lib.timer.FabricTimer;
import com.streamscape.lib.timer.FabricTimerManager;
import com.streamscape.omf.xml.XSerializer;
import com.streamscape.omf.xml.XSerializerFactory;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.SecurityViolationException;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.group.FabricGroup;
import com.streamscape.sef.group.FabricGroupLink;
import com.streamscape.sef.moderator.RequestConsumerReference;
import com.streamscape.sef.network.tlp.acceptor.TLPAcceptor;
import com.streamscape.sef.utils.Utils;

import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Second <i>Group</i> sample.
 *
 * <p>This sample initializes the <code>Sample.Node2</code> node and raises several events within the specific group.
 * At the same time the another <code>Sample.Node1</code> node also raises several events within the same group.
 * Nodes raise events simultaneously with a random delay between successive events (it is needed to shuffle these events).
 * Also each node has the async consumer that receives events from both nodes within the group.
 * Order of events received by each consumer must be identical.
 *
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class GroupSample2
 {
    static RuntimeContext   context;
    static FabricGroup      group;
    static FabricConnection connection1; // It is used for consumers.
    static FabricGroupLink  link1;       // It is used for consumers.
    static FabricConnection connection2; // It is used for event sending.
    static FabricGroupLink  link2;       // It is used for event sending.

    static TestListener     listener;
    static final String     eventId = "event.text";

    public static void main(String[] args)
    {
       try
       {
          // This auxiliary method is required only for automatic creation of TLP acceptor with non-default port (to avoid a manual change).
          prepareRepository();

          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("events.*",                  Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node2");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node2");
          // Sets a path to the to file containing discovery rules for the Fabric.
          System.setProperty(RuntimeContext.DISCOVERY_FABRIC_DIRECTORY, System.getProperty(RuntimeContext.STARTUP_DIR) + "/../DirectoryTable.xdo");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("\nLookup group...\n");
          lookupGroup();

          System.out.println("\nCreate connections...\n");
          createConnections();

          System.out.println("\nCreate consumers...");
          createConsumers();

          // Raises a request to synchronize events sending in the both nodes.
          raiseSynchronizationRequest();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void lookupGroup() throws Exception
    {
       group = context.getGroupManager().lookupGroup("TestGroup");
       System.out.println("   Group 'TestGroup found.'");
       System.out.println("   Group has members: " + group.listMembers());
    }


    static void createConnections() throws Exception
    {
       FabricConnectionFactory connectionFactory = new FabricConnectionFactory();

       connection1 = connectionFactory.createConnection();
       connection1.setName("Sample2Connection1");
       connection1.open();

       link1 = connection1.joinGroup("TestGroup");
       System.out.println("   Group has members: " + group.listMembers());

       connection2 = connectionFactory.createConnection();
       connection2.setName("Sample2Connection2");
       connection2.open();

       link2 = connection2.joinGroup("TestGroup");
       System.out.println("   Group has members: " + group.listMembers());
    }

    static void createConsumers() throws Exception
    {
       // Creates the async consumer in the TestGroup group.
       listener = new TestListener();
       link1.createEventAsyncConsumer("Consumer", listener, eventId, "", EventScope.GLOBAL, true).start();
    }

    static void raiseEvents()
    {
       System.out.println("\nRaise events...\n");

       try
       {
          connection2.bindProducerFor(eventId);

          raiseEvent("Node2.Event1");
          raiseEvent("Node2.Event2");
          raiseEvent("Node2.Event3");
          raiseEvent("Node2.Event4");
          raiseEvent("Node2.Event5");
          raiseEvent("Node2.Event6");
          raiseEvent("Node2.Event7");
          raiseEvent("Node2.Event8");
          raiseEvent("Node2.Event9");
          raiseEvent("Node2.Event10");
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
       }
    }

    static void raiseEvent(String text) throws Exception
    {
       Utils.sleep(new Random(System.currentTimeMillis()).nextInt(20) + 1);

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(eventId);
       textEvent.setText(text);

       // FabricGroupLink has the same interface for events raising as the FabricConnection.
       link2.raiseEvent(textEvent, EventScope.GLOBAL, -1);
       System.out.println("   Event [" + text + "] raised.");
    }

    static void printResults()
    {
       // An order of the received events must be identical for all consumers of the group in all nodes.
       System.out.println();
       for (String text : listener.texts)
          System.out.println("   Event [" + text + "] received.");
    }

    static class TestListener implements FabricEventListener
    {
       private final List<String> texts = new ArrayList<String>();

       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             texts.add(((TextEvent)event).getText());
          }
          catch (SecurityViolationException exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class RaiseEventsTask extends AbstractFabricTimerTask
    {
       public void execute(FabricTimer timer)
       {
          raiseEvents();
          Utils.sleep(2000);
          printResults();
       }
    }

    private static final String syncEventId = "event.sync";
    static void raiseSynchronizationRequest() throws Exception
    {
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();
       if (!prototypeFactory.existsPrototype(syncEventId))
          prototypeFactory.addEventPrototype("TextEvent", syncEventId);

       connection2.bindProducerFor(syncEventId);
       TextEvent request = (TextEvent)context.getEventDatagramFactory().createEvent(syncEventId);
       RequestConsumerReference consumer = context.getModerator().lookupRequestConsumer("Sample.Node1://Client_TLP.Sample1Connection1:Synchronizer");
       connection2.raiseRequest(consumer, request, 1000);

       Date startTime = new Date(request.getTimestamp().getTime() + 1000); // Timer will start in 1 second.
       FabricTimerManager.getInstance().createTimer("Default", "SyncTimer", new RaiseEventsTask(), 1, 1).start(startTime);
    }

    static void prepareRepository() throws Exception
    {
       File nodeDir = new File("Sample.Node2");
       if (!(new File(nodeDir, ".tfcache").exists()))
       {
          XSerializer serializer = XSerializerFactory.getInstance().getDefaultSerializer();

          Utils.createTFCache(nodeDir);
          createAcceptor(nodeDir, serializer);

          XSerializerFactory.getInstance().destroy();
       }
    }

    static void createAcceptor(File nodeDir, XSerializer serializer) throws Exception
    {
       TLPAcceptor acceptor = new TLPAcceptor("Default");
       acceptor.setURL("tlp://127.0.0.1:5001");

       byte[] bytes = serializer.serialize(acceptor).getBytes();
       FileOutputStream stream = new FileOutputStream(new File(Utils.createDirectory(new File(nodeDir, ".tfcache/objects/sys/network/acceptors/tlp")),
                                                               "TLPAcceptor.Default.xdo"));
       stream.write(bytes);
       stream.close();
    }
 }
