from __future__ import absolute_import

import unittest

from stpy.fabric.row_set import RowSet, RowSetProxy

from stpy.fabric.utils import Utils
from stpy.fabric.sl_response import SLResponse

from stpy import HTTPFabricConnection
from stpy import DataspaceType

class TestRowSetProxy(unittest.TestCase):
    def setUp(self):
        Utils.initLogs()
        self.connection = self.openConnection()
        self.accessor = self.initDataspace(self.connection)
        pass

    def tearDown(self):
        self.dropDataspace(self.connection)
        if self.connection:
            self.connection.close()
        pass

    def testRowSetProxyWithDataspaceAccessor(self):

        _self = self
        def execute(query, values = None):
            if values == None:
                return _self.accessor.executeQuery(query)
            else:
                return _self.accessor.executeQuery(query, values)

        def setFetchSize(fetchSize):
            return _self.accessor.setFetchSize(fetchSize)

        def getFetchSize():
            return _self.accessor.getFetchSize()

        def compare(object, sqlQuery):
            _self.assertEquals(object['@type'], "SQLQuery")
            _self.assertEquals(object['sqlScript'], sqlQuery['sqlScript'])

        self._testRowSetProxy(execute = execute, setFetchSize = setFetchSize, getFetchSize = getFetchSize, compare = compare)

    def testRowSetProxyWithSLSession(self):

        session = self.connection.createSLSession()
        self.assertResponse(session.slangRequest("use TSPACE.TEST_TSPACE", 30000), True)

        _self = self
        def execute(query, values = None):
            response = None
            if values == None or len(values) == 0:
                response = session.slangRequest(query, 300000);
                _self.assertResponse(response, True)
            else:
                return _self.accessor.executeQuery(query, values)
            return response.rowSet

        def setFetchSize(fetchSize):
            _self.assertResponse(session.slangRequest("set fetch size " + str(fetchSize), 30000), True);

        def getFetchSize():
            response = session.slangRequest("get fetch size", 30000)
            _self.assertResponse(response, True);
            _self.assertTrue(response.rowSet.next());
            return response.rowSet.getCurrentRow().getColumnByIndex(0)

        def compare(object, sqlQuery):
            if Utils.is_python3():
                _self.assertEquals(type(object), str)
            else:
                _self.assertEquals(type(object), unicode)
            _self.assertEquals(object, "[ OBJECT : SQLQuery ]")

        self._testRowSetProxy(execute = execute, setFetchSize = setFetchSize, getFetchSize = getFetchSize, compare = compare)

    def testFetchSizeCommands(self):
        # thru dataspace accessor
        accessor = self.connection.createDataspaceAccessor(None, DataspaceType.TSPACE, "TEST_TSPACE");
        self.assertEquals(accessor.getFetchSize(), 0);
        accessor.invokeLanguageRequest("set fetch size 2");
        self.assertEquals(accessor.getFetchSize(), 2);

        accessor.setFetchSize(3);
        self.assertEquals(accessor.getFetchSize(), 3);
        response = accessor.invokeLanguageRequest("get fetch size");
        response.rowSet.next();
        self.assertEquals(response.rowSet.getCurrentRow().getColumnByIndex(0), 3);

        # thru sl session
        session = self.connection.createSLSession();
        self.assertFetchSize(session, 300);
        self.assertResponse(session.slangRequest("set fetch size 2", 3000), True);
        self.assertFetchSize(session, 2);

        self.assertResponse(session.slangRequest("use TSPACE.TEST_TSPACE", 3000), True);
        self.assertFetchSize(session, 2);
        self.assertResponse(session.slangRequest("set fetch size 3", 3000), True);
        self.assertFetchSize(session, 3);

        self.assertResponse(session.slangRequest("use ..", 3000), True);
        self.assertFetchSize(session, 3);

        # check command results
        session = self.connection.createSLSession();
        self.assertResponse(session.slangRequest("use TSPACE.TEST_TSPACE", 3000), True);
        self.assertFetchSize(session, 300);

        response = session.slangRequest("create table mytable1 (i int)", 30000);
        self.assertResponse(response, True);
        self.assertIsNone(response.rowSet);

        response = session.slangRequest("insert into mytable1 values (1)", 30000);
        self.assertResponse(response, True);
        self.assertIsNotNone(response.rowSet);
        self.assertRow(response.rowSet, 1, 0, 1);

        response = session.slangRequest("delete from mytable1 where i = 2", 30000);
        self.assertResponse(response, True);
        self.assertIsNotNone(response.rowSet);
        self.assertRow(response.rowSet, 1, 0, 0);

        response = session.slangRequest("update mytable1 set i =2 where i = 1", 30000);
        self.assertResponse(response, True);
        self.assertIsNotNone(response.rowSet);
        self.assertRow(response.rowSet, 1, 0, 1);

        response = session.slangRequest("delete from mytable1 where i = 2", 30000);
        self.assertResponse(response, True);
        self.assertIsNotNone(response.rowSet);
        self.assertRow(response.rowSet, 1, 0, 1);


        response = session.slangRequest("drop table mytable1", 30000);
        self.assertResponse(response, True);
        self.assertIsNone(response.rowSet);

    def assertRow(self, rowSet, row, column, value):
        rowSet.beforeFirst()
        while row > 0:
            row -= 1
            self.assertTrue(rowSet.next())

        c = rowSet.getCurrentRow().getColumnByIndex(column)
        self.assertEquals(c, value)
        # if value instanceof byte[] && c instanceof byte[]:
        #     self.assertEquals((byte[])c, (byte[])value);
        # else:
        #     self.assertEquals(c, value);

    def assertFetchSize(self, session, fetchSize):
        response = session.slangRequest("get fetch size", 300000);
        self.assertResponse(response, True);
        response.rowSet.next();
        self.assertEquals(str(response.rowSet.getCurrentRow().getColumnByIndex(0)), str(fetchSize))

    def assertResponse(self, response, isOK):
        if isOK and not response.isOK:
            if response.exception != None:
                raise response.exception
            if response.text != None:
                raise Exception(response.text)

        self.assertFalse(isOK ^ response.isOK)
        return response;

    def _sqlQuery(self, i):
        return {
            '@type' : "SQLQuery",
            "sqlScript": "select" + str(i),
            "queryType":i
        }

    def _testRowSetProxy(self, execute, setFetchSize, getFetchSize, compare):
        rowSet = None

        # create table
        rowSet = execute("create table test_table (i int, s string, q SQLQuery)")
        self.assertIsNone(rowSet)

        # insert values
        for i in range(0, 50):
            s = "str" + str(i)
            q = self._sqlQuery(i)
            rowSet = execute("insert into test_table values(?, ?, ?)", [i, s, q])
            self.assertIsNotNone(rowSet)
            self.assertTrue(type(rowSet), RowSet)
            self.assertEquals(rowSet.getRowCount(), 1)
            self.assertEquals(rowSet.getMetaData().getColumnCount(), 1)
            self.assertEquals(rowSet.getMetaData().getColumnByIndex(0).name(), "UpdateCount")
            self.assertTrue(rowSet.next())
            self.assertEquals(rowSet.getCurrentRow().getColumnByName("UpdateCount"), 1)
            rowSet.close()

        self._testRowSetProxyPages(execute=execute, setFetchSize=setFetchSize, getFetchSize=getFetchSize,compare=compare)


        # select values without fetch count
        setFetchSize(0)
        self.assertEquals(getFetchSize(), 0)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertEquals(type(rowSet), RowSet)
        self.assertEquals(rowSet.getRowCount(), 50)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)
    
        self._checkRowSetValues(rowSet, compare)
    
        rowSet.close()


        # select values with fetch count = 2
        setFetchSize(2)
        self.assertEquals(getFetchSize(), 2)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertEquals(type(rowSet), RowSetProxy)
        self.assertEquals(rowSet.getRowCount(), 2)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)

        self._checkRowSetValues(rowSet, compare)

        rowSet.close()

        # select values with fetch count = 3
        setFetchSize(3)
        self.assertEquals(getFetchSize(), 3)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertEquals(type(rowSet), RowSetProxy)
        self.assertEquals(rowSet.getRowCount(), 3)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)

        self._checkRowSetValues(rowSet, compare)

        rowSet.close()

        # select values with fetch count = 49
        setFetchSize(49)
        self.assertEquals(getFetchSize(), 49)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertTrue(type(rowSet), RowSetProxy)
        self.assertEquals(rowSet.getRowCount(), 49)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)

        self._checkRowSetValues(rowSet, compare)

        rowSet.close()

        # select values with fetch count = 50
        setFetchSize(50)
        self.assertEquals(getFetchSize(), 50)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertEquals(type(rowSet), RowSet)
        self.assertEquals(rowSet.getRowCount(), 50)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)

        self._checkRowSetValues(rowSet, compare)

        rowSet.close()

        # select values with fetch count = 60
        setFetchSize(60)
        self.assertEquals(getFetchSize(), 60)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertTrue(type(rowSet), RowSet)
        self.assertEquals(rowSet.getRowCount(), 50)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)

        self._checkRowSetValues(rowSet, compare)

        rowSet.close()


        # drop table
        rowSet = execute("drop table test_table")
        self.assertIsNone(rowSet)

    def _testRowSetProxyPages(self, execute, setFetchSize, getFetchSize, compare):
        rowSet = None

        #################################

        # zero fetch size
        setFetchSize(0)
        self.assertEquals(getFetchSize(), 0)

        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertTrue(type(rowSet), RowSet)
        self.assertEquals(rowSet.getRowCount(), 50)
        self.assertEquals(rowSet.getMetaData().getColumnCount(), 3)
        self.assertEquals(rowSet.getPageSize(), 0)

        self.assertFalse(rowSet.nextPage())
        self.assertFalse(rowSet.previousPage())

        rowSet.setPageSize(11)

        # next on the 1st page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 0)

        # 2nd page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 11)

        # 3 page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 22)

        # next on 3 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 23)

        # next on 3 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 24)

        # 3 page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 33)

        # 5 page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 44)

        # 6 page
        self.assertFalse(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 44)

        # next on 5 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 45)

        # next on 5 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 46)

        # 6 page
        self.assertFalse(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 46)

        # previous to 4 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 33)

        # previous to 3 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 22)

        # next on 3 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 23)

        # previous to 2 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 11)

        # previous on 2 page
        self.assertTrue(rowSet.previous())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 10)

        # previous on 1 page
        self.assertFalse(rowSet.previousPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 10)

        # 3 page
        self.assertTrue(rowSet.nextPage())
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 22)

        # previous on 3 page
        self.assertTrue(rowSet.previous())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 21)
        self.assertTrue(rowSet.previous())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 20)

        # 1 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 0)

        rowSet.close()

        ################################

        # non zero fetch size
        setFetchSize(11)
        self.assertEquals(getFetchSize(), 11)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertEquals(type(rowSet), RowSetProxy)

        # 1st page
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getPageSize(), 11)
        self.assertFalse(rowSet.previousPage())

        # next on the 1st page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 0)

        # 2nd page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 11)

        # 3 page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 22)

        # next on 3 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 23)

        # next on 3 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 24)

        # 3 page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 33)

        # 5 page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 6)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 44)

        # 6 page
        self.assertFalse(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 6)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 44)

        # next on 5 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 45)

        # next on 5 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 46)

        # 6 page
        self.assertFalse(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 6)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 46)

        # previous to 4 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 33)

        # previous to 3 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 22)

        # next on 3 page
        self.assertTrue(rowSet.next())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 23)

        # previous to 2 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 11)

        # previous on 2 page
        self.assertTrue(rowSet.previous())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 10)

        # previous on 1 page
        self.assertFalse(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 10)

        # 3 page
        self.assertTrue(rowSet.nextPage())
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 22)

        # previous on 3 page
        self.assertTrue(rowSet.previous())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 21)
        self.assertTrue(rowSet.previous())
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 20)

        # 1 page
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 0)

        rowSet.close()

        #######################################

        # non zero fetch size with set of page size
        setFetchSize(11)
        self.assertEquals(getFetchSize(), 11)
        rowSet = execute("select * from test_table")
        self.assertIsNotNone(rowSet)
        self.assertEquals(type(rowSet), RowSetProxy)

        # 1st page
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getPageSize(), 11)
        self.assertFalse(rowSet.previousPage())

        # set page size
        rowSet.setPageSize(5)
        self.assertEquals(rowSet.getRowCount(), 11)
        self.assertEquals(rowSet.getPageSize(), 5)
        self.assertFalse(rowSet.previousPage())

        # 2nd page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 5)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 11)

        # 3nd page
        self.assertTrue(rowSet.nextPage())
        self.assertEquals(rowSet.getRowCount(), 5)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 16)

        # 2nd
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 5)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 11)

        # 1st
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 5)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 6)

        # 0st
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 5)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 1)

        # -1st
        self.assertTrue(rowSet.previousPage())
        self.assertEquals(rowSet.getRowCount(), 5)
        self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 0)

    def _checkRowSetValues(self, rowSet, compare):
        # from -1 to last
        for i in range(0, 50):
            self.assertTrue(rowSet.next());
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), i);

            compare(rowSet.getCurrentRow().getColumnByIndex(2), self._sqlQuery(i))

        self.assertFalse(rowSet.next());
        self.assertTrue(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # from after last to first
        for i in range(0, 50):
            self.assertTrue(rowSet.previous());
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 49 - i);

        self.assertFalse(rowSet.previous());
        self.assertFalse(rowSet.isAfterLast());
        self.assertTrue(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());


        # from after last to first
        rowSet.afterLast();
        self.assertTrue(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        for i in range(0,50):
            self.assertTrue(rowSet.previous());
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 49-i);

        self.assertFalse(rowSet.previous());
        self.assertFalse(rowSet.isAfterLast());
        self.assertTrue(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # from 5 to last
        rowSet.absolute(5);  # in RowSet implementation first row is 0
        self.assertFalse(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        for i in range(5, 50):
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), i);
            self.assertTrue(rowSet.next() ^ (i == 49));

        self.assertFalse(rowSet.next());
        self.assertTrue(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # from 21 to last
        rowSet.absolute(21);               # in RowSet implementation first row is 0
        self.assertFalse(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        for i in range(0, 21):
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 21-i);
            self.assertTrue(rowSet.previous() ^ (i == 21))

        self.assertFalse(rowSet.previous());
        self.assertFalse(rowSet.isAfterLast());
        self.assertTrue(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # from -1 to last
        rowSet.beforeFirst();
        self.assertFalse(rowSet.isAfterLast());
        self.assertTrue(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        for i in range(0, 50):
            self.assertTrue(rowSet.next());
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), i);

        self.assertFalse(rowSet.next());
        self.assertTrue(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # from first to last
        rowSet.first();
        self.assertFalse(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertTrue(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        for i in range(0, 50):
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), i);
            self.assertTrue(rowSet.next() ^ (i == 49))

        self.assertFalse(rowSet.next());
        self.assertTrue(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # from last to first
        rowSet.last();
        self.assertFalse(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertTrue(rowSet.isLast());

        for i in range(0, 50):
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 49-i);
            self.assertTrue(rowSet.previous() ^ (i == 49))

        self.assertFalse(rowSet.previous());
        self.assertFalse(rowSet.isAfterLast());
        self.assertTrue(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        # test relative
        rowSet.absolute(20);
        rowSet.relative(-10);
        self.assertFalse(rowSet.isAfterLast());
        self.assertFalse(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        for i in range(0, 10):
            self.assertEquals(rowSet.getCurrentRow().getColumnByIndex(0), 10-i);
            self.assertTrue(rowSet.previous() ^ (i == 10))

        self.assertFalse(rowSet.previous());
        self.assertFalse(rowSet.isAfterLast());
        self.assertTrue(rowSet.isBeforeFirst());
        self.assertFalse(rowSet.isFirst());
        self.assertFalse(rowSet.isLast());

        rowSet.beforeFirst();
        self.assertEquals(rowSet.asRowSet().getRowCount(), 50);


    def openConnection(self):
        connection = HTTPFabricConnection("localhost:8888", "Admin", "Admin")
        connection.open()
        self.assertEqual(connection.isOpened(), True)
        self.assertEqual(connection.ping(), "AVAILABLE")
        return connection

    def initDataspace(self, connection):
        slSession = connection.createSLSession()
        slSession.slangRequest("drop dataspace TEST_TSPACE", 30000)
        slSession.slangRequest("create dataspace TEST_TSPACE type TSPACE", 30000)
        slSession.close()

        dataspaceAccessor = connection.createDataspaceAccessor(None, DataspaceType.TSPACE, "TEST_TSPACE")
        dataspaceAccessor.setRequestTimeout(300000)
        self.assertEqual(dataspaceAccessor.getFetchSize(), 0)
        return dataspaceAccessor

    def dropDataspace(self, connection):
        if self.accessor:
            self.accessor.close()
        self.accessor = None
        slSession = connection.createSLSession()
        slSession.slangRequest("drop dataspace TEST_TSPACE", 30000)
        slSession.close()

