"""
Dataspace supports the following types of large object:
    - BLOB - binary large object
    - CLOB - character large object
    - FLOB - file large object

Dataspace LOBs(large objects) returned to client as objects with interface of ``Clob`` or ``Blob``
depending on type of lob.

Maximum size of lob data that will be downloaded at first time can be set with method:
``HTTPDataspaceAccessor.setDownloadableLobSize(size)``
Remaining lob data can be accessed with corresponding LOB methods.

If the size of returned LOB more than downloadable size, lob proxy will be returned.

NOTE: Lob objects allow reading lob data only.

"""

from __future__ import absolute_import


from .type_converter import TypeConverter


class Blob(object):
    """
    Interface for BLOB(binary large object) objects.
    """

    def length(self):
        """
        Returns the number bytes in this BLOB.
        :return: the number bytes in this BLOB.
        """
        raise Exception("Should be implemented in derived classes.")

    def get_bytes(self, pos, length):
        """
        Retrieves all or part of the BLOB value that this Blob
        object represents, as an array of bytes.
        If pos == 0 and length >= data length, original data will be returned.

        :param pos: the ordinal position of the first byte in the BLOB value to be extracted;
                    the first byte is at position 0
        :param length: the number of consecutive bytes to be copied; the value for length must be 0 or greater
        :return:
        """

        raise Exception("Should be implemented in derived classes.")

    def get_all_bytes(self):
        """
        Retrieves and returns all blob content.
        :return: all blob content bytes
        """
        raise Exception("Should be implemented in derived classes.")

class Clob(object):
    """
    Interface for CLOB(character large object) objects.
    """

    def length(self):
        """
        Returns the number bytes in this CLOB.
        :return: the number bytes in this CLOB.
        """
        raise Exception("Should be implemented in inherited classes.")

    def get_sub_string(self, pos, length):
        """
        Retrieves sub string of the CLOB value that this Clob
        object represents, as a string.
        If pos == 0 and length >= data length, original string will be returned.

        :param pos: the ordinal position of the first character in the CLOB value to be extracted;
                    the first character is at position 0
        :param length: the number of consecutive characters to be copied; the value for length must be 0 or greater
        :return:
        """
        raise Exception("Should be implemented in inherited classes.")

    def get_all_string(self):
        """
        Retrieves and returns all clob content.
        :return: all blob content string
        """
        raise Exception("Should be implemented in inherited classes.")


class BlobImpl(Blob):
    """
    Blob implementation for fully downloaded BLOBs.
    """

    def __init__(self, data):
        self._data = data

    def length(self):
        return len(self._data) if self._data is not None else 0

    def get_bytes(self, pos, length):
        if pos < 0:
            raise Exception("pos should be greater or equal to 0")

        if length <= 0:
            raise Exception("length should be greater than 0")

        if self._data is None:
            return None

        if pos >= len(self._data):
            return bytearray(0)

        if pos == 0 and length >= len(self._data):
            return self._data

        if length > len(self._data) - pos:
            length = len(self._data) - pos

        return self._data[pos:pos+length]

    def get_all_bytes(self):
        return self._data


class ClobImpl(Clob):
    """
    Clob implementation for fully downloaded CLOBs.
    """

    def __init__(self, data):
        self._data = data

    def length(self):
        return len(self._data) if self._data is not None else 0

    def get_sub_string(self, pos, length):
        if pos < 0:
            raise Exception("pos should be greater or equal to 0")

        if length <= 0:
            raise Exception("length should be greater than 0")

        if self._data is None:
            return None

        if pos >= len(self._data):
            return ""

        if pos == 0 and length >= len(self._data):
            return self._data

        if length > len(self._data) - pos:
            length = len(self._data) - pos

        return self._data[pos:pos + length]

    def get_all_string(self):
        return self._data

from .abstract_accessible_object_proxy import AccessibleObjectProxy

class AbstractRowSetLobProxy(AccessibleObjectProxy):

    def __init__(self, id):
        self._id = id
        self.rowSetProxy = None

    def getId(self):
        return self._id

    def setId(self, id):
        self._id = id

    def setAccessor(self, accessor):
        self.rowSetProxy = accessor

    def _invokeMethodWithReturn(self, method, params):
        return self.rowSetProxy._invokeRowSetBlobMethodWithReturn(self._id, method, params)

    def _invokeMethod(self, method, params):
        self.rowSetProxy._invokeRowSetBlobMethod(self.getId(), method, params)


class RowSetBlobProxy(Blob, AbstractRowSetLobProxy):
    """
    Blob implementation for partially downloaded BLOBs.
    """
    def __init__(self, id):
        super(RowSetBlobProxy, self).__init__(id)

    def length(self):
        return self._invokeMethodWithReturn("length", [])

    def get_bytes(self, pos, length):

        value = self._invokeMethodWithReturn("getBytes", [pos+1, length])
        from .utils import Utils
        return Utils.b64decode(value)

    def get_all_bytes(self):
        value = self._invokeMethodWithReturn("getBytes", [1, self.length()])
        from .utils import Utils
        return Utils.b64decode(value)


class RowSetClobProxy(Clob, AbstractRowSetLobProxy):
    """
    Clob implementation for partially downloaded CLOBs.
    """
    def __init__(self, id):
        super(RowSetClobProxy, self).__init__(id)

    def length(self):
        return self._invokeMethodWithReturn("length", [])

    def get_sub_string(self, pos, length):
        return self._invokeMethodWithReturn("getSubString", [pos+1, length])

    def get_all_string(self):
        return self._invokeMethodWithReturn("getSubString", [1, self.length()])


class RowSetBlobProxyConverter(TypeConverter):

    @staticmethod
    def readFromMap(rawValue, typeFactory=None):
        return RowSetBlobProxy(rawValue['id'])

    @staticmethod
    def writeToMap(value, typeFactory=None):
        return {
            '@type' : 'RowSetBlobProxy',
            'id' : value.getId()
        }


class RowSetClobProxyConverter(TypeConverter):

    @staticmethod
    def readFromMap(rawValue, typeFactory=None):
        return RowSetClobProxy(rawValue['id'])

    @staticmethod
    def writeToMap(value, typeFactory=None):
        return {
            '@type' : 'RowSetClobProxy',
            'id' : value.getId()
        }

