from __future__ import absolute_import

import logging

from .sl_response import PromptSLResponse
from stpy import HTTPFabricException
from .utils import Utils

logger = logging.getLogger(__name__)


class HTTPSLSession(object):
    def __init__(self, connection, nodeName = None):
        self._name = None
        self._connection = connection
        self._nodeName = nodeName
        self._httpClientWrapper = None
        self._getSLSessionArgs = []
        self._fetchSize = -2
        self._prompt = None

        self._isOpened = False

        self.open()

    def open(self):
        if self._httpClientWrapper:
            self._httpClientWrapper.closeQuiet()
        self._httpClientWrapper = self._connection._createHttpClientWrapper()

        args = []
        if self._nodeName:
            args.append(self._nodeName)

        from .utils import RemoteMethodCallPostprocessor
        call = Utils.remote([Utils.method("createSLSession", args)], RemoteMethodCallPostprocessor.GET_NAME_IF_NOT_NULL)
        r = self._connection._invokeCall(self._httpClientWrapper, call)
        o = Utils.parseJsonResponseAndCheck(r, str)

        if o == None:
            raise HTTPFabricException("Creation of SLSession failed.")

        self._isOpened = True

        self._getSLSessionArgs = [o]

        self._connection._slSessions[self.getName()] = self

    def _reopen(self):
        self.open()

        _self = self
        if self._fetchSize != -2:
            self._connection.reopenAction(lambda : _self.setFetchSize(_self._fetchSize))

        if self._prompt:
            useTo = ""
            if self._prompt.nodeName and self._prompt.nodeName != self._nodeName:
                useTo = self._prompt._nodeName
            if self._prompt.componentType:
                if len(useTo) > 0:
                    useTo += "://"
                    useTo += self._prompt.componentType
                    useTo += "."
                    useTo += self._prompt.componentName

            if len(useTo) > 0:
                command = "use " + useTo
                self._connection._reopenAction(lambda: _self.slangRequest(command, 10000))

    def getName(self):
        if not self._name:
            self._checkOpened()
            r = self._invokeMethod("getName")
            self._name = Utils.parseJsonResponseAndCheck(r, str)
        return self._name

    def getFetchSize(self):
        self._checkOpened()
        r = self._invokeMethod("getFetchSize")
        return Utils.parseJsonResponseAndCheck(r, int)

    def setFetchSize(self, fetchSize):
        self._checkOpened()
        r = self._invokeMethod("setFetchSize", [fetchSize])
        Utils.parseJsonResponseAndCheck(r)
        self._fetchSize = fetchSize

    def getRequestTimeout(self):
        self._checkOpened()
        r = self._invokeMethod("getRequestTimeout")
        return Utils.parseJsonResponseAndCheck(r, int)

    def slangRequest(self, request, timeout = None):
        self._checkOpened()
        arguments = [request, self._calculateSlangTimeout(self._connection, timeout, -1)]
        r = self._invokeMethod("slangRequest", arguments, timeout = Utils.calculateHttpTimeout(timeout, -1))
        o = Utils.parseJsonResponseAndCheck(r, "SLResponse")
        o = self._connection.getTypeFactory().readFromMap(o, "SLResponse")
        o = Utils.processAccessibleObjects(o, self)

        if type(o) == PromptSLResponse:
            self._prompt = o
        return o

    def close(self):
        try:
            self.getName()
            self._invokeMethod("close", [])
        finally:
            if self._name:
                del self._connection._slSessions[self._name]
                if len(self._getSLSessionArgs) == 1:
                    self._connection._closeSLFiles(self._getSLSessionArgs[0])

            if self._httpClientWrapper:
                self._httpClientWrapper.closeQuiet()
                self._httpClientWrapper = None

            self._isOpened = False

    @staticmethod
    def _calculateSlangTimeout(connection, timeout, defaultTimeout):
        if timeout == None:
            timeout = defaultTimeout
        if timeout < 0 and defaultTimeout > 0:
            timeout = defaultTimeout

        if timeout < 0:
            timeout = connection._httpTimeoutMs
        if timeout > 3000:
            timeout -= 3000
        return timeout

    def _invokeMethod(self, methodName, args=[], timeout=-1):
        return self._connection._invokeCall(self._httpClientWrapper, Utils.remote([Utils.method("getSLSession", self._getSLSessionArgs), Utils.method(methodName, args)]), timeout = timeout)

    def _raiseSystemRequest(self, request, timeout):
        args = [request, HTTPSLSession._calculateSlangTimeout(self._connection, timeout, -1)]
        result = self._invokeMethod("raiseSystemRequestInDataspaceAccessor", args, Utils.calculateHttpTimeout(timeout, -1))
        result = Utils.parseJsonResponseAndCheck(result)
        return result

    def _log(self, func, message, *args):
        if self._connection:
            self._connection._log(func, message, *args)
        else:
            Utils.log(func, message, args)

    def _checkOpened(self):
        if not self._isOpened:
            from stpy import HTTPFabricException
            raise HTTPFabricException("This operation is not allowed on closed session.")
