from __future__ import absolute_import

import json
import logging
import threading
import time

from .http_sl_session import HTTPSLSession
from .http_client import HTTPClient
from .http_fabric_exception import HTTPFabricException
from .http_reverse_runnable import HTTPReverseRunnable
from .http_dataspace_accessor import HTTPDataspaceAccessor
from .utils import Utils
from .enums import EventScope
from .enums import HTTPFabricConnectionState

logger = logging.getLogger(__name__)

class HTTPFabricConnection(object):

    def __init__(self, url, username = None, password = None):
        self._url = url
        self._username = username
        self._password = password
        self._isOpened = False
        self._httpClientWrapper = None
        self._sessionId = None
        self._httpTimeoutMs = 30000
        self._reverseConnectionHoldTimeoutMs = 10000
        self._reverseConnectionReplyTimeoutMs = 12000

        self._fileTransfersDirectConsumersTimeoutMs = 30000 * 2/3

        self._reverseRunnable = None
        self._reverseThread = None

        self._dataspaceAccessors = {}
        self._slSessions = {}

        self._stateListenersLock = threading.RLock()
        self._reopenLock = threading.RLock()
        self._stateListeners = []

        self._reconnectAttempts = -1
        self._reconnectInterval = 1
        self._connectionReopenInterval = 30

        self._isReopenOnAnyCall = False
        self._maxRetriesCount = 2
        self._isReopening = False

        # statistics
        self._serverConnectionLostFlag = False
        self._serverConnectionLostStartTime = 0
        self._serverConnectionRepairedTime = 0
        self._reopenTime = 0
        self._reopenCount = 0
        self._reopenFailedTime = 0
        self._reopenError = None
        self._isOpenedRequests = HTTPRequestsStatistics()
        self._allRequests = HTTPRequestsStatistics()

        self._params = {
            'name' : None,
            'description' : None,
            'eventScope' : 'INHERITED',
            'clientId' : None,
            'leasePeriod' : 60000
        }

        self.ssl = None

        from .types_factory import TypeFactory
        self._typeFactory = TypeFactory()

    def password(self, password):
        self._checkNotOpened()
        self._password = password

    def username(self, username):
        self._checkNotOpened()
        self._username = username

    def setHttpTimeoutMs(self, httpTimeoutMs):
        self._httpTimeoutMs = httpTimeoutMs

    def getName(self):
        if self._isOpened:
            return self._component['name']
        else:
            return self._params['name']

    def setReconnectAttempts(self, reconnectAttempts):
        self._reconnectAttempts = reconnectAttempts

    def getReconnectAttempts(self):
        return self._reconnectAttempts

    def setReconnectInterval(self, reconnectInterval):
        self._reconnectInterval = reconnectInterval

    def getReconnectInterval(self):
        return self._reconnectInterval

    def setConnectionReopenInterval(self, connectionReopenInterval):
        self._connectionReopenInterval = connectionReopenInterval

    def getConnectionReopenInterval(self):
        return self._connectionReopenInterval

    def isReopenOnAnyCall(self):
        return self._isReopenOnAnyCall

    def setReopenOnAnyCall(self, isReopenOnAnyCall):
        self._isReopenOnAnyCall = isReopenOnAnyCall

    def isReconnecting(self):
        return self._isOpened and self._serverConnectionLost or self._isReopening or self._reopenFailedTime > self._reopenTime

    def addStateListener(self, listener):
        self._stateListeners.append(listener)

    def isOpened(self):
        return self._isOpened

    def getTypeFactory(self):
        return self._typeFactory;

    def setSsl(self, ssl):
        """
        Sets SSL parameters. It is a dictionary of the following parameters:
            - cert_reqs - specifies whether a certificate is required from the other side of the connection, and whether it will be validated if provided.
                          Should be one of the following values:
                            * CERT_NONE or None - certificates ignored
                            * CERT_OPTIONAL - not required, but validated if provided
                            * CERT_REQUIRED - required and validated
                          If the value of this parameter is not CERT_NONE, then the ca_certs parameter must point to a file of CA certificates.
            - ca_certs - file contains a set of concatenated 'certification authority' certificates, which are used to validate
                       certificates passed from the other end of the connection.
            - ca_cert_dir - a directory containing CA certificates in multiple separate files, as supported by OpenSSL's -CApath flag or
                            the capath argument to SSLContext.load_verify_locations().
            - ssl_version - specifies which version of the SSL protocol to use. Optional.
            - key_file and cert_file - optional files which contain a certificate to be used to identify the local side of the connection.
            - disable_warnings - specifies disable or not InsecureRequestWarning warning, by default True.
        """
        self._checkNotOpened()
        self.ssl = ssl

    def open(self):
        self._reopenLock.acquire()
        try:
            if self._isOpened:
                return

            self._log(logger.debug, "Opening connection...")
            self._resetStatistics()

            self._openInternal()

            self._isOpened = True
            self._openTime = time.time()
            self._onStateChange(HTTPFabricConnectionState.OPENED)

            # TODO:
            #doSetFileTransfersDirectRequestsTimeout(httpClientWrapper);
            self._doSetReverseConnectionHoldTimeout(self._httpClientWrapper)
            #createSLFileRequestConsumer();

            self._log(logger.info, "Connection opened. httpTimeoutMs: %s ms, leasePeriod: %s ms, fileTransfersDirectConsumersTimeoutMs: %s ms, reverseReplyTimeoutMs: %s ms, reverseConnectionHoldTimeoutMs: %s ms",
                      self._httpTimeoutMs, self._leasePeriod, self._fileTransfersDirectConsumersTimeoutMs, self._reverseConnectionReplyTimeoutMs, self._reverseConnectionHoldTimeoutMs)

            self._startReverseThread()
        finally:
            if not self._isOpened and self._httpClientWrapper != None:
                self._httpClientWrapper.closeQuiet()
                self._httpClientWrapper = None
            self._reopenLock.release()

    def _openInternal(self):
        self._httpClientWrapper = self._createHttpClientWrapper()

        scopeObject = {
            '@type' : 'EventScope',
            'value' : self._params['eventScope']
        }
        openModeObject = {
            '@type' : 'LinkMode',
            'value' : 'START'
        }

        openArguments = [self._params['name'], self._params['description'], self._params['clientId'], scopeObject, self._params['leasePeriod'], openModeObject]
        r = self._invokeMethod(self._httpClientWrapper, "open", openArguments, "login")
        o = Utils.parseJsonResponse(r)
        self._log(logger.debug, "response object: %s", o)
        Utils.processServerAnswer(o, "HTTPServerFabricConnectionInfo")
        if o == None:
            raise HTTPFabricException("Null response from server received.")

        self._component      = o['component']
        self._sessionId      = o['sessionId']
        self._clientId       = o['clientId']
        self._leasePeriod    = o['leasePeriod']
        self._sessionTimeout = o['sessionTimeout']
        self._lastAccessTime = o['lastAccessTime']
        self._openLinkMode   = o['openLinkMode']
        self._closeLinkMode  = o['closeLinkMode']

        self._reverseConnectionHoldTimeoutMs = o['reverseConnectionHoldTimeout']
        self._reverseConnectionReplyTimeoutMs = self._reverseConnectionHoldTimeoutMs + 2000

        if self._sessionId == None or len(self._sessionId) == 0 or self._sessionId == "null":
            self._component = {}
            self._sessionId = ""
            raise HTTPFabricException("Failed to open HTTP connection, sessionId is null, please retry again.")

    def close(self):
        self._closeInternal()

    def closeQuiet(self):
        try:
            self.close()
        except Exception as e:
            pass

    def _closeInternalQuiet(self, timeout):
        try:
            self._closeInternal(1)
        except:
            pass

    def _closeInternal(self, timeout = 2000):
        try:
            if not self._isOpened:
                return
            self._log(logger.info, "Closing connection.")
            self._isOpened = False

            with self._reopenLock:
                try:
                    self._invokeMethod(self._httpClientWrapper, "close", [], "logout")
                except Exception as e:
                    self._log(logger.exception, e)

                self._httpClientWrapper.closeQuiet()
                self._httpClientWrapper = None
                self._dataspaceAccessors = {}
                self._slSessions = {}

                self._stopReverseThread(timeout)

                self._component = {}
                self._sessionId = None
        finally:
            self._onStateChange(HTTPFabricConnectionState.CLOSED)


    def _startReverseThread(self):
        self._stopReverseThread()
        self._reverseRunnable = HTTPReverseRunnable(self)

        self._reverseThread = threading.Thread(target = self._reverseRunnable, name = "ReverseThread")
        self._reverseThread.setDaemon(True)
        self._reverseThread.start()

    def _stopReverseThread(self, timeout = 2000):
        if self._reverseRunnable != None:
            self._reverseRunnable.stop()
            self._reverseRunnable = None

        if self._reverseThread != None:
            try:
                self._reverseThread.join(timeout / 1000)
            except Exception as e:
                self._log(logger.exception, e)
            self._reverseThread = None

    def ping(self):
        try:
            r = self._invokeMethod(self._httpClientWrapper, "ping", [])
            o = Utils.parseJsonResponse(r)
            if o != None and o['@type'] == 'PingResult':
                return o['value']
        except Exception as e:
            return "UNAVAILABLE"
        return "UNAVAILABLE"


    def _checkNotOpened(self):
        if self._isOpened:
            raise HTTPFabricException("This operation if not allowed on opened connection.")

    def _checkOpened(self):
        if not self._isOpened:
            raise HTTPFabricException("This operation if not allowed on not opened connection.")

    def _createHttpClient(self, timeout = None):
        httpClient = HTTPClient(self, self._url, timeout, self.ssl)
        httpClient._username = self._username
        httpClient._password = self._password
        return httpClient

    def _createHttpClientWrapper(self, timeout = None):
        return HTTPClientWrapper(self._createHttpClient(timeout), self._url)

    def _fabricConnectionLost(self):
        self._onStateChange(HTTPFabricConnectionState.FABRIC_CONNECTION_LOST)
        self.closeQuiet()


    def _serverConnectionLost(self):
        if not self._serverConnectionLostFlag:
            self._serverConnectionLostFlag = True
            self._serverConnectionLostStartTime = time.time()
            self._onStateChange(HTTPFabricConnectionState.SERVER_CONNECTION_LOST)

    def _serverConnectionRepaired(self):
        if self._serverConnectionLostFlag:
            self._serverConnectionLostFlag = False
            self._serverConnectionRepairedTime = time.time()
            self._onStateChange(HTTPFabricConnectionState.SERVER_CONNECTION_REPAIRED)

    def _onStateChange(self, state, params = None):
        with self._stateListenersLock:
            for listener in self._stateListeners:
                try:
                    listener.onStateEvent(HTTPFabricConnectionStateEvent(state, params))
                except Exception as e:
                    self._log(logger.error, "On state change callback exception.")
                    self._log(logger.exception, e)

    def _closeSLFiles(self, slSessionName):
        # TODO:
        # slFileMessageProcessor.close(slSessionName)
        pass

    def _log(self, func, message, *args):
        if type(message) == str:
            Utils.log(func, self.toString() + " " + message, args)
        else:
            Utils.log(func, self.toString() + " %s", [message])
            Utils.log(func, message, [])

    def toString(self):
        return "[ {}, sessionId: {}, isOpened: {} ]".format(self.getName(), self._sessionId, self._isOpened)

    def getStatistics(self):
        statistics = HTTPStatistics()
        statistics.opened = self.isOpened()
        statistics.openTime = self._openTime
        statistics.connected = self.isOpened() and not self.isReconnecting()
        statistics.lastServerConnectionLostTime = self._serverConnectionLostStartTime
        statistics.lastServerConnectionRepairedTime = self._serverConnectionRepairedTime
        statistics.lastReconnectRetries = self._reverseConnectionErrorsCount \
                if self._reverseConnectionErrorsCountLast == 0 else self._reverseConnectionErrorsCountLast
        statistics.lastReopenTime = self._reopenTime
        statistics.reopenCount = self._reopenCount
        statistics.lastReopenError = self._reopenError
        statistics.lastReopenFailedTime = self._reopenFailedTime
        if self._reopenFailedTime > self._reopenTime:
            statistics.nextReopenAt = self._reopenFailedTime + self.getConnectionReopenInterval() * 1000

        statistics.isOpenedRequests = self._isOpenedRequests
        statistics.reverseRequests = self._reverseRunnable._reverseRequests if self._reverseRunnable else None
        statistics.allRequests = self._allRequests

        return statistics

    def _resetStatistics(self):
        self._serverConnectionLostStartTime = 0
        self._serverConnectionRepairedTime = 0
        self._reopenTime = 0
        self._reopenCount = 0
        self._reopenFailedTime = 0
        self._reopenError = None
        self._isOpenedRequests = HTTPRequestsStatistics()
        self._allRequests = HTTPRequestsStatistics()

        if self._reverseRunnable != None:
            self._reverseRunnable._reverseConnectionErrorsCountLast = 0
            self._reverseRunnable._reverseRequests = HTTPRequestsStatistics()
        pass

    def _doSetReverseConnectionHoldTimeout(self, wrapper):
        try:
            self._invokeMethod(wrapper, "setReverseConnectionHoldTimeout", [self._reverseConnectionHoldTimeoutMs])
            timeout = self._invokeMethod(wrapper, "getReverseConnectionHoldTimeout")
            timeout = Utils.parseJsonResponseAndCheck(timeout, int)
            self._reverseConnectionHoldTimeoutMs = timeout

            self._log(logger.info, "reverseConnectionHoldTimeout set to %s ms.", self._reverseConnectionHoldTimeoutMs)
        except Exception as exception:
            # TODO: remove, for now it means that runtime version is old and doesn't support set of reverseConnectionHoldTimeoutMs
            self._reverseConnectionHoldTimeoutMs = self._leasePeriod / 2
            self._reverseConnectionReplyTimeoutMs = 5000
            self._log(logger.error, "Failed to set reverseConnectionHoldTimeoutMs on server, set it to %s", self._reverseConnectionHoldTimeoutMs)

        if self._reverseConnectionReplyTimeoutMs > self._reverseConnectionHoldTimeoutMs + 2000:
            self._reverseConnectionReplyTimeoutMs = self._reverseConnectionHoldTimeoutMs + 2000
            self._log(logger.info, "reverseReplyTimeoutMs set to %s ms.", self._reverseConnectionReplyTimeoutMs)

    def _invokeMethod(self, wrapper, methodName, args=[], query=None, timeout=-1):
        return self._invokeCall(wrapper, Utils.remote([Utils.method(methodName, args)]), query = query, timeout = timeout)

    def _invokeCall(self, wrapper, remoteCall, query = None, timeout = -1):
        if not self._reopenLock.acquire(False):
            raise HTTPFabricException("Fabric connection reopening is in progress...")
        try:
            # should be never happen, since we have one lock instead of read and write locks
            # if self._isReopening and not self.isOpenCall(remoteCall) and not self.reopenLock.isHeldByCurrentThread():
            #     raise HTTPFabricException("Fabric connection reopening is in progress...");
            # synchronization done for httpClient to allow multi-thread using for touch and confirmation requests
            with wrapper.lock:
                return wrapper.httpClient.invokeCall(remoteCall, query = query, timeout = timeout)
        finally:
            self._reopenLock.release()

    def _reopenFailed(self, exception):
        self._log(logger.error, "Reopen exception: %s", repr(exception))
        if self._sessionId and len(self._sessionId) != 0:
            self._closeSession()
            self._sessionId = "xxx"

        self._reopenFailedTime = time.time()
        self._reopenError = exception
        self._onStateChange(HTTPFabricConnectionState.REOPEN_FAILED, exception)

        raise HTTPFabricException("Reopen failed.", exception)


    def _closeSession(self):
        if self._sessionId and len(self._sessionId) != 0 and self._url:
            try:
                wrapper = self._createHttpClientWrapper(5000)
                data = Utils.dumpMapToJson(Utils.remote([Utils.method("close", [])]))
                params = wrapper.httpClient._makePostParams("logout", data)
                try:
                    wrapper.httpClient._postInternal(params)
                    wrapper.closeQuiet()
                except Exception as e:
                    # TODO: thread to close session in background
                    pass
            except Exception as e:
                pass

    def _checkNotOpenedAndThrowOnReopen(self):
        if not self._isOpened:
            self._isOpened = True
            self._closeInternal(1)
            raise HTTPFabricException("Connection closed.")

    def _reopenAction(self, func):
        try:
            func()
        except Exception as exception:
            self._reopenFailed(exception)

    def _reopen(self):
        if not self._reopenLock.acquire(False):
            return False

        try:
            if not self._isOpened:
                raise HTTPFabricException("Connection closed.")

            self._isReopening = True

            self._onStateChange(HTTPFabricConnectionState.REOPEN_START)

            self._log(logger.info, "Reopening connection.")

            self._sessionId = None

            try:
                self._openInternal()
            except Exception as exception:
                if not self._sessionId:
                    # if connection was not opened we should only throw reopen failed error
                    self._reopenError = exception
                    self._onStateChange(HTTPFabricConnectionState.REOPEN_FAILED, exception)
                    raise exception
                else:
                    self._reopenFailed(exception)

            # TODO:
            # try:
            #     self._doSetFileTransfersDirectRequestsTimeout(self._httpClientWrapper);
            # except Exception as exception:
            #     self._reopenFailed(exception);

            try:
                self._doSetReverseConnectionHoldTimeout(self._httpClientWrapper)
            except Exception as exception:
                self._reopenFailed(exception)

            # TODO:
            # try:
            #     self._requestConsumers.remove(SLFileMessage.getRequestConsumerName())
            #     self._createSLFileRequestConsumer()
            # raise Exception as exception:
            #     self._reopenFailed(exception)

            def reopen(self, func):
                try:
                    func()
                except Exception as exception:
                    self._reopenFailed(exception)
                    self._checkNotOpenedAndThrowOnReopen()

            # TODO:
            # self._log(logger.ingo, "Rebinding events.")
            # for eventId : self._boundEvents:
            #     reopen(self, bindProducerFor(eventId))

            #  slang session
            self._log(logger.info, "Recreating slang sessions.")
            for slSession in self._slSessions.values():
                reopen(self, lambda: slSession._reopen())

            # // direct consumers
            # log(Level.INFO, "Recreating direct event consumers.");
            # {
            # for (HTTPEventConsumer consumer : directConsumers.values())
            # reopen.accept(consumer::open);
            # }
            #
            # // async consumers
            # log(Level.INFO, "Recreating async event consumers.");
            # {
            # for (HTTPEventAsyncConsumer consumer : asyncConsumers.values())
            # reopen.accept(consumer::open);
            # }
            #
            # // receivers
            # log(Level.INFO, "Recreating receivers.");
            # {
            # for (HTTPEventReceiver receiver : receivers.values())
            # reopen.accept(receiver::open);
            # }
            #
            # // request consumers
            # log(Level.INFO, "Recreating request consumers.");
            # {
            # requestConsumers.values().stream().filter(consumer -> !SLFileMessage.getRequestConsumerName().equals(consumer.getName()))
            # .forEach(consumer -> reopen.accept(consumer::open));
            # }
            #
            # // service accessors
            # log(Level.INFO, "Recreating service accessors.");
            # {
            # for (HTTPServiceAccessor serviceAccessor : serviceAccessors.values())
            # reopen.accept(serviceAccessor::open);
            # }

            # dataspace accessors
            self._log(logger.info, "Recreating dataspace accessors.")
            for dataspaceAccessor in self._dataspaceAccessors.values():
                reopen(self, lambda : dataspaceAccessor.open())

            self._log(logger.info, "Connection reopened. httpTimeoutMs: %s ms, leasePeriod: %s ms, \
                        fileTransfersDirectConsumersTimeoutMs: %s ms, reverseReplyTimeoutMs: %s ms, \ reverseConnectionHoldTimeoutMs: %s ms",
                        self._httpTimeoutMs, self._leasePeriod, self._fileTransfersDirectConsumersTimeoutMs, self._reverseConnectionReplyTimeoutMs,
                        self._reverseConnectionHoldTimeoutMs)

            self._isReopening = False
            self._reopenTime = time.time()
            self._reopenCount += 1
            self._onStateChange(HTTPFabricConnectionState.REOPEN_END)

            self._checkNotOpenedAndThrowOnReopen()

            return True
        finally:
            self._isReopening = False
            self._reopenLock.release()


    def createDataspaceAccessor(self, nodeName, dataspaceType, dataspaceName):
        self._checkOpened()
        if self._component['eventScope'] == EventScope.LOCAL:
            return None

        return HTTPDataspaceAccessor(dataspaceType, dataspaceName, self, nodeName)

    def createSLSession(self, nodeName = None):
        self._checkOpened()
        if self._component['eventScope'] == EventScope.LOCAL:
            return None
        return HTTPSLSession(self, nodeName)

    def importSemanticType(self, name):
        self._checkOpened()
        session = self.createSLSession(None)
        try:
            # command = "describe semantic type {} as json(notation=type; level=ROOT_ELEMENT,COMPLEX_OBJECTS;prettyprint=true)".format(name)
            command = "analyze semantic type {} as json".format(name)
            response = session.slangRequest(command)
            if not response.isOK:
                if response.text is not None:
                    raise HTTPFabricException("Import of semantic type failed. Cause: ", response.text)
                if response.exception is not None:
                    raise HTTPFabricException("Import of semantic type failed.", response.exception)
                raise HTTPFabricException("Import of semantic type failed.")

            if response.text is None:
                raise HTTPFabricException("Import of semantic type failed. No json received.")
            self._log(logger.debug, "Semantic type json: {}", response.text)
            map = json.loads(response.text)

            self._typeFactory._buildAndRegisterUserTypeFromTypeMap(map)
        finally:
            session.close()

class HTTPFabricConnectionStateEvent(object):
    def __init__(self, state, params = None):
        self.state = state
        self.params = params


class HTTPFabricConnectionStateListener(object):
    def onStateEvent(self, event):
        pass


class HTTPClientWrapper(object):
    def __init__(self, httpClient, url):
        self.httpClient = httpClient
        self.url = url
        self.lock = threading.RLock()

    def replaceClient(self, httpClient):
        oldClient = self.httpClient
        self.httpClient = httpClient
        self.closeQuiet(oldClient)

    def getHttpClient(self):
        return self.httpClient

    def closeQuiet(self, client = None):
        if not client:
            client = self.httpClient
            self.httpClient = None

        if client != None:
            try:
                client.destroy()
            except Exception:
                pass

    def lock(self):
        self.lock.acquire()

    def unlock(self):
        self.lock.unlock()


class HTTPRequestsStatistics(object):
    def __init__(self):
        self.requestsCount = 0
        self.unauthorizedResponsesCount = 0
        self.successfulResponsesCount = 0
        self.timeoutResponsesCount = 0
        self.errorResponsesCount = 0
        self.exceptionResponsesCount = 0
        self.duplicateResponsesCount = 0
        self.bytesSent = 0
        self.bytesReceived = 0
        self.retriesCount = 0

class HTTPStatistics(object):
    def __init__(self):
        self.opened = None
        self.openTime = None
        self.connected = None
        self.lastServerConnectionLostTime = None
        self.lastServerConnectionRepairedTime = None
        self.lastReconnectRetries = None
        self.lastReopenTime = None
        self.reopenCount = None
        self.lastReopenFailedTime = None
        self.lastReopenError = None
        self.nextReopenAt = None

        self.reverseRequests = None
        self.isOpenedRequests = None
        self.allRequests = None
