from __future__ import absolute_import

import logging

from stpy.fabric.utils import Utils
from stpy.fabric.row_set import RowSet
from stpy import ds
from stpy.ds.exceptions import *

logger = logging.getLogger(__name__)

class Cursor(object):
    """
    Cursor attributes
        .description
            This read-only attribute is a sequence of 7-item sequences.
            Each of these sequences contains information describing one result column:
                name
                type_code
                display_size
                internal_size
                precision
                scale
                null_ok
            The first two items ( name and type_code ) are mandatory,
            the other five are optional and are set to None if no meaningful values can be provided.

            This attribute will be None for operations that do not return rows or
            if the cursor has not had an operation invoked via the .execute*() method yet.

            The type_code can be interpreted by comparing it to the Type Objects specified in the section below.

        .rowcount
            This read-only attribute specifies the number of rows that the last .execute*() produced
            (for DQL statements like SELECT ) or affected (for DML statements like UPDATE or INSERT ).

            The attribute is None in case no .execute*() has been performed on the cursor or the rowcount
            of the last operation cannot be determined by the interface.

        .arraysize
            Defines the fetch size used to fetch rows from server, default one row.
            Also defines default number of rows fetchmany() will fetch.
            This value should be set before callproc() or execute() call.
            A value of 0 or -1 means fetch all rows in first call.
    """

    def __init__(self, connection):
        self.connection = connection
        self.description = None
        self.rowcount = None
        self.arraysize = 1
        self._rowset = None

        self.messages = []
        self.errorhandler = self.connection.errorhandler
        # TODO:
        self.lastrowid = None

    def __del__(self):
        self.close()
        self.errorhandler = None

    def close(self):
        """
            Closes the cursor. Future operations raise ``ProgrammingError``.
            If you are using fetch size, it is very important to
            close the cursor when you are done with it and before creating a
            new one.
            :return:
        """

        if self._rowset:
            self._rowset.close()
            self._rowset = None

        if not self.connection:
            return
        self.connection = None

    def _check_opened(self):
        if self.connection == None:
            self.errorhandler(self, ProgrammingError, ProgrammingError("Cursor closed."))

    def callproc(self, funcname, parameters=None):
        """
        Call a stored dataspace function with the given name.
        The sequence of parameters must contain one entry for each argument that the procedure expects.
        The operation uses .arraysize value as fetch size to fetch rows from server.

        Function returns passed parameters.
        Function results can be accessed through the standard .fetch*() methods.

        :param funcname: procedure name
        :param parameters: arguments
        :return:
        """

        if not Utils.is_python3():
            if isinstance(funcname, unicode):
                funcname = funcname.encode()
        query = "call " + funcname + "("
        if len(parameters) > 0:
            for parameter in parameters:
                query += "?,"
            query = query[0:len(query) - 1]
        query += ")"

        self.execute(query, parameters)

        return parameters

    def execute(self, query, parameters=None):
        """
                Prepare and execute a database operation (query or command).
                Operations and parameters should be in question mark format.
                Parameters may be provided as sequenxe or array.
                The operation uses .arraysize value as fetch size to fetch rows from server.

                A reference to the operation will be retained by the cursor.
                If the same operation object is passed in again, then the cursor can optimize its behavior.
                This is most effective for algorithms where the same operation is used,
                but different parameters are bound to it (many times).

                For maximum efficiency when reusing an operation,
                it is best to use the .setinputsizes() method to specify the parameter types and
                sizes ahead of time.

        :param query: DSQL query in question mark format
        :param parameters: list of parameters of query
        :return:
        """

        self._reset_and_check_before_execute()
        if not Utils.is_python3():
            if isinstance(query, unicode):
                query = query.encode()

        try:
            self._rowset = self.connection._query(query, parameters, self.arraysize)
        except Exception as e:
            self.messages.append((type(e), e))
            self.errorhandler(self, type(e), e)

        if self._rowset != None:
            meta = self._rowset.getMetaData()
            self.description = [
                (
                    column.name(),
                    ds.ds.define_type(column.type()),
                    None,
                    None,
                    column.precision(),
                    column.scale(),
                    column.nullable()
                )
                for column in meta.getColumns()
                ]
            if type(self._rowset) == RowSet:
                self.rowcount = self._rowset.getRowCount()
            else:
                self.rowcount = None

    def fetchone(self):
        """
            Fetch the next row of a query result set, returning a single sequence, or None when no more data is available.
            The operation uses .arraysize value as fetch size to fetch rows from server.

            An OperationalError exception is raised if the previous call to .execute*()
            did not produce any result set or no call was issued yet.

        :return: tuple or None
        """

        del self.messages[:]
        self._check_row_set_not_none()

        if not self._rowset.next():
            return None

        row = self._rowset.getCurrentRow()
        return row._data

    def next(self):
        """
            Fetch the next row of a query result set, returning a single sequence, or None when no more data is available.
            The operation uses .arraysize value as fetch size to fetch rows from server.

            An OperationalError exception is raised if the previous call to .execute*()
            did not produce any result set or no call was issued yet.

        :return: tuple or None
        """

        return self.fetchone()

    def fetchmany(self, size=None):
        """
            Fetch the next set of rows of a query result, returning a sequence of sequences (e.g. a list of tuples).
            An empty sequence is returned when no more rows are available.

            The number of rows to fetch per call is specified by the parameter.
            If it is not given, the cursor's .arraysize determines the number of rows to be fetched.
            The method should try to fetch as many rows as indicated by the size parameter.
            If this is not possible due to the specified number of rows not being available, fewer rows may be returned.

            The operation uses .arraysize value as fetch size to fetch rows from server.

            An OperationalError exception is raised if the previous call to .execute*()
            did not produce any result set or no call was issued yet.

            Note there are performance considerations involved with the size parameter.
            For optimal performance, it is usually best to use the .arraysize attribute.
            If the size parameter is used, then it is best for it to retain the same value from one .fetchmany() call to the next.

        :param size: maximum number of rows to fetch, 0 or negative values mean fetch all rows
        :return: sequence of tuples or empty sequence
        """

        del self.messages[:]
        self._check_row_set_not_none()

        if size == None:
            size = self.arraysize

        result = []
        i = 0
        while (size < 0 or i < size) and self._rowset.next():
            i += 1
            result.append(self._rowset.getCurrentRow()._data)
        return result

    def fetchall(self):
        """
        Fetch all (remaining) rows of a query result, returning them as a sequence of sequences (e.g. a list of tuples).
        Note that the cursor's arraysize attribute can affect the performance of this operation.

        An OperationalError exception is raised if the previous call to .execute*() did not produce any result set or no call was issued yet.
        :return: sequence of tuples
        """

        return self.fetchmany(-1)

    def scroll(self, count, mode='relative'):
        """
            Scroll the cursor in the result set to a new position according to mode .

            If mode is relative (default), value is taken as offset to the current position in the result set,
            if set to absolute , value states an absolute target position.

        :param count: rows count to scroll
        :param mode: relative - scroll count rows from current position, absolute - scroll count rows from rowset begining
        :return: Row on the scrolled position, None if position out of the row set
        """

        del self.messages[:]
        result = None
        if mode == 'relative':
            result = self._rowset.relative(count)
        elif mode == 'absolute':
            result = self._rowset.absolute(count)
        if result == True:
            result = self._rowset.getCurrentRow()._data
        else:
            result = None
        return result

    def __iter__(self):
        return self

    def _check_row_set_not_none(self):
        if self._rowset == None:
            exception = OperationalError("No execute call was performed or previous execute call not produced any result")
            self.messages.append((type(exception), exception))
            self.errorhandler(self, type(exception), exception)

    def _reset_and_check_before_execute(self):
        del self.messages[:]
        self.description = None
        self.rowcount = None
        if self._rowset is not None:
            try:
                self._rowset.close()
            except Exception:
                pass
            self._rowset = None

        self._check_opened()

    def executemany(self, operation, *seq_of_parameters):
        """
            NOT SUPPORTED
        """
        raise NotSupportedError("Execute many is not supported.")

    def nextset(self):
        """
            NOT SUPPORTED
        """
        raise NotSupportedError("Next set is not supported.")

    def setinputsizes(self, *args):
        """Does nothing, required by API."""

    def setoutputsizes(self, *args):
        """Does nothing, required by API."""
