package streaming;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.utils.Utils;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.event.*;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sef.*;
import com.streamscape.sef.enums.EventScope;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: First <i>Streaming</i> sample.
 *
 * <p>This sample initializes the <code>Sample.Node1</code> node and starts first data stream within this node.
 * Also this node can process a second data stream received from the <code>Sample.Node2</code> node.
 *
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class Stream1Sample
 {
    static RuntimeContext   context;
    static FabricConnection connection1; // It is used for consumers.
    static FabricConnection connection2; // It is used for event sending.

    static final String     streamEventId1  = "event.stream1";
    static final String     chunkEventId1   = ChunkEvent.getEventId(streamEventId1);
    static final String     streamEventId2  = "event.stream2";
    static final String     chunkEventId2   = ChunkEvent.getEventId(streamEventId2);

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("streaming.*",               Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node1");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node1");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("Add event prototypes...\n");
          addEventPrototypes();

          System.out.println("Create consumers...\n");
          createConsumers();

          System.out.println("Raise stream event...\n\n");
          raiseStreamEvent();

          System.out.println();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void addEventPrototypes() throws Exception
    {
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();

       // Stream 1
       if (!prototypeFactory.existsPrototype(streamEventId1))
          prototypeFactory.addEventPrototype("StreamEvent", streamEventId1);
       // Note that user does not need to directly add ChunkEvent prototype for corresponding stream.

       // Stream 2
       if (!prototypeFactory.existsPrototype(streamEventId2))
          prototypeFactory.addEventPrototype("StreamEvent", streamEventId2);
       // Note that user does not need to directly add ChunkEvent prototype for corresponding stream.
    }

    static void createConsumers() throws Exception
    {
       connection1 = new FabricConnectionFactory().createConnection();
       connection1.open();

       // Creates the asynchronous consumer for processing of StreamEvent for a first stream.
       connection1.createEventAsyncConsumer("Stream1Consumer", new Stream1Listener(), streamEventId1, null, EventScope.OBSERVABLE, true).start();

       // Creates the asynchronous consumer for processing of ChunkEvent for a first stream.
       connection1.createEventAsyncConsumer("Chunk1Consumer", new Chunk1Listener(), chunkEventId1, null, EventScope.OBSERVABLE, true).start();

       // Creates the asynchronous consumer for processing of StreamEvent for a second stream.
       connection1.createEventAsyncConsumer("Stream2Consumer", new Stream2Listener(), streamEventId2, null, EventScope.GLOBAL, true).start();

       // Creates the asynchronous consumer for processing of ChunkEvent for a second stream.
       connection1.createEventAsyncConsumer("Chunk2Consumer", new Chunk2Listener(), chunkEventId2, null, EventScope.GLOBAL, true).start();
    }

    static class Stream1Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             Utils.sleep(500);
             System.out.println("Confirmation: " + new String(((StreamEvent)event).getBytes()) + "\n\n");
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class Chunk1Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             System.out.print(new String(((ChunkEvent)event).getBytes()));
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class Stream2Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             Utils.sleep(500);
             System.out.println("\nConfirmation: " + new String(((StreamEvent)event).getBytes()) + "\n\n");
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class Chunk2Listener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             System.out.print(new String(((ChunkEvent)event).getBytes()));
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static void raiseStreamEvent() throws Exception
    {
       connection2 = new FabricConnectionFactory().createConnection();
       connection2.open();

       connection2.bindProducerFor(streamEventId1);
       // Note that user does not need to directly bind producer for event id of corresponding ChunkEvent.

       StreamEvent event = (StreamEvent)EventDatagramFactory.getInstance().createEvent(streamEventId1);
       event.writeBytes("This is the data of a first stream.".getBytes());
       event.setChunkSize(3);    // 3 bytes
       event.setChunkDelay(500); // 500 milliseconds

       // User should see that the text of event is displayed on the screen step-by-step.
       // Then a confirmation of this text will be displayed.
       connection2.raiseEvent(event, EventScope.INHERITED, -1);
    }
 }
