package sql;

import java.sql.Array;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Timestamp;
import java.util.List;

import com.streamscape.Trace;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.RuntimeContextException;
import com.streamscape.runtime.mf.admin.dfo.JDBCConfigurationFactory;
import com.streamscape.sdo.excp.SQLQueryException;
import com.streamscape.sdo.excp.SQLQueryParseException;
import com.streamscape.sdo.excp.SQLQueryValidationException;
import com.streamscape.sdo.rowset.RowSet;
import com.streamscape.sdo.rowset.RowSetFactory;
import com.streamscape.sdo.rowset.RowSetPrinter;
import com.streamscape.sdo.sql.SQLColumnSet;
import com.streamscape.sdo.sql.SQLQuery;
import com.streamscape.sdo.sql.SQLQueryFactory;
import com.streamscape.service.osf.jdbc.DatabaseConnection;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: SQLQuery samples. Illustrates oracle specific operations.
 * 
 * <p>Copyright: Copyright (c) 2012</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.4
 * @since 3.3
 */
 public class SQLQuerySamples
 {
    private static Connection connection;

    public static void main(String[] args) throws Exception
    {
       // Enables some traces.
       Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
       Trace.enable("*",                  Trace.Level.ERROR);

       System.setProperty(RuntimeContext.STARTUP_DIR, "./");
       System.setProperty(RuntimeContext.DEPLOYMENT, "./");

       try
       {
          // Initializes the Runtime Context.
          RuntimeContext.getInstance();

          // create connection to database
          initializeOracleConnection();

          // create SQLQuery objects and serialize it 
          generateSQLQuery("DROP TABLE DM.TEST_TABLE", "DropTabe");
          generateSQLQuery("CREATE OR REPLACE TYPE DM.XX_INTARRAY_TYPE IS VARRAY(3) OF INT", "CreateIntArrayType");
          generateSQLQuery("CREATE OR REPLACE TYPE DM.XX_CHARARRAY_TYPE IS VARRAY(3) OF CHAR(10)", "CreateCharArrayType");
          generateSQLQuery("CREATE OR REPLACE TYPE DM.XX_VARCHARARRAY_TYPE IS VARRAY(3) OF VARCHAR2(10)", "CreateVarCharArrayType");
          generateSQLQuery("CREATE TABLE DM.TEST_TABLE (" +
                "ID INT NOT NULL, " +
                "VARCHAR2COLUMN VARCHAR2(27) NOT NULL, " +
                "CHARCOLUMN CHAR(3)," +
                "INTARRAYCOLUMN DM.XX_INTARRAY_TYPE, " +
                "CHARARRAYCOLUMN DM.XX_CHARARRAY_TYPE, " +
                "VARCHARARRAYCOLUMN DM.XX_VARCHARARRAY_TYPE, " +
                "TIMESTAMPCOLUMN TIMESTAMP(6), " +
                "DATECOLUMN DATE, " +
                "BLOBCOLUMN BLOB, " +
                "CLOBCOLUMN CLOB, " +
                "BINARYCOLUMN RAW(100))",
                "CreateTestTable");

          // create types and arrays
          executeSQLQuery("DropTabe");
          executeSQLQuery("CreateIntArrayType");
          executeSQLQuery("CreateCharArrayType");
          executeSQLQuery("CreateVarCharArrayType");
          executeSQLQuery("CreateTestTable");

          ResultSet result = connection.getMetaData().getColumns(connection.getCatalog(), "DM", "TEST_TABLE", null);
          new RowSetPrinter().print(new RowSetFactory().createRowSet(result));

          // createFor methods samples
          createForMethods();

          // array samples
          arraySamples();

          // SQLColumnSet samples
          sqlColumnSetSamples();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
       }
       
       System.exit(0);
    }
    
    private static void generateSQLQuery(String sqlScript, String name) throws SQLQueryParseException, SQLQueryValidationException, RuntimeContextException, SQLQueryException
    {
       SQLQuery sqlQuery = SQLQueryFactory.createQuery(sqlScript);
       sqlQuery.setName(name);
       SQLQueryFactory.saveQueryObject("./", sqlQuery);
    }
    
    private static List<RowSet> executeSQLQuery(String queryName) throws Exception
    {
       SQLQuery sqlQuery = SQLQueryFactory.loadQueryObject("./", queryName); 
       List<RowSet> rowSets = SQLQueryFactory.execute(sqlQuery, 5, connection);
       
       System.out.println("Query '" + queryName + "': " + sqlQuery.getSqlScript());
       System.out.println("Results:");

       if (rowSets.isEmpty())
          System.out.println("\t empty result");

       for (RowSet rowSet : rowSets)
          new RowSetPrinter().print(rowSet);
       System.out.println();

       return rowSets;
    }

    private static void createForMethods() throws Exception
    {
       // create for insert
       SQLQuery sqlQuery = SQLQueryFactory.createForInsert(connection, "[DM].[TEST_TABLE]");
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 1);
       sqlQuery.setParameter("VARCHAR2COLUMN", "varchar2value");
       sqlQuery.setParameter("CHARCOLUMN", "abc");
       sqlQuery.setParameter("INTARRAYCOLUMN", "ARRAY[null, 2, \"3\"]");
       sqlQuery.setParameter("CHARARRAYCOLUMN", "ARRAY[null, \"tyu\", \"wf\"]");
       sqlQuery.setParameter("VARCHARARRAYCOLUMN", "ARRAY[null, \"sdaf\", \"sadf\"]");
       sqlQuery.setParameter("TIMESTAMPCOLUMN", new Timestamp(System.currentTimeMillis()).toString());
       sqlQuery.setParameter("DATECOLUMN", new Timestamp(System.currentTimeMillis()).toString());
       sqlQuery.setParameter("BLOBCOLUMN", "010203");
       sqlQuery.setParameter("CLOBCOLUMN", "clobvalue");
       sqlQuery.setParameter("BINARYCOLUMN", "020304");
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       sqlQuery.setParameter("ID", 2);
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       new RowSetPrinter().print(SQLQueryFactory.execute(SQLQueryFactory.createForSelect(connection, "DM.TEST_TABLE"), 0, connection).get(0));
       System.out.println();

       // create for update
       sqlQuery = SQLQueryFactory.createForUpdate(connection, "[DM].[TEST_TABLE]", new String[] {"INTARRAYCOLUMN", "CHARARRAYCOLUMN", "VARCHARARRAYCOLUMN"}, new String[] {"ID"});
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 1);
       sqlQuery.setParameter("INTARRAYCOLUMN", new Object[] {3, 4, "2"});
       sqlQuery.setParameter("CHARARRAYCOLUMN", "ARRAY[]");
       sqlQuery.setParameter("VARCHARARRAYCOLUMN", "ARRAY['12', '32', '56']");
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       new RowSetPrinter().print(SQLQueryFactory.execute(SQLQueryFactory.createForSelect(connection, "DM.TEST_TABLE"), 0, connection).get(0));
       System.out.println();
       
       // create for select 
       sqlQuery = SQLQueryFactory.createForSelect(connection, "[DM].[TEST_TABLE]", new String[] {"INTARRAYCOLUMN", "CHARARRAYCOLUMN", "VARCHARARRAYCOLUMN"}, new String[] {"ID"});
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 1);
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       System.out.println();
     
       // create for dynamic sql
       sqlQuery = SQLQueryFactory.createForDynamicSQL(connection, "UPDATE DM.TEST_TABLE SET CHARCOLUMN=?, CHARARRAYCOLUMN=?, BLOBCOLUMN=?, CLOBCOLUMN=?, BINARYCOLUMN=? WHERE ID=?");
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 1);
       sqlQuery.setParameter("CHARCOLUMN", "qwe");
       sqlQuery.setParameter("CHARARRAYCOLUMN", "ARRAY['1','2']");
       sqlQuery.setParameter("BLOBCOLUMN", "78F0");
       sqlQuery.setParameter("CLOBCOLUMN", "clob");
       sqlQuery.setParameter("BINARYCOLUMN", "ADFC");
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       new RowSetPrinter().print(SQLQueryFactory.execute(SQLQueryFactory.createForSelect(connection, "DM.TEST_TABLE"), 0, connection).get(0));
       System.out.println();
       
       // create for delete 
       sqlQuery = SQLQueryFactory.createForDelete(connection, "[DM].[TEST_TABLE]", new String[] {"ID"});
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 1);
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       new RowSetPrinter().print(SQLQueryFactory.execute(SQLQueryFactory.createForSelect(connection, "DM.TEST_TABLE"), 0, connection).get(0));
       System.out.println();
    }

    private static void arraySamples() throws Exception
    {
       // select arrays
       SQLQuery sqlQuery = SQLQueryFactory.createForSelect(connection, "[DM].[TEST_TABLE]", new String[] {"INTARRAYCOLUMN", "CHARARRAYCOLUMN", "VARCHARARRAYCOLUMN", "DATECOLUMN"}, new String[] {"ID"});
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 2);
       
       RowSet rowSet = SQLQueryFactory.execute(sqlQuery, 0, connection).get(0);
       new RowSetPrinter().print(rowSet.clone());
       
       rowSet.beforeFirst();
       rowSet.next();
       Array array = rowSet.getArray("CHARARRAYCOLUMN");
       Object [] values = (Object[])array.getArray();
       for(Object value : values)
          System.out.println(value);
       
       // get date
       System.out.println(rowSet.getDate("DATECOLUMN"));
       System.out.println();
       
       // update arrays
       sqlQuery = SQLQueryFactory.createForUpdate(connection, "[DM].[TEST_TABLE]", new String[] {"INTARRAYCOLUMN", "CHARARRAYCOLUMN", "VARCHARARRAYCOLUMN"}, new String[] {"ID"});
       System.out.println(sqlQuery.getSqlScript());
       sqlQuery.setParameter("ID", 2);
       sqlQuery.setParameter("INTARRAYCOLUMN", new Object[] {3, 4, "2"});
       sqlQuery.setParameter("CHARARRAYCOLUMN", "ARRAY[]");
       sqlQuery.setParameter("VARCHARARRAYCOLUMN", "ARRAY['12', '3asd2', 'dsa56']");
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       new RowSetPrinter().print(SQLQueryFactory.execute(SQLQueryFactory.createForSelect(connection, "DM.TEST_TABLE"), 0, connection).get(0));
       System.out.println();
    }

    private static void sqlColumnSetSamples() throws Exception
    {
       // perform select
       SQLQuery sqlQuery = SQLQueryFactory.createForSelect(connection, "[DM].[TEST_TABLE]", new String[] {"VARCHAR2COLUMN", "CHARCOLUMN", "CLOBCOLUMN"});
       System.out.println(sqlQuery.getSqlScript());
       RowSet rowSet = SQLQueryFactory.execute(sqlQuery, 0, connection).get(0);
       new RowSetPrinter().print(rowSet.clone());
       
       // create column set from row set
       SQLColumnSet columnSet = SQLQueryFactory.createColumnSet(rowSet);
       
       // create sql update script bases on column set
       sqlQuery = SQLQueryFactory.createForUpdate("[DM].[TEST_TABLE]", columnSet);
       sqlQuery.setParameter("VARCHAR2COLUMN", "qwe");
       new RowSetPrinter().print(SQLQueryFactory.execute(sqlQuery, 0, connection).get(0));
       new RowSetPrinter().print(SQLQueryFactory.execute(SQLQueryFactory.createForSelect(connection, "DM.TEST_TABLE"), 0, connection).get(0));
    }

    private static void initializeOracleConnection() throws Exception
    {
       DatabaseConnection databaseConnection = JDBCConfigurationFactory.createConnection("OracleServer", "prototype");
       databaseConnection.connect();
       Thread.sleep(1000);
       connection = databaseConnection;
    }
 }
