package service.accessor;

import com.streamscape.Trace;
import com.streamscape.cli.ClientContext;
import com.streamscape.cli.service.ServiceAccessor;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.sdo.enums.RequestDistributionStrategy;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.ServiceFrameworkException;
import com.streamscape.sef.moderator.ModeratorUtils;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Client of the 'client/reliable' sample.
 *
 * <p>Copyright: Copyright (c) 2014</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class Client
 {
    private static FabricConnection connection;
    private static ServiceAccessor  accessor;

    private static final String REQUEST_EVENT_ID = "event.request";
    private static final String REPLY_EVENT_ID   = "event.reply";

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("*", Trace.Level.ERROR);

          // Initializes the Client Context.
          ClientContext context = ClientContext.getInstance();

          System.out.println("\nSetup environment...\n");
          setupEnvironment();

          System.out.println("\nCreate accessor...\n");
          createAccessor();

          System.out.println("\nInvoke direct request...\n");
          invokeDirectRequest();

          System.out.println("\nInvoke non-direct request...\n");
          invokeNonDirectRequest();

          System.out.println("\nInvoke async request...\n");
          invokeAsyncRequest();

          System.out.println("Done.");
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    private static void setupEnvironment() throws Exception
    {
       connection = new FabricConnectionFactory("tlp://localhost:5000").createConnection("User1", "123");
       connection.setName("Client1");
       connection.open();

       // Imports the event prototypes from the node.
       connection.importEventPrototype(REQUEST_EVENT_ID);
       connection.importEventPrototype(REPLY_EVENT_ID);
    }

    private static void createAccessor() throws Exception
    {
       // Creates an accessor to the SampleService.Default service.
       // Since a node name is not specified, the accessor will search a suitable service in all nodes
       // starting with a local node (Sample.Node1 in this sample).
       accessor = connection.createServiceAccessor("SampleService", "Default");

       System.out.println("Accessor created. " + "Accessed service is '" +
                          ModeratorUtils.makeComponentFullName(accessor.getActualNodeName(), accessor.getComponentType(),
                                                               accessor.getComponentName() + "'."));
    }

    private static void invokeDirectRequest() throws Exception
    {
       // Directly invokes the 'RequestHandler' event handler of the accessed service (Service1 in this sample).
       try
       {
          // If an event selector is set for the specified event handler, an event is required for the service request
          // So, this method will throw an exception
          accessor.invokeServiceRequest("RequestHandler", "DirectRequest");
       }
       catch (ServiceFrameworkException exception)
       {
          Trace.logException(Client.class, exception, false);
       }
    }

    private static void invokeNonDirectRequest() throws Exception
    {
       System.out.print("Request with version '1.0'... ");

       TextEvent request = (TextEvent)EventDatagramFactory.getInstance().createEvent(REQUEST_EVENT_ID);
       request.setText("NonDirectRequest");
       request.setEventStringProperty("version", "1.0");

       // Sends the request to all suitable services in all nodes (Service1 and Service2 in this sample).
       // This method takes into account the event selectors that are set for the event handlers.
       // So, this request will be received only by the Service1 in the Sample.Node1 node.
       TextEvent reply = (TextEvent)accessor.invokeServiceRequest(request, RequestDistributionStrategy.AUCTION);
       System.out.println("Reply received from service '" + reply.getText() + "'.");

       System.out.print("Request with version '2.0'... ");

       request = (TextEvent)EventDatagramFactory.getInstance().createEvent(REQUEST_EVENT_ID);
       request.setText("NonDirectRequest");
       request.setEventStringProperty("version", "2.0");

       // This request will be received only by the Service2 in the Sample.Node2 node.
       reply = (TextEvent)accessor.invokeServiceRequest(request, RequestDistributionStrategy.AUCTION);
       System.out.println("Reply received from service '" + reply.getText() + "'.");
    }

    private static void invokeAsyncRequest() throws Exception
    {
       TextEvent request = (TextEvent)EventDatagramFactory.getInstance().createEvent(REQUEST_EVENT_ID);
       request.setText("AsyncRequest");
       request.setEventStringProperty("version", "1.0");

       // Sends the request to all suitable services in all nodes, but does not receive a response.
       // This method takes into account the event selectors that are set for the event handlers.
       // So, this request will be received only by the Service1 in the Sample.Node1 node.
       accessor.invokeAsyncServiceRequest(request);

       request = (TextEvent)EventDatagramFactory.getInstance().createEvent(REQUEST_EVENT_ID);
       request.setText("AsyncRequest");
       request.setEventStringProperty("version", "2.0");

       // This request will be received only by the Service2 in the Sample.Node2 node.
       accessor.invokeAsyncServiceRequest(request);
    }
 }
