package scheduler;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationFactory;
import com.streamscape.runtime.mf.admin.sco.ServiceConfigurationObject;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.RowSetEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sdo.rowset.RowSetPrinter;
import com.streamscape.sef.enums.ComponentState;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.scheduler.EventTaskTrigger;
import com.streamscape.sef.scheduler.RepeatingJob;
import com.streamscape.sef.scheduler.Scheduler;
import com.streamscape.sef.scheduler.TaskList;
import com.streamscape.sef.service.ServiceDescriptor;
import com.streamscape.sef.service.ServiceManager;
import com.streamscape.sef.utils.Utils;
import com.streamscape.service.mail.sender.MailSender;
import com.streamscape.service.mail.sender.SCOProperties;
import com.streamscape.service.osf.config.PasswordPropertyValue;
import com.streamscape.service.osf.config.ServiceConfigurationProperty;
import com.streamscape.service.osf.config.ServicePropertyType;
import com.streamscape.service.osf.enums.InvokeMode;
import com.streamscape.service.osf.evh.EventHandler;

import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: <i>Scheduler</i> sample.
 *
 * <p>Usage example: SchedulerSample user@gmail.com qwerty
 *
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class SchedulerSample
 {
    static RuntimeContext context;

    public static void main(String[] args)
    {
       try
       {
          if (args.length != 2)
             throw new Exception("Invalid parameters.\n" +
                                 "Usage: SchedulerSample <email_user> <email_password>");

          // Enables some traces.
          Trace.enable("*",                         Trace.Level.ERROR);
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, ".");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, ".");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("Add event prototypes...\n");
          addPrototypes();

          System.out.println("Setup MailSender service...\n");
          setUpMailSenderService(args[0], args[1]); // It is just an auxiliary method.

          System.out.println("Create event task triggers...\n");
          createEventTaskTriggers(args[0]);

          System.out.println("Create scheduler job...\n");
          createScheduledJob();

          // Waits for completion.
          Utils.sleep(20000);

          // You should see three tables with result of 'list nodes' execution and should receive three emails.
          context.shutdown();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    private static final String SLANG_EVENT_ID = "event.slang";
    private static final String MAIL_EVENT_ID  = "event.mail";

    static void addPrototypes() throws Exception
    {
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();
       if (!prototypeFactory.existsPrototype(SLANG_EVENT_ID))
          prototypeFactory.addEventPrototype("RowSetEvent", SLANG_EVENT_ID);
       if (!prototypeFactory.existsPrototype(MAIL_EVENT_ID))
          prototypeFactory.addEventPrototype("MailEvent", MAIL_EVENT_ID);
    }

    private static final String SERVICE_TYPE      = "MailSender";
    private static final String SERVICE_NAME      = "Default";
    private static final String SERVICE_FULL_NAME = SERVICE_TYPE + "." + SERVICE_NAME;

    static void setUpMailSenderService(String username, String password) throws Exception
    {
       ServiceManager serviceManager = context.getServiceManager();
       if (!serviceManager.isServiceRegistered(SERVICE_FULL_NAME))
       {
          ServiceConfigurationObject sco = ServiceConfigurationFactory.createServiceConfiguration(context, SERVICE_NAME, SERVICE_TYPE, false);

          sco.setServiceClassName(MailSender.class.getName());
          sco.setInvokeMode(InvokeMode.ASYNC);

          EventHandler handler = new EventHandler(sco, "sendMailFromEvent", "SendMailHandler");
          handler.bindRequestEvent(EventDatagramFactory.getInstance().createEvent(MAIL_EVENT_ID));
          sco.addEventHandler(handler);

          ServiceConfigurationProperty property = sco.createProperty(SCOProperties.SMTP_SERVER_NAME, ServicePropertyType.STRING, null);
          property.setValue("smtp.gmail.com");
          sco.addProperty(property);

          property = sco.createProperty(SCOProperties.SMTP_SERVER_PORT, ServicePropertyType.STRING, null);
          property.setValue("587");
          sco.addProperty(property);

          property = sco.createProperty(SCOProperties.SMTP_SERVER_USER, ServicePropertyType.STRING, null);
          property.setValue(username);
          sco.addProperty(property);

          property = sco.createProperty(SCOProperties.SMTP_SERVER_PASSWD, ServicePropertyType.PASSWORD, null);
          property.setValue(new PasswordPropertyValue(password));
          sco.addProperty(property);

          property = sco.createProperty(SCOProperties.START_TLS_REQUIRED, ServicePropertyType.BOOLEAN, null);
          property.setValue(true);
          sco.addProperty(property);

          property = sco.createProperty(SCOProperties.SSL_REQUIRED, ServicePropertyType.BOOLEAN, null);
          property.setValue(false);
          sco.addProperty(property);

          property = sco.createProperty(SCOProperties.REPLY_TO_ADDRESS, ServicePropertyType.STRING, null);
          property.setValue(username);
          sco.addProperty(property);

          ServiceConfigurationFactory.saveConfigurationObject(sco);

          ServiceDescriptor service = new ServiceDescriptor(SERVICE_TYPE, SERVICE_NAME);
          serviceManager.registerService(service);
          serviceManager.runServiceAs(SERVICE_FULL_NAME, "user1", "123");
       }
       else
       {
          if (serviceManager.getServiceState(SERVICE_FULL_NAME) == ComponentState.STARTED)
             serviceManager.stopService(SERVICE_FULL_NAME);

          ServiceConfigurationObject sco = ServiceConfigurationFactory.loadConfigurationObject(context, SERVICE_NAME, SERVICE_TYPE, false);
          sco.getProperty(SCOProperties.SMTP_SERVER_USER).setValue(username);
          sco.getProperty(SCOProperties.SMTP_SERVER_PASSWD).setValue(new PasswordPropertyValue(password));
          sco.getProperty(SCOProperties.REPLY_TO_ADDRESS).setValue(username);
          ServiceConfigurationFactory.saveConfigurationObject(sco);
       }

       if (serviceManager.getServiceState(SERVICE_FULL_NAME) != ComponentState.STARTED)
          serviceManager.startService(SERVICE_FULL_NAME);
    }

    static void createEventTaskTriggers(String emailAddress) throws Exception
    {
       Scheduler scheduler = context.getScheduler();

       // Creates a first trigger for execution of the SLANG command.
       EventTaskTrigger slangTrigger;
       if (!scheduler.existsTrigger("SlangTrigger"))
       {
          slangTrigger = scheduler.createTrigger("SlangTrigger",
                                                 "{" +
                                                 "   RowSet rs = sys.query('list nodes');" +
                                                 "   RowSetEvent slangEvent = new [" + SLANG_EVENT_ID + "];" +
                                                 "   slangEvent.init(rs);" +
                                                 "   raise event slangEvent;" +
                                                 "}");
          slangTrigger.setTaskName("SlangQuery");
          slangTrigger.enable();
       }

       // Creates a auxiliary listener to print result of the SLANG command execution.
       createResultListener();

       // Creates a second trigger to send email after execution of the SLANG command.
       EventTaskTrigger mailTrigger;
       if (!scheduler.existsTrigger("MailTrigger"))
       {
          mailTrigger = scheduler.createTrigger("MailTrigger",
                                                "{" +
                                                "   Event taskEvent = this.getCurrentEvent();" +
                                                "   String subject = 'Scheduler sample.';" +
                                                "   String body = 'Scheduled task &apos;' + taskEvent.taskName + '&apos; executed successfully.';" +
                                                "   String recipients = '" + emailAddress + "';" +
                                                "   Event mailEvent = events.createMailEvent('" + MAIL_EVENT_ID + "', subject, body, null, " +
                                                "recipients, null, null);" +
                                                "   raise event mailEvent;" +
                                                "}");
          mailTrigger.setTaskName("MailPublisher");
          mailTrigger.enable();
       }
    }

    private static void createResultListener() throws Exception
    {
       FabricConnection connection = new FabricConnectionFactory().createConnection();
       connection.open(); // This connection will be closed at node shutdown.
       connection.createEventAsyncConsumer("ResultListener", event ->
       {
          try
          {
             new RowSetPrinter().print(((RowSetEvent)event).getRowSet());
             System.out.println();
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }, SLANG_EVENT_ID, null, EventScope.OBSERVABLE, true).start();
    }

    static void createScheduledJob() throws Exception
    {
       Scheduler scheduler = context.getScheduler();

       // Removes old task list.
       scheduler.dropTaskList("SampleTaskList");

       // Creates a task list with two tasks.
       TaskList taskList = scheduler.createTaskList("SampleTaskList");

       taskList.addEventTask("SlangQuery");    // Task for execution of SLANG command.
       taskList.addEventTask("MailPublisher"); // Task for sending email.
       scheduler.updateTaskList(taskList);

       scheduler.enableTaskList("SampleTaskList"); // Task list is ready for execution.

       // Removes old job.
       scheduler.dropJob("SampleJob");

       // Creates a repeating job with two tasks and three repeats.
       RepeatingJob job = scheduler.createRepeatingJob("SampleJob");

       job.setStartTime(new Date(System.currentTimeMillis() + 1000)); // 1 second.
       job.setRepeatInterval(5);
       job.setRepeatIntervalUnit(TimeUnit.SECONDS);
       job.setRepeatCount(3);
       scheduler.updateJob(job);

       scheduler.setJobTaskList("SampleJob", "SampleTaskList");

       scheduler.enableJob("SampleJob"); // Job schedules its task list for execution.
    }
 }
