package http;

import com.streamscape.lib.http.HTTPClient;
import com.streamscape.lib.http.HTTPClientResponse;
import com.streamscape.lib.http.NVPair;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: OAuth2 samples. Demonstrates Username-Password flow.
 * 
 * <p>Copyright: Copyright (c) 2012</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Nikita Kutuzov
 * @version 3.3
 */
 public class OAuthSample
 {
    public static void main(String[] arguments) throws Exception
    {
       // create HTTP Client
       HTTPClient httpClient = new HTTPClient("localhost", 9999);
       httpClient.setAllowUserInteraction(false);
       httpClient.setContext(new Object());

       System.out.println("Requesting access token...");

       // token request
       NVPair [] formParams = new NVPair [] {
             new NVPair("grant_type", "password"),
             new NVPair("client_id", "restclient"),
             new NVPair("username", "http"),
             new NVPair("password", "http")
          }; 

       HTTPClientResponse response = httpClient.Post("/security/oauth2/token", formParams);

       String accessToken = null;
       if (response.getStatusCode() == 200)
       {
          // response is ok
          String tokenObject = new String(response.getData());
          System.out.println("Access token successfuly recieved:\n " + tokenObject);
          
          // response example:
//          {
//             "access_token" : "18f9dbcdd881c53b6b39895557a808", 
//             "expires_in" : 60, 
//             "token_type" : "Bearer"
//           }
          
          Pattern access_token_pattern =
                Pattern.compile("(?s).*\"access_token\"\\s*:\\s*\"([^\"]*)\".*");
          Matcher matcher = access_token_pattern.matcher(tokenObject);
          
          if (matcher.find() && matcher.group().equals(tokenObject))
             accessToken = matcher.group(1);

          // using access token
          System.out.println("Trying to use access token...");

          httpClient.setDefaultHeaders(new NVPair[] {new NVPair("Authorization", "Bearer " + accessToken)});
          httpClient.enableOAuthAuthentication();

          response = httpClient.Get("/exchange/raiseEvent?responseFormat=json&requestFormat=json&eventId=test.event.id&data={\"@type\":\"TextEvent\",\"data\":\"Text Value\"}");
          

          if (response.getStatusCode() == 401)
          {
             // access_token is expired or invalid
             // error description in header
             // error example
             // www-authenticate: Bearer realm="Exchange", error="invalid_token", error_description="The access token is expired or invalid." 

             System.out.println("Access token authentication failed:");
             System.out.println(response.getHeader("WWW-Authenticate"));
          }
          else
          {
             // access_token is valid and authorization is OK 
             System.out.println("Access token authentication OK");
          }
       }
       else if (response.getStatusCode() == 400)
       {
          // access token error 

          // error example
//          {
//             "error" : "invalid_client", 
//             "description" : "Authentication failed for user: http"
//          }
          System.out.println("Access token request failed:");
          System.out.println(new String(response.getData()));
       }

       // revoke token
       if (accessToken != null)
       {
          System.out.println("Revoke access token...");

          formParams = new NVPair [] {
                new NVPair("token", accessToken),
                new NVPair("client_id", "restclient"),
          }; 
          response = httpClient.Post("/security/oauth2/revoke", formParams);

          if (response.getStatusCode() == 200)
          {
             // revoke is ok
             System.out.println("Revoke access token successfull");
          } 
          else if (response.getStatusCode() == 400)
          {
             // revoke failed
             System.out.println("Revoke access failed:");
             System.out.println(new String(response.getData()));
          }
       }
    }
 }
