package group;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.timer.*;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.SecurityViolationException;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sef.*;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.group.FabricGroup;
import com.streamscape.sef.group.FabricGroupLink;
import com.streamscape.sef.utils.Utils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Random;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: First <i>Group</i> sample.
 *
 * <p>This sample initializes the <code>Sample.Node1</code> node and raises several events within the specific group.
 * At the same time the another <code>Sample.Node2</code> node also raises several events within the same group.
 * Nodes raise events simultaneously with a random delay between successive events (it is needed to shuffle these events).
 * Also each node has the async consumer that receives events from both nodes within the group.
 * Order of events received by each consumer must be identical.
 *
 * <p>Copyright: Copyright (c) 2013</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class GroupSample1
 {
    static RuntimeContext   context;
    static FabricGroup      group;
    static FabricConnection connection1; // It is used for consumers.
    static FabricGroupLink  link1;       // It is used for consumers.
    static FabricConnection connection2; // It is used for event sending.
    static FabricGroupLink  link2;       // It is used for event sending.

    static TestListener     listener;
    static final String     eventId = "event.text";

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("events.*",                  Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node1");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node1");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("\nCreate group...\n");
          createGroup();

          System.out.println("\nCreate connections...\n");
          createConnections();

          System.out.println("\nCreate consumers...");
          createConsumers();

          // Creates the request consumer to synchronize events raising in the both nodes.
          connection1.createRequestConsumer("Synchronizer", new RaiseEventsSynchronizer(), EventScope.GLOBAL);
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void createGroup() throws Exception
    {
       group = context.getGroupManager().createGroup("TestGroup", "Test group");
       System.out.println("   Group 'TestGroup created.'");
       System.out.println("   Group has members: " + group.listMembers());
    }

    static void createConnections() throws Exception
    {
       FabricConnectionFactory connectionFactory = new FabricConnectionFactory();

       connection1 = connectionFactory.createConnection();
       connection1.setName("Sample1Connection1");
       connection1.open();

       link1 = connection1.joinGroup("TestGroup");
       System.out.println("   Group has members: " + group.listMembers());

       connection2 = connectionFactory.createConnection();
       connection2.setName("Sample1Connection2");
       connection2.open();

       link2 = connection2.joinGroup("TestGroup");
       System.out.println("   Group has members: " + group.listMembers());
    }

    static void createConsumers() throws Exception
    {
       // Adds event prototype.
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();
       if (!prototypeFactory.existsPrototype(eventId))
          prototypeFactory.addEventPrototype("TextEvent", eventId);

       // Creates the async consumer in the TestGroup group.
       listener = new TestListener();
       link1.createEventAsyncConsumer("Consumer", listener, eventId, "", EventScope.GLOBAL, true).start();
    }

    static void raiseEvents()
    {
       System.out.println("\nRaise events...\n");

       try
       {
          connection2.bindProducerFor(eventId);

          raiseEvent("Node1.Event1");
          raiseEvent("Node1.Event2");
          raiseEvent("Node1.Event3");
          raiseEvent("Node1.Event4");
          raiseEvent("Node1.Event5");
          raiseEvent("Node1.Event6");
          raiseEvent("Node1.Event7");
          raiseEvent("Node1.Event8");
          raiseEvent("Node1.Event9");
          raiseEvent("Node1.Event10");
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
       }
    }

    static void raiseEvent(String text) throws Exception
    {
       Utils.sleep(new Random(System.currentTimeMillis()).nextInt(20) + 1);

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(eventId);
       textEvent.setText(text);

       // FabricGroupLink has the same interface for events raising as the FabricConnection.
       link2.raiseEvent(textEvent, EventScope.GLOBAL, -1);
       System.out.println("   Event [" + text + "] raised.");
       Utils.sleep(new Random(System.currentTimeMillis()).nextInt(20) + 1);
    }

    static void printResults()
    {
       // An order of the received events must be identical for all consumers of the group in all nodes.
       System.out.println();
       for (String text : listener.texts)
          System.out.println("   Event [" + text + "] received.");
    }

    static class TestListener implements FabricEventListener
    {
       private final List<String> texts = new ArrayList<String>();

       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             texts.add(((TextEvent)event).getText());
          }
          catch (SecurityViolationException exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    static class RaiseEventsSynchronizer implements FabricRequestListener
    {
       public ImmutableEventDatagram onRequest(ImmutableEventDatagram event) throws FabricRequestException
       {
          try
          {
             Date startTime = new Date(event.getTimestamp().getTime() + 1000); // Timer will start in 1 second.
             FabricTimerManager.getInstance().createTimer("Default", "SyncTimer", new RaiseEventsTask(), 1, 1).start(startTime);
          }
          catch (FabricTimerException exception)
          {
             exception.printStackTrace();
             throw new FabricRequestException(exception);
          }
          return null;
       }
    }

    static class RaiseEventsTask extends AbstractFabricTimerTask
    {
       public void execute(FabricTimer timer)
       {
          raiseEvents();
          Utils.sleep(2000);
          printResults();
       }
    }
 }
