package events;

import java.io.File;
import java.io.FileOutputStream;

import com.streamscape.Trace;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.omf.xml.XSerializer;
import com.streamscape.omf.xml.XSerializerFactory;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.operation.security.ListUsersOperation;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.enums.ExceptionStrategy;
import com.streamscape.sdo.enums.ReplyMatchStrategy;
import com.streamscape.sdo.enums.RequestDistributionStrategy;
import com.streamscape.sdo.event.AcknowledgementEvent;
import com.streamscape.sdo.event.DataEvent;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sdo.rowset.RowSetPrinter;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.moderator.Moderator;
import com.streamscape.sef.network.tlp.acceptor.TLPAcceptor;
import com.streamscape.sef.utils.Utils;
import com.streamscape.slex.slang.SLSession;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: Second <i>Runtime</i> sample.
 *
 * <p>This sample initializes the <code>Sample.Node2</code> node and raises events to the <code>Sample.Node1</code> node.
 * <br>The sample uses one embedded <code>FabricConnection</code> for event sending.
 *
 * <p>Copyright: Copyright (c) 2011</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.2
 */
 public class RuntimeSample2
 {
    static RuntimeContext   context;
    static FabricConnection connection;

    static final String     textEventId1       = "event.text1";
    static final String     textEventId2       = "event.text2";
    static final String     dataEventId        = "event.data";
    static final String     ackEventId         = "event.ack";
    static final String     directErrorEventId = "event.error.direct";
    static final String     asyncErrorEventId  = "event.error.async";
    static final String     receiverEventId    = "event.receiver";

    public static void main(String[] args)
    {
       try
       {
          // This auxiliary method is required only for automatic creation of TLP acceptor with non-default port (to avoid a manual change).
          prepareRepository();

          // Enables some traces.
          Trace.enable("com.streamscape.runtime.*", Trace.Level.INFO);
          Trace.enable("com.streamscape.sef.*",     Trace.Level.INFO);
          Trace.enable("events.*",                  Trace.Level.ERROR);

          // Sets a startup directory of the Runtime.
          System.setProperty(RuntimeContext.STARTUP_DIR, "Sample.Node2");
          // Sets a path to the directory containing a deployment descriptor archive (stdeploy.jar).
          System.setProperty(RuntimeContext.DEPLOYMENT, "Sample.Node2");
          // Sets a path to the to file containing discovery rules for the Fabric.
          System.setProperty(RuntimeContext.DISCOVERY_FABRIC_DIRECTORY, System.getProperty(RuntimeContext.STARTUP_DIR) + "/../DirectoryTable.xdo");

          // Initializes the Runtime Context.
          context = RuntimeContext.getInstance();

          System.out.println("\nCreate connections...\n");
          createConnection();

          System.out.println("\nUse moderator...\n");
          useModerator();

          System.out.println("\nRaise events...\n");
          raiseEvents();

          System.out.println("\nRaise requests...\n");
          raiseRequests();

          System.out.println("\nInvoke SLANG requests...\n");
          invokeSlangRequests();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void createConnection() throws Exception
    {
       // This connection will work under the Runtime user (specified in the Deployment Descriptor).
       connection = new FabricConnectionFactory().createConnection();
       connection.open(); // Now the connection can perform various operations.
    }

    static void useModerator()
    {
       // Moderator can access components and consumers with GLOBAL event scope contained in other nodes of the Fabric:
       System.out.println("   Component: " + context.getModerator().lookupComponent("Sample.Node1://Client_TLP.NodeConnection"));
       System.out.println("   " + context.getModerator().lookupRequestConsumer("Sample.Node1://Client_TLP.NodeConnection:RequestConsumer2"));
    }

    static void raiseEvents() throws Exception
    {
       // To have possibility of raising an event, connection should bind the corresponding event id to itself.
       // After a connection will bind any event id, this connection becomes event producer.
       // Pay attention that an event id must be bound to a connection only once.
       connection.bindProducerFor(textEventId1);
       connection.bindProducerFor(textEventId2);
       connection.bindProducerFor(dataEventId);
       connection.bindProducerFor(ackEventId);
       connection.bindProducerFor(directErrorEventId);
       connection.bindProducerFor(asyncErrorEventId);
       connection.bindProducerFor(receiverEventId);

       System.out.println("   Raising text event...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId1);
       textEvent.setText("event");
       // This event will get to TextEventListener and AllEventsListener.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       // Note that an event instance can be raised only once.

       System.out.println("   Raising data event 1...");

       DataEvent dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("sss", 1));
       // This event will get to DataEventListener1 (but not to DataEventListener2 due to selector) and AllEventsListener.
       connection.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising data event 2...");

       dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("sss", 2));
       dataEvent.setEventDoubleProperty("doubleProperty", 1.1);
       // This event will get to DataEventListener2 (but not to DataEventListener1 due to selector) and AllEventsListener.
       connection.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising data event 3...");

       dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("zzz", 1));
       dataEvent.setEventDoubleProperty("doubleProperty", 1.1);
       // This event will get to AllEventsListener only.
       connection.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising direct error event 1...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(directErrorEventId);
       textEvent.setText("error");

       try
       {
          // This event will get to DirectEventListenerWithException and perhaps AllEventsListener (it is order-dependent).
          connection.raiseEvent(textEvent, EventScope.INHERITED, -1);
       }
       catch (FabricEventException exception)
       {
          Trace.logException(RuntimeSample2.class, exception, false);
       }

       // All FabricEventExceptions thrown by direct consumers will be ignored.
       connection.setExceptionStrategy(ExceptionStrategy.IGNORE);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising direct error event 2...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(directErrorEventId);
       textEvent.setText("error");
       // This event will get to DirectEventListenerWithException and AllEventsListener. Exception will be ignored.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising async error event...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(asyncErrorEventId);
       textEvent.setText("error");
       // This event will get to AsyncEventListenerWithException and AllEventsListener. Exception will be ignored.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising receiver event 1...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(receiverEventId);
       textEvent.setText("event1");
       // This event will get to AllEventsListener and receiver.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising receiver event 2...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(receiverEventId);
       textEvent.setText("event2");
       // This event will get to AllEventsListener and receiver.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
    }

    static void raiseRequests() throws Exception
    {
       System.out.println("   Raising first non-direct request...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request1");
       textEvent.setReplyTo(ackEventId); // Acknowledgement event will have the specified event id.

       // This event will get to EventListenerWithAck (which will raise acknowledgement) and AllEventsListener.
       // The current thread will wait for acknowledgement for 1 second.
       AcknowledgementEvent ack = connection.raiseRequest(textEvent, EventScope.INHERITED, RequestDistributionStrategy.AUCTION,
                                                           ReplyMatchStrategy.REPLY_TO, 5000);
       System.out.println("      Acknowledgement: " + ack.getEventId());

       System.out.println("   Raising second non-direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request2");
       // Since 'ReplyTo' is not set, an acknowledgement event will have an auto-generated event id.

       // This event will get to EventListenerWithAck (which will raise acknowledgement) and AllEventsListener.
       // The current thread will wait for acknowledgement for 2 seconds.
       ack = connection.raiseRequest(textEvent, EventScope.INHERITED, RequestDistributionStrategy.AUCTION, ReplyMatchStrategy.REPLY_TO, 2000);
       System.out.println("      Acknowledgement: " + ack.getEventId());

       Moderator moderator = connection.getModerator();

       System.out.println("   Raising first direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId1);
       textEvent.setText("request3");

       // This event will get directly to RequestConsumer2 located in the Sample.Node1 node.
       // The current thread will wait for reply for 1 seconds.
       ImmutableEventDatagram reply = connection.raiseRequest(moderator.lookupRequestConsumer("Sample.Node1://Client_TLP.NodeConnection:RequestConsumer2"),
                                                              textEvent, 1000);
       System.out.println("      Reply: " + reply.getEventId());
    }

    static void invokeSlangRequests() throws Exception
    {
       // Creates Semantic Language session to the Sample.Node1 node.
       SLSession session = connection.createSLSession("Sample.Node1");

       System.out.println("   First SLANG request...\n");

       // Invokes a request using a plain text command.
       SLResponse response = session.slangRequest("list users", 1000);
       // Result data is contained in the RowSet.
       new RowSetPrinter().print(response.getRowSet());

       System.out.println("\n   Second SLANG request...\n");

       // Invokes a request using a special SLStatement object.
       response = session.slangRequest(new ListUsersOperation.Definition(), 1000);
       new RowSetPrinter().print(response.getRowSet());
    }

    static void prepareRepository() throws Exception
    {
       File nodeDir = new File("Sample.Node2");
       if (!(new File(nodeDir, ".tfcache").exists()))
       {
          XSerializer serializer = XSerializerFactory.getInstance().getDefaultSerializer();

          Utils.createTFCache(nodeDir);
          createAcceptor(nodeDir, serializer);

          XSerializerFactory.getInstance().destroy();
       }
    }

    static void createAcceptor(File nodeDir, XSerializer serializer) throws Exception
    {
       TLPAcceptor acceptor = new TLPAcceptor("Default");
       acceptor.setURL("tlp://127.0.0.1:5001");

       byte[] bytes = serializer.serialize(acceptor).getBytes();
       FileOutputStream stream = new FileOutputStream(new File(Utils.createDirectory(new File(nodeDir, ".tfcache/objects/sys/network/acceptors/tlp")),
                                                               "TLPAcceptor.Default.xdo"));
       stream.write(bytes);
       stream.close();
    }
 }
