package events;

import com.streamscape.Trace;
import com.streamscape.cli.ClientContext;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.concurrent.FabricThreadManager;
import com.streamscape.runtime.mf.operation.security.ListUsersOperation;
import com.streamscape.sdo.EventDatagram;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.enums.AcknowledgeAction;
import com.streamscape.sdo.enums.ExceptionStrategy;
import com.streamscape.sdo.enums.ReplyMatchStrategy;
import com.streamscape.sdo.enums.RequestDistributionStrategy;
import com.streamscape.sdo.event.AcknowledgementEvent;
import com.streamscape.sdo.event.DataEvent;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sef.FabricRequestException;
import com.streamscape.sdo.mf.admin.DatagramPrototypeFactory;
import com.streamscape.sdo.mf.admin.SemanticTypeFactory;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sdo.rowset.RowSetPrinter;
import com.streamscape.sef.*;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.moderator.*;
import com.streamscape.sef.utils.Utils;
import com.streamscape.slex.slang.SLSession;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: First <i>Client</i> sample.
 *                 <p>This sample creates the client connected to <code>Sample.Node1</code> node
 *                 and makes some typical operations within this node.
 *                 Also this node can process events received from the second sample client.
 *
 * <p>Copyright: Copyright (c) 2011</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.2
 */
 public class ClientSample1
 {
    static ClientContext    context;
    static FabricConnection connection;

    static final String     textEventId1       = "event.text1";
    static final String     textEventId2       = "event.text2";
    static final String     dataEventId        = "event.data";
    static final String     ackEventId         = "event.ack";
    static final String     directErrorEventId = "event.error.direct";
    static final String     asyncErrorEventId  = "event.error.async";
    static final String     receiverEventId    = "event.receiver";

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("com.streamscape.cli.*", Trace.Level.INFO);
          Trace.enable("events.*",              Trace.Level.ERROR);

          // Initializes the Client Context.
          context = ClientContext.getInstance();

          System.out.println("\nCreate connections...\n");
          createConnection();

          System.out.println("\nAdd prototypes...\n");
          addPrototypes();

          System.out.println("\nCreate consumers...\n");
          createConsumers();

          System.out.println("\nUse moderator...\n");
          useModerator();

          System.out.println("\nRaise events...\n");
          raiseEvents();

          System.out.println("\nRaise requests...\n");
          raiseRequests();

          System.out.println("\nInvoke SLANG requests...\n");
          invokeSlangRequests();

          System.out.println("\nUse receiver...\n");
          useReceiver();

          System.out.println();
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    static void createConnection() throws Exception
    {
       // This client will connect to Sample.Node1 node.
       // This connection will work under the user 'user1'.
       connection = new FabricConnectionFactory("tlp://127.0.0.1:5000").createConnection("user1", "123");
       connection.setName("ClientConnection");
       connection.open();
    }

    static void addPrototypes() throws Exception
    {
       DatagramPrototypeFactory prototypeFactory = context.getDatagramPrototypeFactory();

       // TextEvent prototypes.
       if (!prototypeFactory.existsPrototype(textEventId1))
          prototypeFactory.addEventPrototype("TextEvent", textEventId1);
       if (!prototypeFactory.existsPrototype(textEventId2))
          prototypeFactory.addEventPrototype("TextEvent", textEventId2);

       // DataEvent prototype with three properties.
       if (!prototypeFactory.existsPrototype(dataEventId))
       {
          // Creates a semantic type for data class of the prototype.
          SemanticTypeFactory typeFactory = context.getSemanticTypeFactory();
          if (!typeFactory.existsSemanticType("SampleData"))
             context.getSemanticTypeFactory().addSemanticType(typeFactory.createSemanticType("SampleData", SampleData.class));

          DataEvent prototype = (DataEvent)context.getEventDatagramFactory().newEventInstance("DataEvent");
          prototype.setData(new SampleData());

          prototype.addAnnotation("stringProperty", "//data//stringField"); /** Annotated property for {@link events.SampleData.stringField}. */
          prototype.addAnnotation("intProperty",    "//data//intField");    /** Annotated property for {@link events.SampleData.intField}. */
          prototype.setEventDoubleProperty("doubleProperty", 0.0);

          prototypeFactory.addDataEventPrototype(dataEventId, prototype);
       }

       // AcknowledgementEvent prototype.
       if (!prototypeFactory.existsPrototype(ackEventId))
          prototypeFactory.addEventPrototype("AcknowledgementEvent", ackEventId);

       // TextEvent prototypes for error listeners.
       if (!prototypeFactory.existsPrototype(directErrorEventId))
          prototypeFactory.addEventPrototype("TextEvent", directErrorEventId);
       if (!prototypeFactory.existsPrototype(asyncErrorEventId))
          prototypeFactory.addEventPrototype("TextEvent", asyncErrorEventId);

       // TextEvent prototype for receiver.
       if (!prototypeFactory.existsPrototype(receiverEventId))
          prototypeFactory.addEventPrototype("TextEvent", receiverEventId);
    }

    static void createConsumers() throws Exception
    {
       // Creates the direct consumer for processing of FabricModeratorAdvisory events.
       // FabricModeratorAdvisory events are raised by the Exchange and always have OBSERVABLE event scope.
       // Direct consumers should be used very carefully (the best way is to use them for quick stable operations).
       connection.createEventConsumer("ModeratorAdvisoryConsumer", new ModeratorAdvisoryListener(), FabricModeratorAdvisory.EVENT_ID,
                                      null, EventScope.OBSERVABLE, true);

       // Creates the direct consumer for processing of 'event.text' events.
       // Note that last parameter is false to be able a receiving of events raised fron the same Fabric connection.
       connection.createEventConsumer("TextEventConsumer", new TextEventListener(), textEventId1, null, EventScope.INHERITED, false);

       // Creates the first asynchronous consumer for processing of 'event.data' events.
       EventAsyncConsumer consumer = connection.createEventAsyncConsumer("DataEventConsumer1", new DataEventListener1(), dataEventId,
                                                                          "stringProperty == 'sss' AND intProperty == 1", EventScope.INHERITED, false);
       consumer.start(); // Do not forget to start async consumer!

       // Creates the second asynchronous consumer for processing of 'event.data' events.
       connection.createEventAsyncConsumer("DataEventConsumer2", new DataEventListener2(), dataEventId,
                                           "doubleProperty == 1.1 AND intProperty != 1", EventScope.INHERITED, false).start();

       // Creates the asynchronous consumer for processing of 'event.text2' events and raising of acknowledgements on these events.
       connection.createEventAsyncConsumer("EventListenerWithAckConsumer", new EventListenerWithAck(), textEventId2, null,
                                            EventScope.INHERITED, false).start();

       // Creates the direct consumer for processing of 'event.error.direct' events.
       connection.createEventConsumer("DirectConsumerWithException", new EventListenerWithException(), directErrorEventId, null,
                                       EventScope.GLOBAL, false);

       // Creates the asynchronous consumer for processing of 'event.error.async' events.
       connection.createEventAsyncConsumer("AsyncConsumerWithException", new EventListenerWithException(), asyncErrorEventId, null,
                                            EventScope.GLOBAL, false).start();

       // Creates the request consumer for processing of any event directly raised to him.
       connection.createRequestConsumer("RequestConsumer", new RequestListener(), EventScope.INHERITED);
    }

    static void useModerator() throws Exception
    {
       Moderator moderator = connection.getModerator();

       // In the Client each opened Fabric connection is represented by Fabric component.
       // Fabric components can be accessed through the Moderator using their read-only references:
       ComponentReference componentReference = moderator.lookupComponent(ModeratorUtils.makeComponentFullName(connection));
       System.out.println("   Component: " + componentReference);

       // Component reference contains references for all consumers created by the corresponding Fabric connection:
       for (ConsumerReference consumerReference : componentReference.getConsumers(EventScope.INHERITED))
          System.out.println("      " + consumerReference);

       // Consumer references can also be accessed through the Moderator itself:
       System.out.println("   " + moderator.lookupRequestConsumer(ModeratorUtils.makeConsumerFullName(connection, "RequestConsumer")));
    }

    static void raiseEvents() throws Exception
    {
       // To have possibility of raising an event, connection should bind the corresponding event id to itself.
       // After a connection will bind any event id, this connection becomes event producer.
       // Pay attention that an event id must be bound to a connection only once.
       connection.bindProducerFor(textEventId1);
       connection.bindProducerFor(textEventId2);
       connection.bindProducerFor(dataEventId);
       connection.bindProducerFor(ackEventId);
       connection.bindProducerFor(directErrorEventId);
       connection.bindProducerFor(asyncErrorEventId);

       System.out.println("   Raising text event...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId1);
       textEvent.setText("event");
       // This event will get to TextEventListener and corresponding listeners in the Sample.Node1 node.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       // Note that an event instance can be raised only once.

       System.out.println("   Raising data event 1...");

       DataEvent dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("sss", 1));
       // This event will get to DataEventListener1 and corresponding listeners in the Sample.Node1 node.
       connection.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising data event 2...");

       dataEvent = (DataEvent)context.getEventDatagramFactory().createEvent(dataEventId);
       dataEvent.setData(new SampleData("sss", 2));
       dataEvent.setEventDoubleProperty("doubleProperty", 1.1);
       // This event will get to DataEventListener2 and corresponding listeners in the Sample.Node1 node.
       connection.raiseEvent(dataEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising direct error event 1...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(directErrorEventId);
       textEvent.setText("error");

       try
       {
          // This event will get to EventListenerWithException and corresponding listeners in the Sample.Node1 node.
          connection.raiseEvent(textEvent, EventScope.INHERITED, -1);
       }
       catch (FabricEventException exception)
       {
          Trace.logException(RuntimeSample2.class, exception, false);
       }

       // All FabricEventExceptions thrown by direct consumers will be ignored.
       connection.setExceptionStrategy(ExceptionStrategy.IGNORE);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising direct error event 2...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(directErrorEventId);
       textEvent.setText("error");
       // This event will get to EventListenerWithException and corresponding listeners in the Sample.Node1 node.
       // Exception will be ignored.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising async error event...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(asyncErrorEventId);
       textEvent.setText("error");
       // This event will get to EventListenerWithException and corresponding listeners in the Sample.Node1 node.
       // Exception will be ignored.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
    }

    static void raiseRequests() throws Exception
    {
       System.out.println("   Raising first non-direct request...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request1");
       textEvent.setReplyTo(ackEventId); // Acknowledgement event will have the specified event id.

       // This event will get to EventListenerWithAck located and corresponding listeners in the Sample.Node1 node.
       // This thread will wait for acknowledgement for 1 second.
       AcknowledgementEvent ack = connection.raiseRequest(textEvent, EventScope.INHERITED, RequestDistributionStrategy.AUCTION,
                                                          ReplyMatchStrategy.REPLY_TO, 1000);
       System.out.println("      Acknowledgement: " + ack.getEventId());

       System.out.println("   Raising second non-direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request2");
       // Since 'ReplyTo' is not set, an acknowledgement event will have an auto-generated event id.

       // This event will get to EventListenerWithAck and corresponding listeners in the Sample.Node1 node.
       // This thread will wait for acknowledgement for 2 seconds.
       ack = connection.raiseRequest(textEvent, EventScope.INHERITED, RequestDistributionStrategy.AUCTION, ReplyMatchStrategy.REPLY_TO, 2000);
       System.out.println("      Acknowledgement: " + ack.getEventId());

       Moderator moderator = connection.getModerator();

       System.out.println("   Raising first direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId1);
       textEvent.setText("request3");

       // This event will get directly to RequestConsumer1 located in the Sample.Node1 node.
       // This thread will wait for reply for 1 seconds.
       ImmutableEventDatagram reply = connection.raiseRequest(moderator.lookupRequestConsumer("Sample.Node1://Client_TLP.NodeConnection:RequestConsumer1"),
                                                              textEvent, 1000);
       System.out.println("      Reply: " + reply.getEventId());

       System.out.println("   Raising second direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request4");

       // This event will get directly to RequestConsumer2 located in the Sample.Node1 node.
       reply = connection.raiseRequest(moderator.lookupRequestConsumer("Sample.Node1://Client_TLP.NodeConnection:RequestConsumer2"), textEvent, 1000);
       System.out.println("      Reply: " + reply.getEventId());

       System.out.println("   Raising third direct request...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(textEventId2);
       textEvent.setText("request5");

       // This event will get directly to RequestConsumer located in the Sample.Node1 node.
       reply = connection.raiseRequest(moderator.lookupRequestConsumer("Sample.Node1://Client_TLP.ClientConnection:RequestConsumer"), textEvent, 1000);
       System.out.println("      Reply: " + reply.getEventId());
    }

    static void invokeSlangRequests() throws Exception
    {
       // Creates Semantic Language session to the Sample.Node1 node.
       SLSession session = connection.createSLSession();

       System.out.println("\n   First SLANG request...\n");

       // Invokes a request using a plain text command.
       SLResponse response = session.slangRequest("list users", 1000);
       // Result data is contained in the RowSet.
       new RowSetPrinter().print(response.getRowSet());

       System.out.println("\n   Second SLANG request...\n");

       // Invokes a request using a special SLStatement object.
       response = session.slangRequest(new ListUsersOperation.Definition(), 1000);
       new RowSetPrinter().print(response.getRowSet());
    }

    static void useReceiver() throws Exception
    {
       // Creates the receiver consumer for a obtaining of 'event.receiver' events.
       final EventReceiver receiver = connection.createEventReceiver("Receiver", receiverEventId, null, EventScope.GLOBAL, false);

       connection.bindProducerFor(receiverEventId);

       // Waits an event for 2 seconds. No event will be obtained.
       ImmutableEventDatagram event = receiver.receive(1000);
       if (event == null)
          System.out.println("\n   No event received.\n");

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising receiver event 1...");

       TextEvent textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(receiverEventId);
       textEvent.setText("event1");
       // This event will get to AllEventsListener and receiver.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       Utils.sleep(200); // It is only needed for consecutive printing.
       System.out.println("   Raising receiver event 2...");

       textEvent = (TextEvent)context.getEventDatagramFactory().createEvent(receiverEventId);
       textEvent.setText("event2");
       // This event will get to AllEventsListener and receiver.
       connection.raiseEvent(textEvent, EventScope.INHERITED, -1);

       // Waits an event for 2 seconds. First event will be obtained.
       event = receiver.receive(1000);
       System.out.println("\n   Received " + event.getEventId() + " : " + ((TextEvent)event).getText());

       // Waits indefinitely until event will be available. Second event will be obtained.
       // Use such method very carefully because it can block the current thread forever.
       event = receiver.receive();
       System.out.println("   Received " + event.getEventId() + " : " + ((TextEvent)event).getText());

       // Tries to obtain an event without a waiting. No event will be obtained.
       event = receiver.receiveNoWait();
       if (event == null)
          System.out.println("\n   No event received.\n");

       // Creates and starts thread for a obtaining of 'event.receiver' events from other node and clients.
       FabricThreadManager.getInstance().createThread("ReceiverThread", "Receives remote events.",
                                                      new Runnable()
                                                      {
                                                         @Override
                                                         public void run()
                                                         {
                                                            while (!Thread.interrupted())
                                                            {
                                                               try
                                                               {
                                                                  // Waits indefinitely until next event will be available.
                                                                  TextEvent event = (TextEvent)receiver.receive();
                                                                  System.out.println("      Receiver: " + event.getEventId() + " : " + event.getText());
                                                               }
                                                               catch (Exception exception)
                                                               {
                                                                  exception.printStackTrace();
                                                               }
                                                            }
                                                         }
                                                      }).start();
    }

    // Event listener for FabricModeratorAdvisory events.
    static class ModeratorAdvisoryListener  implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          FabricModeratorAdvisory advisory = (FabricModeratorAdvisory)event;
          System.out.println("      FabricModeratorAdvisoryListener: " + advisory.getType() + ", " + advisory.getEntity());
       }
    }

    // Event listener for 'event.text' events.
    static class TextEventListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      TextEventListener: " + event.getEventId() + " : " + ((TextEvent)event).getText());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // First event listener for 'event.data' events.
    static class DataEventListener1 implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      DataEventListener1: " + event.getEventId() + " : " + ((DataEvent)event).getData());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // Second event listener for 'event.data' events.
    static class DataEventListener2 implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      DataEventListener2: " + event.getEventId() + " : " + ((DataEvent)event).getData());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // Event listener for 'event.text' events with acknowledgement.
    static class EventListenerWithAck implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             System.out.println("      EventListenerWithAck: " + event.getEventId() + " : " + ((TextEvent)event).getText());
             connection.acknowledgeEvent((EventDatagram)event, true, AcknowledgeAction.ACKNOWLEDGE, null, EventScope.INHERITED, 0);
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }
       }
    }

    // Event listener for 'event.error.*' events. It is used in direct and async consumers.
    static class EventListenerWithException implements FabricEventListener
    {
       @Override
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             System.out.println("      EventListenerWithException: " + event.getEventId() + " : " + ((TextEvent)event).getText());
          }
          catch (Exception exception)
          {
             exception.printStackTrace();
          }

          throw new FabricEventException("EventListenerWithException.");
       }
    }

    // Request listener for any event.
    static class RequestListener implements FabricRequestListener
    {
       public ImmutableEventDatagram onRequest(ImmutableEventDatagram event) throws FabricRequestException
       {
          System.out.println("      RequestListener: " + event.getEventId());
          return event;
       }
    }
 }
