package client.reliable;

import com.streamscape.Trace;
import com.streamscape.cli.ClientContext;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.lib.utils.Utils;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.enums.AcknowledgeAction;
import com.streamscape.sdo.enums.ReplyMatchStrategy;
import com.streamscape.sdo.enums.RequestDistributionStrategy;
import com.streamscape.sdo.event.AcknowledgementEvent;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.event.TextEvent;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.moderator.FabricModeratorAdvisory;
import com.streamscape.sef.moderator.ModeratorAdvisoryType;

/**
 * <p>Title: Java Samples</p>
 *
 * <p>Description: First client of the 'client/reliable' sample.
 *
 * <p>This client must be connected to the ProxyRuntime node.
 * <br>The client raises 10 events with 5 second delay and also listens (with raising acknowledgement)
 * for such events raised by the other client.
 *
 * <p>Copyright: Copyright (c) 2014</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Mikhail Filichev
 * @version 3.4
 */
 public class Client1
 {
    private static FabricConnection connection;

    private static final String EVENT_ID = "event.sample";

    public static void main(String[] args)
    {
       try
       {
          // Enables some traces.
          Trace.enable("*", Trace.Level.ERROR);

          // Initializes the Client Context.
          ClientContext context = ClientContext.getInstance();

          System.out.println("\nSetup environment...\n");
          setupEnvironment();

          System.out.println("\nRaise events...\n");
          raiseEvents();

          System.out.println("\nDone.\n");
       }
       catch (Exception exception)
       {
          exception.printStackTrace();
          System.exit(1);
       }
    }

    private static void setupEnvironment() throws Exception
    {
       connection = new FabricConnectionFactory("tlp://localhost:5000").createConnection("User1", "123");
       connection.setName("Client1");
       connection.open();

       // Imports the event prototype from the node.
       connection.importEventPrototype(EVENT_ID);

       // Creates the asynchronous consumer that raises an acknowledgement on the received event.
       connection.createEventAsyncConsumer("Consumer", new EventListener(), EVENT_ID, null, EventScope.INHERITED, true).start();

       // Creates the asynchronous consumer that monitors disconnection of the proxy node.
       connection.createEventAsyncConsumer("DisconnectionConsumer", new NodeDisconnectionListener(), FabricModeratorAdvisory.EVENT_ID,
                                           "type = 'NODE_DISCONNECTED' OR type = 'NODE_DISCONNECTED_FORCIBLY'", EventScope.INHERITED, true).start();

       // Creates the asynchronous consumer that monitors reconnection of this client.
       connection.createEventAsyncConsumer("ReconnectionConsumer", new ClientReconnectionListener(), FabricModeratorAdvisory.EVENT_ID,
                                           "type = 'CLIENT_RECONNECTED'", EventScope.INHERITED, true).start();
    }

    private static void raiseEvents() throws Exception
    {
       connection.bindProducerFor(EVENT_ID);

       // Raises an event every 5 seconds (until all 10 events will be raised).
       for (int i = 1; i <= 10;)
       {
          Utils.sleep(5000); // 5 seconds.

          TextEvent event = (TextEvent)EventDatagramFactory.getInstance().createEvent(EVENT_ID);
          event.setText("#" + i);

          try
          {
             // Raises the event and waits for acknowledgement for 1 second.
             System.out.print("Raising event " + event.getText() + "...");
             AcknowledgementEvent ack = connection.raiseRequest(event, EventScope.INHERITED, RequestDistributionStrategy.AUCTION,
                                                                ReplyMatchStrategy.REPLY_TO, 1000);
             System.out.println(" Acknowledgement received for event " + ack.getData() + ".\n");
             ++i;
          }
          catch (Exception exception)
          {
             System.out.println(" Acknowledgement NOT received.");
             Trace.logException(Client1.class, exception, false);
             System.out.println();
          }
       }
    }

    // Event listener that just raises an acknowledgement on the received event.
    private static class EventListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             // Raises acknowledgement.
             connection.acknowledgeEvent((TextEvent)event, true, AcknowledgeAction.ACKNOWLEDGE, null, EventScope.INHERITED, -1);
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    // Event listener that monitors disconnection of the proxy node.
    private static class NodeDisconnectionListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             FabricModeratorAdvisory advisory = (FabricModeratorAdvisory)event;
             System.out.println("Proxy node disconnected" + (advisory.getType() == ModeratorAdvisoryType.NODE_DISCONNECTED ? ".\n" : " forcibly.\n"));
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }

    // Event listener that monitors reconnection of this client.
    private static class ClientReconnectionListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          try
          {
             System.out.println("Client reconnected.\n");
          }
          catch (Exception exception)
          {
             throw new FabricEventException(exception);
          }
       }
    }
 }
