from __future__ import absolute_import

import unittest, os

from stpy.fabric.row_set import RowSet, RowSetProxy
from stpy.fabric.lobs import Blob, Clob, BlobImpl, ClobImpl, RowSetBlobProxy, RowSetClobProxy

from stpy.fabric.utils import Utils
from stpy.fabric.sl_response import SLResponse

from stpy import HTTPFabricConnection, SLFileSessionContext
from stpy import DataspaceType

class TestRowSetLobProxy(unittest.TestCase):
    def setUp(self):
        Utils.initLogs()
        self.connection = self.openConnection()
        self.accessor = self.initDataspace(self.connection)
        pass

    def tearDown(self):
        self.dropDataspace(self.connection)
        if self.connection:
            self.connection.close()
        pass

    def testRowSetLobProxy(self):
        bytes10 = self._createBytes(10)
        bytes100 = self._createBytes(100)
        bytes1000 = self._createBytes(1000)
        bytes10000 = self._createBytes(10000)

        string10 = self._createString(10)
        string100 = self._createString(100)
        string1000 = self._createString(1000)
        string10000 = self._createString(10000)

        _self = self
        def removeFiles():
            _self._removeFile("testbytesfile10.txt")
            _self._removeFile("testbytesfile100.txt")
            _self._removeFile("testbytesfile1000.txt")
            _self._removeFile("testbytesfile10000.txt")

            _self._removeFile("teststringfile10.txt")
            _self._removeFile("teststringfile100.txt")
            _self._removeFile("teststringfile1000.txt")
            _self._removeFile("teststringfile10000.txt")

        removeFiles()
        self._createFile("testbytesfile10.txt",    bytes10)
        self._createFile("testbytesfile100.txt",   bytes100)
        self._createFile("testbytesfile1000.txt",  bytes1000)
        self._createFile("testbytesfile10000.txt", bytes10000)

        self._createFile("teststringfile10.txt",    string10)
        self._createFile("teststringfile100.txt",   string100)
        self._createFile("teststringfile1000.txt",  string1000)
        self._createFile("teststringfile10000.txt", string10000)

        self.accessor.setRequestTimeout(300000)
        self.slSession = self.connection.createSLSession()
        self.assertResponse(self.slSession.slangRequest("use TSPACE.TEST_TSPACE", 30000), True)

        self.slSession.slangRequest("drop collection LobsTable", 30000)
        self.assertResponse(self.slSession.slangRequest("create persistent table LobsTable(id int identity, b Blob, c Clob, fb flob, fc flob)", 30000), True)

        # TODO: implement readFile and file request consumers in HTTPFabricConnection and use SLFileSessionContext.CLIENT
        self.accessor.setSessionContext(SLFileSessionContext.SERVER)
        self.accessor.executeQuery("insert into LobsTable values (1, ?, ?, readFile('testbytesfile10.txt'),    readFile('teststringfile10.txt'))", [bytes10, string10])
        self.accessor.executeQuery("insert into LobsTable values (2, ?, ?, readFile('testbytesfile100.txt'),   readFile('teststringfile100.txt'))", [bytes100, string100])
        self.accessor.executeQuery("insert into LobsTable values (3, ?, ?, readFile('testbytesfile1000.txt'),  readFile('teststringfile1000.txt'))", [bytes1000, string1000])
        self.accessor.executeQuery("insert into LobsTable values (4, ?, ?, readFile('testbytesfile10000.txt'), readFile('teststringfile10000.txt'))", [bytes10000, string10000])
        self.accessor.setSessionContext(SLFileSessionContext.SERVER)
        self.accessor.setFetchSize(1000);

        # by default download all lobs in dataspace accessor
        self.assertEqual(self.accessor.getDownloadableLobSize(), -1)

        rowSet = self.accessor.executeQuery("select * from LobsTable")

        self.assertIsNotNone(rowSet)
        self.assertEqual(type(rowSet), RowSet)

        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes10)
        self.checkClobIsNotProxy(rowSet, string10)

        self.assertTrue(rowSet.next())
        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes1000)
        self.checkClobIsNotProxy(rowSet, string1000)

        self.assertTrue(rowSet.next())
        self.assertFalse(rowSet.next())

        rowSet.close()

        # blob size 100
        self.assertEqual(self.accessor.getDownloadableLobSize(), -1)
        self.accessor.setDownloadableLobSize(100)
        self.assertEqual(self.accessor.getDownloadableLobSize(), 100)

        rowSet = self.accessor.executeQuery("select * from LobsTable")
        self.assertIsNotNone(rowSet);
        self.assertEqual(type(rowSet), RowSetProxy)

        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes10)
        self.checkClobIsNotProxy(rowSet, string10)

        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes100)
        self.checkClobIsProxy(rowSet, string100, False)

        self.assertTrue(rowSet.next())
        self.checkBlobIsProxy(rowSet, bytes1000)
        self.checkClobIsProxy(rowSet, string1000, True)

        self.assertTrue(rowSet.next())
        self.assertFalse(rowSet.next())

        rowSet.close();

        # blob size 0
        self.accessor.setDownloadableLobSize(0)
        self.assertEqual(self.accessor.getDownloadableLobSize(), 0)

        rowSet = self.accessor.executeQuery("select * from LobsTable")
        self.assertIsNotNone(rowSet)
        self.assertEqual(type(rowSet), RowSetProxy)

        self.assertTrue(rowSet.next())
        self.checkBlobIsProxy(rowSet, bytes10)
        self.checkClobIsProxy(rowSet, string10, True)

        self.assertTrue(rowSet.next());
        self.checkBlobIsProxy(rowSet, bytes100);
        self.checkClobIsProxy(rowSet, string100, True);

        self.assertTrue(rowSet.next());
        self.assertTrue(rowSet.next());
        self.assertFalse(rowSet.next());

        rowSet.close();

        # fetch size 1 and blob size 20
        self.accessor.setDownloadableLobSize(20);
        self.assertEqual(self.accessor.getDownloadableLobSize(), 20)
        self.accessor.setFetchSize(1);
        self.assertEqual(self.accessor.getFetchSize(), 1)

        rowSet = self.accessor.executeQuery("select * from LobsTable");
        self.assertIsNotNone(rowSet);
        self.assertEqual(type(rowSet), RowSetProxy)

        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes10)
        self.checkClobIsNotProxy(rowSet, string10)

        self.assertTrue(rowSet.next())
        self.checkBlobIsProxy(rowSet, bytes100)
        self.checkClobIsProxy(rowSet, string100, True)

        self.assertTrue(rowSet.next())
        self.checkBlobIsProxy(rowSet, bytes1000)
        self.checkClobIsProxy(rowSet, string1000, True)

        self.assertTrue(rowSet.next())
        self.checkBlobIsProxy(rowSet, bytes10000)
        self.checkClobIsProxy(rowSet, string10000, True)

        self.assertFalse(rowSet.next());

        rowSet.close();

        # asRowSet
        self.accessor.setDownloadableLobSize(10);
        self.accessor.setFetchSize(2);

        rowSet = self.accessor.executeQuery("select * from LobsTable");
        self.assertIsNotNone(rowSet)
        self.assertEqual(type(rowSet), RowSetProxy)

        rowSet = rowSet.asRowSet();
        self.assertIsNotNone(rowSet)
        self.assertEqual(type(rowSet), RowSet)

        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes10)
        self.checkClobIsNotProxy(rowSet, string10)

        self.assertTrue(rowSet.next())
        self.assertTrue(rowSet.next())
        self.checkBlobIsNotProxy(rowSet, bytes1000)
        self.checkClobIsNotProxy(rowSet, string1000)

        self.assertTrue(rowSet.next())
        self.assertFalse(rowSet.next())

        rowSet.close()

        # test free and truncate methods
        # self.accessor.setDownloadableBlobSize(1)
        # self.accessor.setFetchSize(1)
        #
        # rowSet = self.accessor.executeQuery("select * from LobsTable")
        # self.assertIsNotNone(rowSet)
        # self.assertEqual(type(rowSet), RowSetProxy)
        #
        # rowSet.next()
        # self.checkBlobIsProxy(rowSet, bytes10)
        # self.checkClobIsProxy(rowSet, string10, True)
        #
        # blob = rowSet.getBlob(2);
        # blob.truncate(4);
        # blob.free();
        #
        # clob = rowSet.getClob(3);
        # clob.truncate(3);
        # clob.free();
        #
        # clob = rowSet.getClob(5);
        # clob.truncate(3);
        # clob.free();
        #
        # rowSet = self.accessor.executeQuery("select * from LobsTable")
        # self.assertNotNull(rowSet)
        # self.assertEqual(type(rowSet), RowSetProxy)
        #
        # rowSet.next()
        # #self.checkBlobIsProxy(rowSet, Arrays.copyOf(bytes10, 4)); #TODO: fix
        # self.checkClobIsProxy(rowSet, string10.substring(0, 3), True)

        # don't download lobs in slang session
        response = self.slSession.slangRequest("select * from LobsTable", 30000)
        self.assertResponse(response, True)

        rowSet = response.rowSet
        self.assertTrue(rowSet.next())
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1), "[ BLOB ]")
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2), "[ CLOB ]")
        self.assertTrue(rowSet.next())
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1), "[ BLOB ]")
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2), "[ CLOB ]")
        self.assertTrue(rowSet.next())
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1), "[ BLOB ]")
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2), "[ CLOB ]")
        self.assertTrue(rowSet.next())
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1), "[ BLOB ]")
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2), "[ CLOB ]")
        self.assertFalse(rowSet.next())

        rowSet.close()

        self.slSession.slangRequest("drop collection LobsTable", 30000)

        removeFiles()

        pass

    def checkBlobIsNotProxy(self, rowSet, bytes):
        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(1)), BlobImpl)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1).get_all_bytes(), bytes)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1).get_bytes(0, rowSet.getCurrentRow().getColumnByIndex(1).length()), bytes)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1).get_bytes(3, 5), bytes[3:8])

        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(3)), BlobImpl)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(3).get_all_bytes(), bytes)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(3).get_bytes(0, rowSet.getCurrentRow().getColumnByIndex(3).length()), bytes)

    def checkClobIsNotProxy(self, rowSet, s):
        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(2)), ClobImpl)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2).get_all_string(), s)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2).get_sub_string(0, rowSet.getCurrentRow().getColumnByIndex(2).length()), s)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2).get_sub_string(3, 5), s[3:8])

        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(4)), BlobImpl) # by default flob is binary
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(4).get_all_bytes().decode("utf-8"), s)


    def checkBlobIsProxy(self, rowSet, bytes):
        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(1)), RowSetBlobProxy)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1).get_bytes(0, rowSet.getCurrentRow().getColumnByIndex(1).length()), bytes);
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(1).get_all_bytes(), bytes);

        self._testRowSetBlobProxyMethods(rowSet.getCurrentRow().getColumnByIndex(1), bytes);

        ############

        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(3)), RowSetBlobProxy)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(3).get_bytes(0, rowSet.getCurrentRow().getColumnByIndex(3).length()), bytes);
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(3).get_all_bytes(), bytes);

        self._testRowSetBlobProxyMethods(rowSet.getCurrentRow().getColumnByIndex(3), bytes);

    def checkClobIsProxy(self, rowSet, s, andFlob):
        self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(2)), RowSetClobProxy)
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2).get_sub_string(0, rowSet.getCurrentRow().getColumnByIndex(2).length()), s);
        self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(2).get_all_string(), s);

        self._testRowSetClobProxyMethods(rowSet.getCurrentRow().getColumnByIndex(2), s);

        ################

        if andFlob:
            self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(4)), RowSetBlobProxy)
            # self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(4).length(), len(s))
            # self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(4).get_sub_string(0, rowSet.getCurrentRow().getColumnByIndex(2).length()), s);
            # self.assertEqual(rowSet.getCurrentRow().getColumnByIndex(4).get_all_string(), s);
            # self._testRowSetClobProxyMethods(rowSet.getCurrentRow().getColumnByIndex(4), s);
        else:
            self.assertEqual(type(rowSet.getCurrentRow().getColumnByIndex(4)), BlobImpl)

    def _testRowSetBlobProxyMethods(self, blob, value):
        self.assertEqual(type(blob), RowSetBlobProxy)
        self.assertEqual(blob.length(), len(value))

        self.assertEqual(blob.get_bytes(0, len(value)), value)
        if len(value) > 50:
            self.assertEqual(blob.get_bytes(9, 50), value[9:59])

    def _testRowSetClobProxyMethods(self, clob, value):
        self.assertEqual(clob.length(), len(value))

        self.assertEqual(clob.get_sub_string(0, clob.length()), value);
        self.assertEqual(clob.get_sub_string(1, clob.length() - 3), value[1:len(value) - 2]);

    def openConnection(self):
        connection = HTTPFabricConnection("localhost:8888", "Admin", "Admin")
        connection.open()
        self.assertEqual(connection.isOpened(), True)
        self.assertEqual(connection.ping(), "AVAILABLE")
        return connection

    def initDataspace(self, connection):
        slSession = connection.createSLSession()
        slSession.slangRequest("drop dataspace TEST_TSPACE", 30000)
        slSession.slangRequest("create dataspace TEST_TSPACE type TSPACE", 30000)
        slSession.close()

        dataspaceAccessor = connection.createDataspaceAccessor(None, DataspaceType.TSPACE, "TEST_TSPACE")
        dataspaceAccessor.setRequestTimeout(300000)
        self.assertEqual(dataspaceAccessor.getFetchSize(), 0)
        return dataspaceAccessor

    def dropDataspace(self, connection):
        if self.accessor:
            self.accessor.close()
        self.accessor = None
        slSession = connection.createSLSession()
        slSession.slangRequest("drop dataspace TEST_TSPACE", 30000)
        slSession.close()

    def assertResponse(self, response, isOK):
        if isOK and not response.isOK:
            if response.exception != None:
                raise response.exception
            if response.text != None:
                raise Exception(response.text)

        self.assertFalse(isOK ^ response.isOK)
        return response;

    def _createBytes(self, size):
        bytes = bytearray(size)
        for i in range(0, size):
            bytes[i] = i%256
        return bytes

    def _createString(self, size):
        if Utils.is_python3():
            return "".join(chr(ord('A') + i % (ord('Z') - ord('A'))) for i in range(size))
        else:
            return "".join(unichr(ord('A') + i % (ord('Z') - ord('A'))) for i in range(size))

    def _getNodeDirectory(self):
        return os.getcwd() + "/target/"

    def _getFilePath(self, filename):
        return self._getNodeDirectory() + filename

    def _removeFile(self, filename):
        try:
            os.remove(self._getFilePath(filename))
        except OSError:
            pass

    def _createFile(self, filename, content):
        mode = 'w+'
        if Utils.is_python3():
            if type(content) == bytearray or type(content) == bytes:
                mode = 'wb'
        with open(self._getFilePath(filename), mode) as f:
            f.write(content)





