from __future__ import absolute_import

import decimal
import unittest
from datetime import datetime, date, time
import pytz # $ pip install pytz
from tzlocal import get_localzone # $ pip install tzlocal

from stpy import ds
from stpy.ds.exceptions import *
from stpy.fabric.row_set import RowSet, RowSetProxy
from stpy.fabric.utils import Utils

class TestDsapi(unittest.TestCase):
    def setUp(self):
        Utils.initLogs()
        pass

    def tearDown(self):
        pass

    def test_connect(self):
        db = self._connect()

        cursor = db.cursor()
        cursor.execute("select 1")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 1)

        db.close()

        pass

    def test_close(self):

        # close before cursor
        db = self._connect()

        cursor = db.cursor()
        cursor.execute("select 1")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 1)

        db.close()

        try:
            cursor = db.cursor()
            self.assertEquals(0, 1)
        except Exception as e:
            self.assertEquals(e.message, "Connection is closed.")

            # close after cursor
        db = self._connect()

        cursor = db.cursor()
        db.close()

        try:
            cursor.execute("select 1")
            self.assertEquals(0, 1)
        except Exception as e:
            self.assertEquals(e.message, "Connection is closed.")

    def test_callproc(self):
        db = self._connect()
        cursor = db.cursor()

        # FunctionThatReturnsInt
        r = cursor.callproc("FunctionThatReturnsInt", (10,))
        self.assertEquals(len(r), 1)
        self.assertEquals(r[0], 10)

        r = cursor.fetchall()
        self.assertEquals(len(cursor.description), 1)
        self.assertEquals(len(cursor.description[0]), 7)
        self.assertEquals(cursor.description[0][0], "@p0")
        self.assertEquals(cursor.description[0][1], ds.NUMBER)

        self.assertEquals(len(r), 1)
        self.assertEquals(len(r[0]), 1)
        self.assertEquals(r[0][0], 10)

        db.close()

    def test_TspaceTableWithSimpleTypes(self):
        db = self._connect()
        cursor = db.cursor()
        cursor.arraysize = -1

        #     "  id int, charColumn2 char(3), varcharColumn3 varchar(100), longvarcharColumn4 longvarchar(100)" +
        #     ", nvarcharColumn5 nvarchar(100), stringColumn6 string" +
        #     ", dateColumn7 sqldate, timeColumn8 sqltime, timestampColumn9 sqltimestamp" +
        #     ", tinyintColumn10 tinyint, smallintColumn11 smallint, integerColumn12 integer, bigintColumn13 bigint, realColumn14 real, doubleColumn15 double, floatColumn16 float" +
        #     ", numericColumn17 numeric(11, 6), decimalColumn18 decimal(10, 5), booleanColumn19 boolean" +
        #     ", binaryColumn20 binary(3), varbinaryColumn21 varbinary(100), clobColumn22 clob, blobColumn23 blob(10), bitColumn24 bit" +
        #     ", arrayColumn25 int array[10] default null, otherColumn26 other, eventColumn27 event" +
        #     ", charColumn28 char, varcharColumn29 varchar(1), longmaxColumn30 decimal, varcharwithhexColumn31 varchar(100), clobhexColumn32 clob, varcharColumn33 varchar(100)"+

        # delete
        cursor.execute("delete FROM TestTableWithSimpleTypes")
        cursor.execute("delete FROM TestTableWithSimpleTypes")
        self._assertUpdateCount(cursor, 0)

        # select from empty table
        cursor.execute("select * from TestTableWithSimpleTypes")
        self.assertIsNone(cursor.next())

        strdatetimeinlocal = get_localzone().localize(datetime(2017, 1, 18, 15, 16, 17, 123000))

        # insert into TestTableWithSimpleTypes
        cursor.execute("insert into TestTableWithSimpleTypes values ( \
                              1, 'c2', 'c3', 'c4',\
                              'c5', 'c6',\
                              '2017-01-18', '15:16:17','2017-01-18 15:16:17.123',\
                              10,11,12,13,14.1,15.2,16.3,\
                              17.1234567,18.12345678,true,\
                              null, null, 'clob22', null, null,\
                              null, null, null,\
                              'c', 's', 30.23456, 'c31', 'c32', 'c33'\
                              )")
        self._assertUpdateCount(cursor, 1)

        # insert into TestTableWithSimpleTypes, using arguments
        cursor.execute("insert into TestTableWithSimpleTypes values (\
                              ?, ?, ?, ?,\
                              ?, ?,\
                              ?, ?,?,\
                              ?,?,?,?,?,?,?,\
                              ?,?,?,\
                              ?, ?, ?, ?, ?,\
                              ?, ?, ?,\
                              ?, ?, ?, ?, ?, ?)",
                       [2, 'c22', 'c32', 'c42',
                        'c52', 'c62',
                        '2017-01-18', '15:16:17','2017-01-18 15:16:17.123',
                        20,21,22,23,24.1,25.2,26.3,
                        27.1234567,28.12345678, True,
                        bytearray([11,12,13]), bytearray(b'bytesseq'), 'clob22', 'cXdl', 0,
                        [1,2,3], None, None,
                        'd', 't', 32.23456, 'c32', 'c33', 'c34']
                       );
        self._assertUpdateCount(cursor, 1)

        # insert into TestTableWithSimpleTypes, using date and decimals objects
        dateDate = date(2016, 12, 13)
        dateTime = time(15, 16, 17)
        dateTimestamp = datetime(2016, 12, 13, 15, 16, 17, 123000)
        cursor.execute("insert into TestTableWithSimpleTypes values (\
                              ?, ?, ?, ?,\
                              ?, ?,\
                              ?, ?,?,\
                              ?,?,?,?,?,?,?,\
                              ?,?,?,\
                              ?, ?, ?, ?, ?,\
                              ?, ?, ?,\
                              ?, ?, ?, ?, ?, ?)",
                       [3, 'c22', 'c32', 'c42',
                        'c52', 'c62',
                        dateDate, dateTime, dateTimestamp,
                        20,21,22,23,24.1,25.2,26.3,
                        decimal.Decimal('27.1234567'),decimal.Decimal('28.12345678'), True,
                        None, None, None, None, None,
                        None, None, None,
                        'd', 't', decimal.Decimal('32.23456'), 'c32', 'c33', 'c34']
                       );
        self._assertUpdateCount(cursor, 1)

        # insert into TestTableWithSimpleTypes, using date in locale timezone
        dateTimestampLocal = get_localzone().localize(datetime(2016, 12, 13, 15, 16, 17, 123000))
        cursor.execute("insert into TestTableWithSimpleTypes values (\
                              ?, ?, ?, ?,\
                              ?, ?,\
                              ?, ?,?,\
                              ?,?,?,?,?,?,?,\
                              ?,?,?,\
                              ?, ?, ?, ?, ?,\
                              ?, ?, ?,\
                              ?, ?, ?, ?, ?, ?)",
                       [4, 'c24', 'c32', 'c42',
                        'c52', 'c62',
                        dateDate, dateTime, dateTimestampLocal,
                        20,21,22,23,24.1,25.2,26.3,
                        decimal.Decimal('27.1234567'),decimal.Decimal('28.12345678'), True,
                        None, None, None, None, None,
                        None, None, None,
                        'd', 't', decimal.Decimal('32.23456'), 'c32', 'c33', 'c34']
                       );
        self._assertUpdateCount(cursor, 1)

        # select * from TestTableWithSimpleTypes
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")

        self.assertEquals(len(cursor.description), 33)
        self.assertEquals(len(cursor.description[0]), 7)
        self.assertEquals(cursor.description[0][0], "id")
        self.assertEquals(cursor.description[0][1], ds.NUMBER)
        self.assertEquals(cursor.description[1][0], "charColumn2")
        self.assertEquals(cursor.description[1][1], ds.STRING)
        self.assertEquals(cursor.description[6][0], "dateColumn7")
        self.assertEquals(cursor.description[6][1], ds.DATE)
        self.assertEquals(cursor.description[7][0], "timeColumn8")
        self.assertEquals(cursor.description[7][1], ds.TIME)
        self.assertEquals(cursor.description[8][0], "timestampColumn9")
        self.assertEquals(cursor.description[8][1], ds.TIMESTAMP)
        self.assertEquals(cursor.description[20][0], "varbinaryColumn21")
        self.assertEquals(cursor.description[20][1], ds.BINARY)
        self.assertEquals(cursor.rowcount, 4)

        row = cursor.next()
        self.assertEqual(row[0], 1)
        self.assertEqual(row[1], 'c2 ')
        self.assertEqual(row[2], "c3")
        self.assertEqual(row[3], "c4")
        self.assertEqual(row[4], "c5")
        self.assertEqual(row[5], "c6")
        self.assertEqual(row[6].strftime("%Y-%m-%d"), "2017-01-18")
        self.assertEqual(row[7].strftime("%H:%M:%S"), "15:16:17")
        #self.assertEqual(row[8].strftime("%Y-%m-%d %H:%M:%S.%f"), "2017-01-18 12:16:17.123000") # here datetime in UTC timezone, works for MSK(+3) timezone only
        self.assertEqual(row[8].strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.astimezone(pytz.utc).strftime("%Y-%m-%d %H:%M:%S.%f")) # here datetime returned in UTC timezone
        self.assertEqual(row[8].astimezone(get_localzone()).strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.strftime("%Y-%m-%d %H:%M:%S.%f")) # convert returned datetime to local

        self.assertEqual(row[9], 10)
        self.assertEqual(row[10], 11)
        self.assertEqual(row[11], 12)
        self.assertEqual(row[12], 13)
        self.assertEqual(row[13], 14.1)
        self.assertEqual(row[14], 15.2)
        self.assertEqual(row[15], 16.3)
        self.assertEqual(row[16], decimal.Decimal('17.123457'))
        self.assertEqual(row[17], decimal.Decimal('18.12346'))
        self.assertEqual(row[18], True)
        self.assertEqual(row[19], None)
        self.assertEqual(row[20], None)
        self.assertEqual(row[21].get_all_string(), "clob22")
        self.assertEqual(row[22], None)
        self.assertEqual(row[23], None)
        self.assertEqual(row[24], None)
        self.assertEqual(row[25], None)
        self.assertEqual(row[26], None)
        self.assertEqual(row[27], "c")
        self.assertEqual(row[28], "s")
        self.assertEqual(row[29], decimal.Decimal('30'))  # TODO: check
        self.assertEqual(row[30], "c31")
        self.assertEqual(row[31].get_all_string(), "c32")
        self.assertEqual(row[32], "c33")

        row = cursor.next()
        self.assertEqual(row[0], 2)
        self.assertEqual(row[1], 'c22')
        self.assertEqual(row[2], "c32")
        self.assertEqual(row[3], "c42")
        self.assertEqual(row[4], "c52")
        self.assertEqual(row[5], "c62")
        self.assertEqual(row[6].strftime("%Y-%m-%d"), "2017-01-18")
        self.assertEqual(row[7].strftime("%H:%M:%S"), "15:16:17")
        #self.assertEqual(row[8].strftime("%Y-%m-%d %H:%M:%S.%f"), "2017-01-18 12:16:17.123000") # here datetime in UTC timezone, works for MSK(+3) timezone only
        self.assertEqual(row[8].strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.astimezone(pytz.utc).strftime("%Y-%m-%d %H:%M:%S.%f")) # here datetime returned in UTC timezone
        self.assertEqual(row[8].astimezone(get_localzone()).strftime("%Y-%m-%d %H:%M:%S.%f"), strdatetimeinlocal.strftime("%Y-%m-%d %H:%M:%S.%f")) # convert returned datetime to local
        self.assertEqual(row[9], 20)
        self.assertEqual(row[10], 21)
        self.assertEqual(row[11], 22)
        self.assertEqual(row[12], 23)
        self.assertEqual(row[13], 24.1)
        self.assertEqual(row[14], 25.2)
        self.assertEqual(row[15], 26.3)
        self.assertEqual(row[16], decimal.Decimal('27.123457'))
        self.assertEqual(row[17], decimal.Decimal('28.12346'))
        self.assertEqual(row[18], True)
        self.assertEqual(row[19], bytearray([11,12,13]))
        self.assertEqual(row[20], bytearray(b'bytesseq'))
        self.assertEqual(row[21].get_all_string(), "clob22")
        self.assertEqual(row[22].get_all_bytes(), bytearray(b'qwe'))
        self.assertEqual(row[23], False)
        self.assertEqual(row[24], [1,2,3])
        self.assertEqual(row[25], None)
        self.assertEqual(row[26], None)
        self.assertEqual(row[27], "d")
        self.assertEqual(row[28], "t")
        self.assertEqual(row[29], decimal.Decimal('32.23456')) # TODO: check
        self.assertEqual(row[30], "c32")
        self.assertEqual(row[31].get_all_string(), "c33")
        self.assertEqual(row[32], "c34")

        row = cursor.next()
        self.assertEqual(row[0], 3)
        self.assertEqual(row[1], 'c22')
        self.assertEqual(row[6].strftime("%Y-%m-%d"), dateDate.strftime("%Y-%m-%d"))
        self.assertEqual(row[7].strftime("%H:%M:%S"), dateTime.strftime("%H:%M:%S"))
        #self.assertEqual(row[8].strftime("%Y-%m-%d %H:%M:%S.%f"), "2016-12-13 15:16:17.000000") #TODO: check why milliseconds lost
        self.assertEqual(row[8].strftime("%Y-%m-%d %H:%M:%S"), dateTimestamp.strftime("%Y-%m-%d %H:%M:%S")) # TODO: check why milliseconds lost
        self.assertEqual(row[16], decimal.Decimal('27.123457'))
        self.assertEqual(row[17], decimal.Decimal('28.12346'))
        self.assertEqual(row[29], decimal.Decimal('32')) # TODO: check

        row = cursor.next()
        self.assertEqual(row[0], 4)
        self.assertEqual(row[1], 'c24')
        self.assertEqual(row[6].strftime("%Y-%m-%d"), dateDate.strftime("%Y-%m-%d"))
        self.assertEqual(row[7].strftime("%H:%M:%S"), dateTime.strftime("%H:%M:%S"))
        # TODO: check why milliseconds lost
        # datetime returned in UTC
        self.assertEqual(row[8].astimezone(get_localzone()).strftime("%Y-%m-%d %H:%M:%S"), dateTimestampLocal.strftime("%Y-%m-%d %H:%M:%S"))

        row = cursor.next()
        self.assertEqual(row, None)

        # test fetchone
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")
        row = cursor.fetchone()
        self.assertEqual(row[0], 1)
        row = cursor.fetchone()
        self.assertEqual(row[0], 2)
        row = cursor.fetchone()
        self.assertEqual(row[0], 3)
        row = cursor.fetchone()
        self.assertEqual(row[0], 4)
        row = cursor.fetchone()
        self.assertEqual(row, None)

        # test fetchmany
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")
        row = cursor.fetchmany()
        self.assertEqual(len(row), 4)
        self.assertEqual(row[0][0], 1)
        self.assertEqual(row[1][0], 2)
        self.assertEqual(row[2][0], 3)
        self.assertEqual(row[3][0], 4)
        row = cursor.fetchmany()
        self.assertEqual(len(row), 0)

        # test fetchmany
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")
        row = cursor.fetchmany(2)
        self.assertEqual(len(row), 2)
        self.assertEqual(row[0][0], 1)
        self.assertEqual(row[1][0], 2)
        row = cursor.fetchmany(10)
        self.assertEqual(len(row), 2)
        self.assertEqual(row[0][0], 3)
        row = cursor.fetchmany(10)
        self.assertEqual(len(row), 0)

        # test fetchmany
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")
        row = cursor.fetchmany(100)
        self.assertEqual(len(row), 4)
        self.assertEqual(row[0][0], 1)
        self.assertEqual(row[1][0], 2)
        self.assertEqual(row[2][0], 3)
        self.assertEqual(row[3][0], 4)
        row = cursor.fetchmany(10)
        self.assertEqual(len(row), 0)

        # test fetchall
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")
        row = cursor.fetchall()
        self.assertEqual(len(row), 4)
        self.assertEqual(row[0][0], 1)
        self.assertEqual(row[1][0], 2)
        self.assertEqual(row[2][0], 3)
        self.assertEqual(row[3][0], 4)
        row = cursor.fetchall()
        self.assertEqual(len(row), 0)

        #test scroll
        cursor.execute("select * from TestTableWithSimpleTypes ORDER BY id")
        row = cursor.scroll(1, 'relative')
        self.assertEqual(row[0], 1)
        row = cursor.next();
        self.assertEqual(row[0], 2)
        row = cursor.scroll(-1, 'relative')
        self.assertEqual(row[0], 1)
        row = cursor.scroll(1, 'absolute')
        self.assertEqual(row[0], 2)

        row = cursor.scroll(-1, 'absolute')
        self.assertEqual(row[0], 3)

        row = cursor.scroll(-10, 'absolute')
        self.assertEqual(row, None)

        cursor.close()
        db.close()

        pass

    def test_TestTableWithComplexObject(self):

        # without import
        db = self._connect()

        self._testTestTableWithComplexObject(db, TestDsapi.createMyComplexObject, self.assertEquals)

        # with import
        db.import_semantic_type("MyComplexObject")
        def createMyComplexObject(id):
            return TestDsapi.createMyComplexObjectWithFactory(id, db.get_type_factory())
        def assertEquals(obj1, obj2):
            map1 = db.get_type_factory().writeToMap(obj1)
            map2 = db.get_type_factory().writeToMap(obj2)
            json1 = Utils.dumpMapToJson(map1)
            json2 = Utils.dumpMapToJson(map2)
            self.assertEquals(json1, json2)

        self._testTestTableWithComplexObject(db, createMyComplexObject, assertEquals)

        db.close()

    def _testTestTableWithComplexObject(self, db, createMyComplexObject, assertEquals):
        cursor = db.cursor()
        cursor.arraysize = -1

        cursor.execute("delete FROM TestTableWithComplexObject")
        cursor.execute("delete FROM TestTableWithComplexObject")
        self._assertUpdateCount(cursor, 0)

        # create table TestTableWithComplexObject(id int, obj MyComplexObject, primary key (id))
        cursor.execute("insert INTO TestTableWithComplexObject VALUES (?,?)", [1, createMyComplexObject(1)])
        self._assertUpdateCount(cursor, 1)
        cursor.execute("insert INTO TestTableWithComplexObject VALUES (?,?)", [2, createMyComplexObject(2)])
        self._assertUpdateCount(cursor, 1)
        cursor.execute("insert INTO TestTableWithComplexObject VALUES (?,?)", [3, createMyComplexObject(3)])
        self._assertUpdateCount(cursor, 1)

        # select * from TestTableWithComplexObject
        cursor.execute("select * from TestTableWithComplexObject ORDER BY id")
        self.assertEquals(len(cursor.description), 2)
        self.assertEquals(len(cursor.description[0]), 7)
        self.assertEquals(cursor.description[0][0], "id")
        self.assertEquals(cursor.description[0][1], ds.NUMBER)
        self.assertEquals(cursor.description[1][0], "obj")
        self.assertEquals(cursor.description[1][1], ds.STRING)
        self.assertEquals(cursor.rowcount, 3)

        row = cursor.next()
        self.assertEqual(row[0], 1)
        assertEquals(row[1], createMyComplexObject(1))

        row = cursor.next()
        self.assertEqual(row[0], 2)
        assertEquals(row[1], createMyComplexObject(2))

        row = cursor.next()
        self.assertEqual(row[0], 3)
        assertEquals(row[1], createMyComplexObject(3))

        row = cursor.next()
        self.assertEqual(row, None)

        cursor.close()

    @staticmethod
    def createMyComplexObject(id):
        import pytz
        return {
            '@type' : 'MyComplexObject',
            'i' : id,
            's' : 's' + str(id),
            'b' : True,

            'byteField' : id,
            'byteArrayField' : 'cXdl',
            'shortField' : id,
            'intField' : id,
            'longField' : id,
            'floatField' : float(id),
            'doubleField' : float(id),
            'booleanField' : True,
            'stringField' : "stringField" + str(id),

            'bigIntegerField' : 1000 + id,
            'bigDecimalField' : 12345.223 + id,

            'utilDateField' : int((datetime(2016, 12, 13, 15, 16, 17, 123) - datetime(1970, 1, 1)).total_seconds() * 1000),
            'sqlDateField' : date(2016, 12, 13).strftime("%Y-%m-%d"),
            'sqlTimeField' : time(15, 16, 17).strftime("%H:%M:%S"),
            'sqlTimestampField' : int((datetime(2016, 12, 13, 15, 16, 17, 123) - datetime(1970, 1, 1)).total_seconds() * 1000),

            'integerArray' : [id + 1,id + 2, id + 3],
            'integerList'  : [id + 5, id + 6],
            'subobject' : TestDsapi.createMyComplexSubObject(id + 1),
            'subobjectArray' : [
                TestDsapi.createMyComplexSubObject(id + 2),
                TestDsapi.createMyComplexSubObject(id + 3)
            ],
            'subobjectList' : [
                TestDsapi.createMyComplexSubObject(id + 4),
                TestDsapi.createMyComplexSubObject(id + 5)
            ],

            'subjectMap' : {
                # '@type' : 'map',
                'k6' : TestDsapi.createMyComplexSubObject(id + 6),
                'k7' : TestDsapi.createMyComplexSubObject(id + 7)
            },

            'subobjectNoSemType' : TestDsapi.createMyComplexSubObjectNoSemType(id + 6)
        }

    @staticmethod
    def createMyComplexSubObject(id):
        return {
            '@type' : 'MyComplexSubObject',
            's' : 's' + str(id)
        }

    @staticmethod
    def createMyComplexSubObjectNoSemType(id):
        return {
            '@type' : 'com.streamscape.sef.network.http.MyComplexSubObjectNoSemType',
            's' : 's' + str(id)
        }

    @staticmethod
    def createMyComplexObjectWithFactory(id, typeFactory):
        result = typeFactory.instantiateType("MyComplexObject")
        result.i = id
        result.s = 's' + str(id)
        result.b = True

        result.byteField = id
        result.byteArrayField = bytearray([11,12,13])
        result.shortField = id
        result.intField = id
        result.longField = id
        result.floatField = float(id)
        result.doubleField = float(id)
        result.booleanField = True
        result.stringField = "stringField" + str(id)

        result.bigIntegerField = 1000 + id
        result.bigDecimalField = 12345.223 + id

        result.utilDateField = datetime(2016, 12, 13, 15, 16, 17, 123)
        result.sqlDateField = date(2016, 12, 13)
        result.sqlTimeField = time(15, 16, 17)
        result.sqlTimestampField = datetime(2016, 12, 13, 15, 16, 17, 123)

        result.integerArray = [id + 1,id + 2, id + 3]
        result.integerList = [id + 5, id + 6]
        result.subobject = TestDsapi.createMyComplexSubObjectWithFactory(id + 1, typeFactory)
        result.subobjectArray = [
            TestDsapi.createMyComplexSubObjectWithFactory(id + 2, typeFactory),
            TestDsapi.createMyComplexSubObjectWithFactory(id + 3, typeFactory)
        ]
        result.subobjectList = [
            TestDsapi.createMyComplexSubObjectWithFactory(id + 4, typeFactory),
            TestDsapi.createMyComplexSubObjectWithFactory(id + 5, typeFactory)
        ]

        result.subjectMap = {
            'k6' : TestDsapi.createMyComplexSubObjectWithFactory(id + 6, typeFactory),
            'k7' : TestDsapi.createMyComplexSubObjectWithFactory(id + 7, typeFactory)
        }

        result.subobjectNoSemType = TestDsapi.createMyComplexSubObjectNoSemTypeWithFactory(id + 6, typeFactory)

        return result

    @staticmethod
    def createMyComplexSubObjectWithFactory(id, typeFactory):
        result = typeFactory.instantiateType("MyComplexSubObject")
        result.s = 's' + str(id)
        return result

    @staticmethod
    def createMyComplexSubObjectNoSemTypeWithFactory(id, typeFactory):
        result = typeFactory.instantiateType("com.streamscape.sef.network.http.MyComplexSubObjectNoSemType")
        result.s = 's' + str(id)
        return result
    
    def test_arraysize(self):
        db = self._connect()
        cursor = db.cursor()
        cursor.arraysize = -1

        cursor.execute("drop table arraysize_test_table if exists")
        cursor.execute("create table arraysize_test_table(id int, s string)")

        for i in range(0, 50):
            cursor.execute("insert into arraysize_test_table values ({}, '{}')".format(i, "q" + str(i)))

        # fetch all rows at first call
        cursor.arraysize = -1
        cursor.execute(" select * from arraysize_test_table ORDER BY id")
        self.assertEquals(type(cursor._rowset), RowSet)
        self.assertEquals(len(cursor._rowset._rows), 50)
        self.assertEquals(len(cursor.fetchall()), 50)
        self.assertEquals(cursor.rowcount, 50)

        # set fetch size to 50
        cursor.arraysize = 50
        cursor.execute(" select * from arraysize_test_table ORDER BY id")
        self.assertEquals(type(cursor._rowset), RowSet)
        self.assertEquals(len(cursor._rowset._rows), 50)
        self.assertEquals(len(cursor.fetchall()), 50)
        self.assertEquals(cursor.rowcount, 50)

        # set fetch size to 10
        cursor.arraysize = 10
        cursor.execute(" select * from arraysize_test_table ORDER BY id")
        self.assertEquals(type(cursor._rowset), RowSetProxy)
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(len(cursor.fetchall()), 50)
        self.assertEquals(cursor.rowcount, None)

        # set fetch size to 10 and test navigation methods
        cursor.arraysize = 10
        cursor.execute(" select * from arraysize_test_table ORDER BY id")
        self.assertEquals(type(cursor._rowset), RowSetProxy)
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(cursor.rowcount, None)
        for id in range(0, 15):
            row = cursor.next()
            self.assertEqual(row[0], id)

        for id in range(15, 21):
            row = cursor.fetchone()
            self.assertEqual(row[0], id)

        row = cursor.fetchmany()
        self.assertEquals(len(row), 10)
        self.assertEquals(row[0][0], 21)
        self.assertEquals(row[9][0], 30)

        row = cursor.fetchmany(3)
        self.assertEquals(len(row), 3)
        self.assertEquals(row[0][0], 31)
        self.assertEquals(row[2][0], 33)

        row = cursor.fetchmany(-1)
        self.assertEquals(len(row), 16)
        self.assertEquals(row[0][0], 34)
        self.assertEquals(row[15][0], 49)

        row = cursor.scroll(-9, "relative")
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(cursor._rowset._rows[0]._data[0], 40)
        self.assertEquals(cursor._rowset._rows[9]._data[0], 49)
        self.assertEquals(row[0], 41)

        row = cursor.scroll(-5, "relative")
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(cursor._rowset._rows[0]._data[0], 36)
        self.assertEquals(cursor._rowset._rows[9]._data[0], 45)
        self.assertEquals(row[0], 36)

        row = cursor.scroll(6, "relative")
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(cursor._rowset._rows[0]._data[0], 36)
        self.assertEquals(cursor._rowset._rows[9]._data[0], 45)
        self.assertEquals(row[0], 42)

        row = cursor.scroll(13, "absolute")
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(cursor._rowset._rows[0]._data[0], 13)
        self.assertEquals(cursor._rowset._rows[9]._data[0], 22)
        self.assertEquals(row[0], 13)

        row = cursor.scroll(0, "absolute")
        self.assertEquals(len(cursor._rowset._rows), 10)
        self.assertEquals(cursor._rowset._rows[0]._data[0], 0)
        self.assertEquals(cursor._rowset._rows[9]._data[0], 9)
        self.assertEquals(row[0], 0)

        cursor.execute("drop table arraysize_test_table")

        cursor.close()
        db.close();

        pass

    def test_autocommit(self):
        db = self._connect()
        self.assertEquals(db.get_autocommit(), True)

        # create table
        cursor = db.cursor()
        cursor.execute("drop TABLE test_table_for_autocommit if exists")
        cursor.execute("create TABLE test_table_for_autocommit (id int, s string)")

        # insert with autocommit True
        cursor.execute("insert into test_table_for_autocommit values(1, 'q1')")
        self._assertUpdateCount(cursor, 1)

        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 1)

        try:
            db.rollback()
            self.assertEquals(1, 0)
        except Exception as e:
            self.assertEquals(e.message, "Cannot rollback transaction when autocommit is enabled.")
        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 1)


        #insert with autocommit False and rollback
        db.autocommit(False)
        cursor.execute("insert into test_table_for_autocommit values(2, 'q2')")
        self._assertUpdateCount(cursor, 1)

        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 2)

        db.rollback()
        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 1)

        #insert with autocommit False and commit
        db.autocommit(False)
        cursor.execute("insert into test_table_for_autocommit values(3, 'q3')")
        self._assertUpdateCount(cursor, 1)

        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 2)

        db.commit()
        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 2)

        db.rollback()
        cursor.execute("select count(*) from test_table_for_autocommit")
        r = cursor.fetchall()
        self.assertEquals(r[0][0], 2)

        db.close()


    def _assertUpdateCount(self, cursor, count):
        self.assertEquals(len(cursor.description), 1)
        self.assertEquals(len(cursor.description[0]), 7)
        self.assertEquals(cursor.description[0][0], "UpdateCount")
        self.assertEquals(cursor.description[0][1], ds.NUMBER)
        self.assertEquals(cursor.rowcount, 1)

        row = cursor.fetchone()
        self.assertEquals(len(row), 1)
        self.assertEquals(row[0], count)

        row = cursor.fetchone()
        self.assertEquals(row, None)

    def _connect(self):
        return ds.connect(url ="http://localhost:8888",
                          username = "Admin", password = "Admin",
                          dataspace = "tspace.swagger_tspace", timeout = 30, autocommit = True)

