from __future__ import absolute_import

import logging

from stpy.fabric.utils import Utils
from stpy import HTTPFabricException
from .type_converter import TypeConverter

logger = logging.getLogger(__name__)


class TypeFactory(object):
    """
    Type factory is a registry of::
        - system types converters
        - semantic types imported from dataspace runtime
        - user defined converters for some types

    Once type is imported or registered in ``TypeFacotry``::
        - the object of this type can be created
        - the object of this type can be serialized to json
        - the object of this type can be deserialized from json

    """

    class ByteArrayConverter(TypeConverter):

        @staticmethod
        def readFromMap(rawValue, typeFactory=None):
            if rawValue == None:
                return None

            import base64
            value = base64.b64decode(rawValue)
            bytes = bytearray()
            bytes.extend(value)
            return bytes

        @staticmethod
        def writeToMap(value, typeFactory=None):
            if value == None:
                return None

            import base64
            encoded = base64.b64encode(value)
            if Utils.is_python3():
                encoded = encoded.decode('ascii');
            return encoded

    class DateConverter(TypeConverter):

        @staticmethod
        def readFromMap(rawValue, typeFactory=None):
            if rawValue == None:
                return None

            from .utils import Utils
            return Utils.fromMillisInUtc(float(rawValue))

        @staticmethod
        def writeToMap(value, typeFactory=None):
            if value == None:
                return None

            from .utils import Utils
            return Utils.toMillisInUtc(value)

    class SqlDateConverter(TypeConverter):

        @staticmethod
        def readFromMap(rawValue, typeFactory=None):
            if rawValue == None:
                return None

            from dateutil.parser import parse
            return parse(rawValue).date()

        @staticmethod
        def writeToMap(value, typeFactory=None):
            if value == None:
                return None

            return value.strftime("%Y-%m-%d")

    class SqlTimeConverter(TypeConverter):

        @staticmethod
        def readFromMap(rawValue, typeFactory=None):
            if rawValue == None:
                return None

            from dateutil.parser import parse
            return parse(rawValue).time()

        @staticmethod
        def writeToMap(value, typeFactory=None):
            if value == None:
                return None

            return value.strftime("%H:%M:%S")

    class SqlTimestampConverter(TypeConverter):

        @staticmethod
        def readFromMap(rawValue, typeFactory=None):
            if rawValue == None:
                return None

            from .utils import Utils
            return Utils.fromMillisInUtc(float(rawValue))

        @staticmethod
        def writeToMap(value, typeFactory=None):
            if value == None:
                return None

            from .utils import Utils
            return Utils.toMillisInUtc(value)

    from .row_set import RowSetConverter
    from .sl_response import SLResponseConverter
    from .lobs import RowSetClobProxyConverter, RowSetBlobProxyConverter

    _systemTypesMap = {
        "RowSet" : RowSetConverter,
        "RowSetProxy" : RowSetConverter,
        "SLResponse" : SLResponseConverter,
        "PromptSLResponse" : SLResponseConverter,
        "byte[]" : ByteArrayConverter,
        "bytearray" : ByteArrayConverter,
        "date" : DateConverter,
        "SqlDate" : SqlDateConverter,
        "SqlTime" : SqlTimeConverter,
        "SqlTimestamp" : SqlTimestampConverter,

        "RowSetBlobProxy" : RowSetBlobProxyConverter,
        "RowSetClobProxy" : RowSetClobProxyConverter
    }

    def __init__(self):
        self._userTypesMap = {}

    def registerType(self, name, converter):
        """
        Registers user defined converter for type with specified name.
        Converter should implement interface ``TypeConverter``
        :param name: type name
        :param converter: converter should implement interface ``TypeConverter``
        """
        self._userTypesMap[name] = converter

    def unregisterType(self, name):
        """
        Unregister converter for type with specified name.
        :param name: type name
        :return:
        """
        del self._userTypesMap[name]

    def _buildAndRegisterUserTypeFromJsonMap(self, map):
        if type(map) != dict:
            return

        typeName = map.get("@type", None)
        if typeName is None:
            raise HTTPFabricException("Map {} doesn't contains @type.".format(map))

        if TypeFactory._systemTypesMap.get(typeName, None) is not None or self._userTypesMap.get(typeName, None) is not None:
            return

        typeName = str(typeName)
        model = {}
        for key, value in map.items():
            if key == '@type':
                continue
            if type(value) == dict:
                self._buildAndRegisterUserTypeFromJsonMap(value)
                model[key] = {}
            elif type(value) == list:
                if len(value) > 0:
                    self._buildAndRegisterUserTypeFromJsonMap(value[0])
                model[key] = []
            else:
                model[key] = value

        typeConverter = TypeConverterUntyped(typeName)
        typeConverter._typeClass = type(typeName, (), model)

        self._userTypesMap[typeName] = typeConverter
        pass

    def _buildAndRegisterUserTypeFromTypeMap(self, map):
        if type(map) != dict:
            return

        typeName = map.get("typeName", None)
        if typeName is None:
            raise HTTPFabricException("Map {} doesn't contains typeName.".format(map))

        if TypeFactory._systemTypesMap.get(typeName, None) is not None or self._userTypesMap.get(typeName, None) is not None:
            return

        if map.get("fields", None) is None:
            return

        if typeName == "list" or typeName.endswith("[]"):
            fields = map.get("fields", [])
            if len(fields) > 0:
                self._buildAndRegisterUserTypeFromTypeMap(fields[0])
            return

        if typeName == "map":
            fields = map.get("fields", [])
            if len(fields) > 0:
                for field in fields:
                    if type(field) == dict and field.get('fieldName', None) == "value":
                        self._buildAndRegisterUserTypeFromTypeMap(field)
            return


        typeName = str(typeName)
        model = {}
        typeConverter = TypeConverterTyped(typeName)

        for field in map.get("fields", []):
            fieldName = field.get("fieldName")
            fieldTypeName  = field.get("typeName")

            self._buildAndRegisterUserTypeFromTypeMap(field)
            model[fieldName] = self.typeDefault(fieldTypeName)
            typeConverter._fields[fieldName] = fieldTypeName

        typeConverter._typeClass = type(typeName, (), model)

        self._userTypesMap[typeName] = typeConverter
        pass

    def typeDefault(self, typeName):
        return None

    def instantiateType(self, name):
        """
        Create an instance of object with specified type name.
        :param name: type name
        :return: object instance
        """
        converter = TypeFactory._systemTypesMap.get(name, None)
        if converter == None:
            converter = self._userTypesMap.get(name, None)

        if converter == None:
            raise HTTPFabricException("Converter for type '" + name + "' not found.")

        return converter.typeClass(name)()


    def readFromMap(self, rawValue, expectedType = None, raiseExceptionIfConverterNotFound = True):
        """
        Deserializes object instance from raw value. Raw value can be a dictionary or primitive value.
          - if raw value is primitive and no ``expectedType`` specified the same value will be returned.
          - if raw value is dictionary then value of @type key element will be used ar value type
          - if ``expectedType`` is specified then it will be used as value type
        Converter corresponding to value type will be used for deserialization.

        :param rawValue: primitive value or dictionary
        :param expectedType: expected type of value
        :param raiseExceptionIfConverterNotFound: raise exception if converter not found or not
        :return: deserialized object instance
        """
        if rawValue == None:
            return None

        if type(rawValue) == expectedType:
            return rawValue

        # handle arrays
        if isinstance(rawValue, list):
            if expectedType != None and expectedType.endswith("[]"):
                expectedType = expectedType[0:len(expectedType) - 2]
            else:
                expectedType = None
            result = []
            for v in rawValue:
                result.append(self.readFromMap(v, expectedType, False))
            return result

        rawValueType = None
        # check if it is map and contains @type fields
        if isinstance(rawValue, dict):
            rawValueType = rawValue.get('@type', None)

        if rawValueType == "map":
            result = {}
            for key, value in rawValue.items():
                if key == "@type":
                    continue
                result[key] = self.readFromMap(value, None, raiseExceptionIfConverterNotFound)
            return result

        if rawValueType == None:
            rawValueType = expectedType
        if rawValueType == None:
            if raiseExceptionIfConverterNotFound:
                raise HTTPFabricException("Object @type or expectedType not specified.")
            else:
                return rawValue

        converter = TypeFactory._systemTypesMap.get(rawValueType, None)
        if converter == None:
            converter = self._userTypesMap.get(rawValueType, None)

        if converter == None:
            if raiseExceptionIfConverterNotFound:
                raise HTTPFabricException("Converter for type '" + rawValueType + "' not found.")
            else:
                return rawValue

        return getattr(converter, 'readFromMap')(rawValue, self)

    def writeToMap(self, value, valueType = None, raiseExceptionIfConverterNotFound = True, ):
        """
        Serializes specified value to dictionary.
        If ``valueType`` is not specified it will be calculated as ``type(value).__name__``
        Converter corresponding to ``valueType`` will be used for serialization.

        :param value: value to serialize
        :param valueType: value type
        :param raiseExceptionIfConverterNotFound: raise exception if converter not found or not
        :return: dictionary with serialized value
        """
        if value == None:
            return None
        if type(value) == dict:
            result = {}
            for k, v in value.items():
                result[k] = self.writeToMap(v, raiseExceptionIfConverterNotFound=raiseExceptionIfConverterNotFound)
            return result

        # handler arrays
        if isinstance(value, list): # or isinstance(value, tuple):
            result = []
            for v in value:
                result.append(self.writeToMap(v, raiseExceptionIfConverterNotFound=raiseExceptionIfConverterNotFound))
            return result

        if valueType == None:
            valueType = type(value).__name__
        converter = TypeFactory._systemTypesMap.get(valueType, None)
        if converter == None:
            converter = self._userTypesMap.get(valueType, None)

        if converter == None:
            if raiseExceptionIfConverterNotFound:
                raise HTTPFabricException("Converter for type '" + valueType + "' not found.")
            else:
                return value

        return getattr(converter, 'writeToMap')(value, self)


class TypeConverterUntyped(TypeConverter):

    def __init__(self, name):
        self._name = name
        self._typeClass = None
        self._readFromMap = None

    def typeClass(self, name = None):
        return self._typeClass

    def readFromMap(self, map, expectedType = None):
        return self._readFromMap(map, expectedType)

class TypeConverterTyped(TypeConverter):

    def __init__(self, name):
        self._name = name
        self._fields = {}
        self._typeClass = None
        self._readFromMap = None
        self._writeFromMap = None

    def typeClass(self, name = None):
        return self._typeClass

    def readFromMap(self, rawValue, typeFactory=None):
        if self._readFromMap is not None:
            return self._readFromMap(rawValue, typeFactory)

        if not isinstance(rawValue, dict):
            raise HTTPFabricException("Failed to serialize raw value {} to type {}. Raw value is not map.", rawValue, self._name)

        result = self.typeClass()()
        for fieldName, fieldValue in rawValue.items():
            if fieldName == "@type":
                continue

            fieldType = self._fields.get(fieldName, None)
            if fieldType is None:
                _log("Unknown field {} with value {} in type {}", fieldName, fieldValue, self._name)
                continue

            fieldValue = typeFactory.readFromMap(fieldValue, fieldType, False)
            setattr(result, fieldName, fieldValue)

        return result

    def writeToMap(self, value, typeFactory=None):
        if self._writeFromMap is not None:
            return self._writeFromMap(value, typeFactory)

        if isinstance(value, dict):
            return value

        if type(value) != self.typeClass():
            raise HTTPFabricException("Cannot write type {} in converter of type {}.", type(value), self.typeClass())

        result = {
            "@type" : type(value).__name__
        }
        for fieldName, fieldType in self._fields.items():

            fieldValue = getattr(value, fieldName)
            fieldValue = typeFactory.writeToMap(fieldValue, valueType=fieldType, raiseExceptionIfConverterNotFound=False)
            result[fieldName] = fieldValue

        return result


def _log(self, func, message, *args):
    from .utils import Utils
    Utils.log(func, message, args)
