from __future__ import absolute_import

import logging
import threading

from stpy.fabric.http_sl_session import HTTPSLSession

from .utils import Utils
from .enums import SLFileSessionContext

logger = logging.getLogger(__name__)


class HTTPDataspaceAccessor(object):
    accessorId = 0
    lock = threading.RLock()

    def __init__(self, dataspaceType, dataspaceName, connection, nodeName=None):
        self._nodeName = nodeName
        self._dataspaceType = dataspaceType
        self._connection = connection
        self._dataspaceName = dataspaceName
        self._isOpened = False

        self._requestTimeout = -2
        self._sessionContext = SLFileSessionContext.CLIENT
        self._autocommit = None
        self._autoSwitching = None

        self._readOnly = None
        self._transactionIsolation = None
        self._transferBufferSize = None
        self._fetchSize = None
        self._downloadableLobSize = None

        with HTTPDataspaceAccessor.lock:
            HTTPDataspaceAccessor.accessorId += 1
            self._uniqueAccessorName = "fabricconnection_" + str(HTTPDataspaceAccessor.accessorId)

        self._getDataspaceAccessorMethod = Utils.method("getDataspaceAccessor", [self._uniqueAccessorName])
        self._httpClientWrapper = None

        self.open()

    def open(self):
        if self._httpClientWrapper:
            self._httpClientWrapper.closeQuiet()

        self._httpClientWrapper = self._connection._createHttpClientWrapper()

        call = Utils.remote([Utils.method("createDataspaceAccessor",
                                          [
                                              self._nodeName,
                                              Utils.enum("DataspaceType", self._dataspaceType),
                                              self._dataspaceName,
                                              self._uniqueAccessorName
                                          ])])
        r = self._connection._invokeCall(self._httpClientWrapper, call)
        o = Utils.parseJsonResponseAndCheck(r)

        self._connection._dataspaceAccessors[self._uniqueAccessorName] = self

        self._isOpened = True

        try:
            self.setSessionContext(self._sessionContext)
        except Exception as exception:
            self._log(logger.error, "Failed to set accessor session context to client, for %s", self.getName())
            self._log(logger.exception, exception)

        _self = self
        if self._requestTimeout != -2:
            self._connection._reopenAction(lambda: _self.setRequestTimeout(_self._requestTimeout))

        if self._autoSwitching != None:
            self._connection._reopenAction(lambda: _self.setAutoSwitching(_self._autoSwitching))
        if self._autocommit != None:
            self._connection._reopenAction(lambda: _self.setAutoCommit(_self._autocommit))

        if self._readOnly != None:
            self._connection._reopenAction(lambda: _self.setReadOnly(_self._readOnly))
        if self._transactionIsolation != None:
            self._connection._reopenAction(lambda: _self.setTransactionIsolation(_self._transactionIsolation))
        if self._transferBufferSize != None:
            self._connection._reopenAction(lambda: _self.setTransferBufferSize(_self._transferBufferSize))
        if self._fetchSize != None:
            self._connection._reopenAction(lambda: _self.setFetchSize(_self._fetchSize))

        if self._downloadableLobSize != None:
            self._connection._reopenAction(lambda: _self.setDownloadableLobSize(_self._downloadableLobSize))

    def close(self):
        try:
            self._invokeMethod("close")
        finally:
            del self._connection._dataspaceAccessors[self._uniqueAccessorName]
            self._isOpened = False

        if self._httpClientWrapper != None:
            self._httpClientWrapper.closeQuiet()
            self._httpClientWrapper = None

    def getName(self):
        self._checkOpened()
        r = self._invokeMethod("getName")
        o = Utils.parseJsonResponseAndCheck(r, str)
        return o

    def isOpened(self):
        if not self._connection.isOpened() or self._httpClientWrapper == None:
            return False

        r = self._invokeMethod("isOpened")
        o = Utils.parseJsonResponseAndCheck(r, bool)

        return o

    def isAvailable(self):
        if not self._connection.isOpened() or self._httpClientWrapper == None:
            return False

        r = self._invokeMethod("isAvailable")
        o = Utils.parseJsonResponseAndCheck(r, bool)
        return o

    def autoSwitching(self):
        self._checkOpened()
        r = self._invokeMethod("autoSwitching")
        o = Utils.parseJsonResponseAndCheck(r, bool)
        return o

    def setAutoSwitching(self, autoSwitching):
        self._checkOpened()
        r = self._invokeMethod("setAutoSwitching", [autoSwitching])
        Utils.parseJsonResponseAndCheck(r)
        self._autoSwitching = autoSwitching
        return

    def getRequestTimeout(self):
        self._checkOpened()
        if self._requestTimeout == -2:
            r = self._invokeMethod("getRequestTimeout")
            self._requestTimeout = Utils.parseJsonResponseAndCheck(r, int)
        return self._requestTimeout

    def setRequestTimeout(self, timeout):
        self._checkOpened()
        r = self._invokeMethod("setRequestTimeout", [timeout])
        Utils.parseJsonResponseAndCheck(r)
        self._requestTimeout = timeout
        return

    def setReadOnly(self, readOnly):
        self._checkOpened()
        r = self._invokeMethod("setReadOnly", [readOnly])
        Utils.parseJsonResponseAndCheck(r)
        self._readOnly = readOnly

    def isReadOnly(self):
        self._checkOpened()
        r = self._invokeMethod("isReadOnly")
        o = Utils.parseJsonResponseAndCheck(r, bool)
        return o

    def setTransactionIsolation(self, level):
        self._checkOpened()
        r = self._invokeMethod("setTransactionIsolation", [level])
        Utils.parseJsonResponseAndCheck(r)
        self._transactionIsolation = level

    def getTransactionIsolation(self):
        self._checkOpened()
        r = self._invokeMethod("getTransactionIsolation")
        o = Utils.parseJsonResponseAndCheck(r, int)
        return o

    def setTransferBufferSize(self, transferBufferSize):
        self._checkOpened()
        r = self._invokeMethod("setTransferBufferSize", [transferBufferSize])
        Utils.parseJsonResponseAndCheck(r)
        self._transferBufferSize = transferBufferSize

    def getTransferBufferSize(self):
        self._checkOpened()
        r = self._invokeMethod("getTransferBufferSize")
        o = Utils.parseJsonResponseAndCheck(r, int)
        return o

    def setFetchSize(self, fetchSize):
        self._checkOpened()
        r = self._invokeMethod("setFetchSize", [fetchSize])
        Utils.parseJsonResponseAndCheck(r)
        self._fetchSize = fetchSize

    def getFetchSize(self):
        self._checkOpened()
        r = self._invokeMethod("getFetchSize")
        self._fetchSize = Utils.parseJsonResponseAndCheck(r, int)
        return self._fetchSize

    def setDownloadableLobSize(self, downloadableLobSize):
        self._checkOpened()
        r = self._invokeMethod("setDownloadableBlobSize", [downloadableLobSize])
        Utils.parseJsonResponseAndCheck(r)
        self._downloadableLobSize = downloadableLobSize

    def getDownloadableLobSize(self):
        self._checkOpened()
        r = self._invokeMethod("getDownloadableBlobSize")
        o = Utils.parseJsonResponseAndCheck(r, int)
        return o

    def setSessionContext(self, sessionContext):
        # TODO: check why short semantic type name doesn't work
        self._checkOpened()
        r = self._invokeMethod("setSessionContext", [Utils.enum("com.streamscape.slex.file.SLFileSessionContext", sessionContext)])
        Utils.parseJsonResponseAndCheck(r)
        self._sessionContext = sessionContext
        return

    def getSessionContext(self):
        self._checkOpened()
        r = self._invokeMethod("getSessionContext")
        o = Utils.parseJsonResponseAndCheck(r, "SLFileSessionContext")
        return o['value']

    def setAutoCommit(self, autocommit):
        self._checkOpened()
        r = self._invokeMethod("setAutoCommit", [autocommit])
        Utils.parseJsonResponseAndCheck(r)
        self._autocommit = autocommit
        return

    def getAutoCommit(self):
        self._checkOpened()
        r = self._invokeMethod("getAutoCommit")
        o = Utils.parseJsonResponseAndCheck(r, bool)
        return o

    def commit(self):
        self._checkOpened()
        self._invokeMethod("commit")

    def rollback(self):
        self._checkOpened()
        self._invokeMethod("rollback")

    def executeQuery(self, query, args = []):
        self._checkOpened()
        arguments = [query]
        if args is not None and len(args) > 0:
            args = [Utils.convertPythonToDs(arg, self._connection.getTypeFactory()) for arg in args]
            arguments.append(args)
        r = self._invokeMethod("executeQuery", arguments, timeout=Utils.calculateHttpTimeout(-1, self._requestTimeout))
        o = Utils.parseJsonResponseAndCheck(r)
        o = self._connection.getTypeFactory().readFromMap(o, "RowSet")
        o = Utils.processAccessibleObjects(o, self)
        return o

    def invokeLanguageRequest(self, statement, timeout=None):
        self._checkOpened()
        if timeout == None:
            r = self._invokeMethod("invokeLanguageRequest", [statement], timeout=Utils.calculateHttpTimeout(-1, self._requestTimeout))
        else:
            r = self._invokeMethod("invokeLanguageRequest",
                                   [statement, HTTPSLSession.calculateSlangTimeout(self.connection, timeout, self._requestTimeout)],
                                   timeout=Utils.calculateHttpTimeout(timeout, self._requestTimeout))

        o = Utils.parseJsonResponseAndCheck(r)
        o = self._connection.getTypeFactory().readFromMap(o, "SLResponse")
        o = Utils.processAccessibleObjects(o, self)
        return o

    def _raiseSystemRequest(self, request, timeout):
        args = [request, HTTPSLSession._calculateSlangTimeout(self._connection, timeout, self._requestTimeout)]
        result = self._invokeMethod("raiseSystemRequest", args, timeout)
        result = Utils.parseJsonResponseAndCheck(result)
        return result

    def _invokeMethod(self, methodName, args=[], timeout=-1):
        return self._connection._invokeCall(self._httpClientWrapper, Utils.remote([self._getDataspaceAccessorMethod, Utils.method(methodName, args)]),
                                            timeout=timeout)

    def _log(self, func, message, *args):
        if self._connection:
            self._connection._log(func, message, *args)
        else:
            Utils.log(func, message, args)

    def _checkOpened(self):
        if not self._isOpened:
            from stpy import HTTPFabricException
            raise HTTPFabricException("This operation is not allowed on closed accessor.")
