from __future__ import absolute_import

import logging

from stpy import HTTPFabricConnection
from stpy.ds.exceptions import *
from stpy.ds.cursor import Cursor

logger = logging.getLogger(__name__)


def defaulterrorhandler(connection, cursor, errorclass, errorvalue):
    """
    If cursor is not None, (errorclass, errorvalue) is appended to
    cursor.messages; otherwise it is appended to
    connection.messages. Then errorclass is raised with errorvalue as
    the value.

    You can override this with your own error handler by assigning it
    to the instance.

    """
    error = errorclass, errorvalue
    if cursor:
        cursor.messages.append(error)
    else:
        connection.messages.append(error)
    # del cursor
    # del connection
    raise errorvalue


class Connection(object):
    """ """
    # TODO:
    errorhandler = defaulterrorhandler

    def __init__(self, url, dataspace, username, password, autocommit=True, timeout=30, ssl=None):
        parts = dataspace.split(".")
        if len(parts) != 2:
            raise ProgrammingError("Invalid dataspace name '{}' specified. Should be in format <type>.<name>.".format(dataspace))

        self._dataspace = dataspace
        self._dataspace_type = parts[0].upper()
        self._dataspace_name = parts[1]

        self._connection = HTTPFabricConnection(url, username, password)
        self._connection.setHttpTimeoutMs(timeout * 1000)
        self._autocommit = autocommit

        # TODO:
        self.messages = []

        try:
            self._connection.open()
        except Exception as e:
            raise OperationalError("Open connection failed.", e)

        try:
            self._accessor = self._connection.createDataspaceAccessor(None, parts[0].upper(), parts[1])
        except Exception as e:
            self.close()
            raise OperationalError("Cannot connect to dataspace {}.".format(dataspace), e)

        try:
            if not self._accessor.isAvailable():
                self.close()
                raise OperationalError("Dataspace {} is not available.".format(dataspace))
        except OperationalError as e:
            self.close();
            raise e
        except Exception as e:
            self.close()
            raise OperationalError("Dataspace {} is not available.".format(dataspace), e)

        try:
            self._accessor.setAutoCommit(autocommit)
        except Exception as e:
            self.close()
            raise OperationalError("Failed to set autocommit to {}.".format(autocommit))

    def close(self):
        """
            Close the connection now (rather than whenever .__del__() is called).

            The connection will be unusable from this point forward;
            an Error (or subclass) exception will be raised if any operation is attempted with the connection.
            The same applies to all cursor objects trying to use the connection.
            Note that closing a connection without committing the changes first will cause an implicit rollback to be performed.
        """

        if self._connection:
            try:
                self._connection.close()
            except Exception as e:
                pass
            self._connection = None
            self._accessor = None

    def cursor(self):
        """
            Return a new Cursor Object using the connection.
            :return: Cursor
        """
        self._check_opened()
        return Cursor(self)

    def get_autocommit(self):
        """
            Returns autocommit enabled/disable.
            :return: bool
        """
        return self._autocommit

    def autocommit(self, on):
        """
            Enables/disables autocommit.

            :param on: bool
        """
        on = bool(on)
        if self.get_autocommit() != on:
            if self._accessor is not None:
                try:
                    self._accessor.setAutoCommit(on)
                    self._autocommit = on
                except Exception as e:
                    raise OperationalError("Set autocommit operation failed.", e)
            else:
                self._autocommit = on

    def commit(self):
        """
            Commits any pending transaction to the database.
            If there is no any started transaction this method successfully does nothing.
        """
        self._check_opened()
        try:
            self._accessor.commit()
        except Exception as e:
            raise OperationalError("Commit operation failed.", e)

    def rollback(self):
        """
            Rollback dataspace transaction if any started.
            If autocommit is enabled raises exception.
        """
        self._check_opened()
        if self._autocommit:
            raise OperationalError("Cannot rollback transaction when autocommit is enabled.")
        try:
            self._accessor.rollback()
        except Exception as e:
            raise OperationalError("Rollback operation failed.", e)

    def import_semantic_type(self, name):
        """
        Imports semantic type from dataspace runtime to this connection.
        After semantic type has been imported objects of this type will be automatically
        serialized to object of this type and objects of this type can be created using ``TypeFactory``.
        :param name: semantic type name
        """
        self._check_opened()
        try:
            self._connection.importSemanticType(name)
        except Exception as e:
            raise OperationalError("Import operation failed.", e)

    def set_downloadable_lob_size(self, size):
        """
        Sets maximum downloadable lob size in bytes.
        This size defines maximum amount of bytes that should be
        downloaded to client on first call.

        -1 means download full lob data on first call
        :param size:
        :return:
        """
        self._accessor.setDownloadableLobSize(size)

    def get_downloadable_lob_size(self):
        """
        Returns max downloadable lob size in bytes.
        :param size:
        :return:
        """
        self._accessor.getDownloadableLobSize()

    def get_type_factory(self):
        """
        Returns ``TypeFactory`` reference.
        :return: ``TypeFactory`` reference
        """
        return self._connection.getTypeFactory()

    def __exit__(self, exc, value, tb):
        try:
            if exc:
                self.rollback()
            else:
                self.commit()
        except Exception as e:
            logger.error("failed to exit connection")
            logger.error(e)
            pass

    def _query(self, query, args=None, fetchSize=None):
        self._check_opened()
        try:
            if fetchSize is not None and fetchSize < 0:
                fetchSize = 0
            if fetchSize is not None and fetchSize >= 0:
                if self._accessor._fetchSize != fetchSize: # fetch size for this accessor can be changed here only
                    self._accessor.setFetchSize(fetchSize)

            if args is None:
                return self._accessor.executeQuery(query)
            else:
                return self._accessor.executeQuery(query, args)
        except Exception as e:
            raise OperationalError("Query failed.", e)

    def _check_opened(self):
        if self._accessor is None:
            raise OperationalError("Connection is closed.")
