/*
 * Decompiled with CFR 0.152.
 */
package com.streamscape.lib.evfs;

import com.streamscape.Trace;
import com.streamscape.lib.evfs.DirectorySnapshotListener;
import com.streamscape.lib.evfs.FileChangeListener;
import com.streamscape.lib.evfs.FileSortStrategy;
import com.streamscape.sdo.SDOException;
import com.streamscape.sdo.SecurityViolationException;
import com.streamscape.sdo.enums.FileEventType;
import com.streamscape.sdo.enums.FileState;
import com.streamscape.sdo.file.FileMetaInfo;
import com.streamscape.sdo.mf.admin.DatagramFactoryException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

public class BaseDirectoryMonitor
implements Runnable {
    protected File rootDirectory;
    protected String monitorName;
    protected List<FileMetaInfo> snapshot;
    protected List<FileChangeListener> fileChangeListeners;
    protected List<DirectorySnapshotListener> snapshotListeners;
    protected DirectoryState state;
    protected boolean started;
    protected boolean recursiveScan;
    protected boolean existingAsCreated;
    protected boolean sendSnapshots;
    protected boolean notifyDirectories;
    protected boolean isLockedStatesEnabled = true;
    protected FileSortStrategy sortStrategy = FileSortStrategy.SERVER_DEFAULT;
    private volatile boolean isProcessing;
    private final Object isProcessingLock = new Object();

    public BaseDirectoryMonitor(String name, String directory, boolean recursiveScan, FileSortStrategy sortStrategy, boolean existingAsCreated) throws IllegalArgumentException {
        if (name == null) {
            throw new IllegalArgumentException("Name cannot be null.");
        }
        if (directory == null) {
            throw new IllegalArgumentException("Directory cannot be null.");
        }
        this.monitorName = name;
        this.recursiveScan = recursiveScan;
        this.existingAsCreated = existingAsCreated;
        this.sortStrategy = sortStrategy;
        this.rootDirectory = new File(directory);
        if (!this.rootDirectory.exists()) {
            throw new IllegalArgumentException("Directory '" + this.rootDirectory.getAbsolutePath() + "' does not exist.");
        }
        if (!this.rootDirectory.isDirectory()) {
            this.rootDirectory = null;
            throw new IllegalArgumentException("'" + this.rootDirectory.getAbsolutePath() + "' is not a directory.");
        }
        this.state = DirectoryState.AVAILABLE;
        this.snapshotListeners = new ArrayList<DirectorySnapshotListener>();
        this.fileChangeListeners = new ArrayList<FileChangeListener>();
        this.initSnapshot();
        this.started = false;
        this.isProcessing = false;
    }

    protected void initSnapshot() {
        this.snapshot = this.existingAsCreated ? new ArrayList() : this.getFileObjects();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        Object object = this.isProcessingLock;
        synchronized (object) {
            this.isProcessing = true;
            this.isProcessingLock.notifyAll();
        }
        try {
            this.process();
        }
        catch (Exception exception) {
            this.logException(exception, true);
        }
        finally {
            Object object2 = this.isProcessingLock;
            synchronized (object2) {
                this.isProcessing = false;
                this.isProcessingLock.notifyAll();
            }
        }
    }

    public boolean isNotifyDirectories() {
        return this.notifyDirectories;
    }

    public void resetSnapshot() {
        this.snapshot = this.getFileObjects();
    }

    public void setNotifyDirectories(boolean notifyDirectories) {
        if (this.notifyDirectories != notifyDirectories) {
            this.notifyDirectories = notifyDirectories;
            this.snapshot = this.getFileObjects();
        }
    }

    public boolean isLockedStatesEnabled() {
        return this.isLockedStatesEnabled;
    }

    public void setLockedStatesEnabled(boolean lockedStatesEnabled) {
        this.isLockedStatesEnabled = lockedStatesEnabled;
    }

    public void start() {
        this.logDebug("Directory monitor '" + this.monitorName + "' started.");
        this.started = true;
    }

    public void suspend() {
        this.started = false;
    }

    public boolean isStarted() {
        return this.started;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void waitProcessingFinished() {
        Object object = this.isProcessingLock;
        synchronized (object) {
            while (this.isProcessing) {
                try {
                    this.isProcessingLock.wait();
                }
                catch (Exception exception) {}
            }
        }
    }

    public void stop() {
    }

    public List<FileMetaInfo> getFileObjects() {
        return this.rootDirectory != null ? this.getFileObjects(this.rootDirectory) : new ArrayList();
    }

    public synchronized ConcurrentHashMap<String, FileMetaInfo> getWatchList() {
        if (this.snapshot == null) {
            return null;
        }
        ConcurrentHashMap<String, FileMetaInfo> map = new ConcurrentHashMap<String, FileMetaInfo>(this.snapshot.size());
        for (FileMetaInfo fileState : this.snapshot) {
            map.put(fileState.getName(), new FileMetaInfo(new File(fileState.getPath())));
        }
        return map;
    }

    protected List<FileMetaInfo> getFileObjects(File root) {
        ArrayList<FileMetaInfo> result = new ArrayList<FileMetaInfo>();
        String[] fileNames = root.list();
        if (fileNames != null) {
            for (String fileName : fileNames) {
                FileMetaInfo fileObject;
                File file = new File(root.getAbsolutePath() + "/" + fileName);
                if (file.isFile()) {
                    fileObject = new FileMetaInfo(file);
                    result.add(fileObject);
                    continue;
                }
                if (!file.isDirectory()) continue;
                if (this.notifyDirectories) {
                    fileObject = new FileMetaInfo(file);
                    result.add(fileObject);
                }
                if (!this.recursiveScan) continue;
                List<FileMetaInfo> objects = this.getFileObjects(file);
                if (objects == null) {
                    return null;
                }
                result.addAll(objects);
            }
        } else {
            return null;
        }
        switch (this.sortStrategy) {
            case MODIFICATION_TIME: {
                Collections.sort(result, new Comparator<FileMetaInfo>(this){

                    @Override
                    public int compare(FileMetaInfo o1, FileMetaInfo o2) {
                        return new Long(o1.getLastModifiedTime()).compareTo(o2.getLastModifiedTime());
                    }
                });
                break;
            }
        }
        return result;
    }

    public void process() throws Exception {
        List<FileMetaInfo> newSnapshot;
        File checkAvailability;
        if (!this.started) {
            return;
        }
        if (this.snapshot == null) {
            this.initSnapshot();
            if (this.snapshot == null) {
                Trace.logError(this, "Failed to initialize files snapshot due to I/O errors.");
                return;
            }
        }
        if (this.isLockedStatesEnabled) {
            this.logDebug("'" + this.monitorName + "' is looking for updates...");
        }
        if (!(checkAvailability = new File(this.rootDirectory.getPath())).exists()) {
            this.state = DirectoryState.UNAVAILABLE;
            FileMetaInfo fileObject = new FileMetaInfo(checkAvailability);
            fileObject.setEventType(FileEventType.DIRECTORY_UNAVAILABLE);
            this.raiseEvent(fileObject);
            this.logDebug("'" + this.monitorName + "' looking for updates finished (root directory not found).");
            return;
        }
        if (this.state == DirectoryState.UNAVAILABLE) {
            this.state = DirectoryState.AVAILABLE;
            FileMetaInfo fileObject = new FileMetaInfo(checkAvailability);
            fileObject.setEventType(FileEventType.DIRECTORY_AVAILABLE);
            this.raiseEvent(fileObject);
        }
        if ((newSnapshot = this.getFileObjects()) == null) {
            Trace.logError(this, "Failed to get new files snapshot due to I/O errors.");
            return;
        }
        for (FileMetaInfo fileObject : this.snapshot) {
            if (newSnapshot.contains(fileObject)) continue;
            this.logDebug("Deleted file '" + fileObject.getAbsoluteFileName() + "' detected.");
            fileObject.setLastState(FileState.DELETED);
            fileObject.setSizeDelta(-fileObject.getSize());
            this.raiseEvent(fileObject);
        }
        for (FileMetaInfo fileObject : newSnapshot) {
            FileMetaInfo oldFileObject;
            int index = this.snapshot.indexOf(fileObject);
            FileMetaInfo fileMetaInfo = oldFileObject = index != -1 ? this.snapshot.get(index) : null;
            if (oldFileObject != null) {
                if (oldFileObject.getLastState() == FileState.CREATING) {
                    if (this.isFileFree(fileObject)) {
                        this.logDebug("New file '" + fileObject.getAbsoluteFileName() + "' detected.");
                        fileObject.setLastState(FileState.CREATED);
                        fileObject.setSizeDelta(fileObject.getSize() - oldFileObject.getSize());
                        this.raiseEvent(fileObject);
                        continue;
                    }
                    this.logDebug("File '" + fileObject.getAbsoluteFileName() + "' is still creating...");
                    fileObject.setLastState(FileState.CREATING);
                    continue;
                }
                if (oldFileObject.getLastModifiedTime() != fileObject.getLastModifiedTime()) {
                    if (this.isFileFree(fileObject)) {
                        this.logDebug("Modified file '" + fileObject.getAbsoluteFileName() + "' detected.");
                        fileObject.setLastState(FileState.MODIFIED);
                        fileObject.setSizeDelta(fileObject.getSize() - oldFileObject.getSize());
                        this.raiseEvent(fileObject);
                        continue;
                    }
                    fileObject.setLastState(FileState.MODIFYING);
                    if (oldFileObject.getLastState() == FileState.MODIFYING) continue;
                    this.logDebug("Modifying file '" + fileObject.getAbsoluteFileName() + "' detected.");
                    fileObject.setSizeDelta(fileObject.getSize() - oldFileObject.getSize());
                    this.raiseEvent(fileObject);
                    continue;
                }
                if (oldFileObject.getLastState() != FileState.MODIFYING) continue;
                if (this.isFileFree(fileObject)) {
                    this.logDebug("Modified file '" + fileObject.getAbsoluteFileName() + "' detected.");
                    fileObject.setLastState(FileState.MODIFIED);
                    fileObject.setSizeDelta(fileObject.getSize() - oldFileObject.getSize());
                    this.raiseEvent(fileObject);
                    continue;
                }
                fileObject.setLastState(FileState.MODIFYING);
                continue;
            }
            if (this.snapshot.contains(fileObject)) continue;
            if (this.isFileFree(fileObject)) {
                this.logDebug("New file '" + fileObject.getAbsoluteFileName() + "' detected.");
                fileObject.setLastState(FileState.CREATED);
                fileObject.setSizeDelta(fileObject.getSize());
                this.raiseEvent(fileObject);
                continue;
            }
            this.logDebug("Creating file '" + fileObject.getAbsoluteFileName() + "' detected.");
            fileObject.setLastState(FileState.CREATING);
            fileObject.setSizeDelta(fileObject.getSize());
            this.raiseEvent(fileObject);
        }
        this.snapshot = newSnapshot;
        this.sendSnapshot(this.snapshot);
    }

    protected void sendSnapshot(List<FileMetaInfo> snapshot) {
        if (this.isSendSnapshots()) {
            for (DirectorySnapshotListener listener : this.snapshotListeners) {
                listener.onEvent(snapshot);
            }
        }
    }

    public boolean addListener(FileChangeListener listener) {
        return this.fileChangeListeners.add(listener);
    }

    public boolean removeListener(FileChangeListener listener) {
        return this.fileChangeListeners.remove(listener);
    }

    protected boolean addListener(DirectorySnapshotListener listener) {
        return this.snapshotListeners.add(listener);
    }

    protected boolean removeListener(DirectorySnapshotListener listener) {
        return this.snapshotListeners.remove(listener);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected boolean isFileFree(FileMetaInfo fileObject) {
        boolean result;
        if (!this.isLockedStatesEnabled || fileObject.isDirectory()) {
            return true;
        }
        RandomAccessFile raf = null;
        FileLock lock = null;
        try {
            raf = new RandomAccessFile(fileObject.getAbsoluteFileName(), "rw");
            if (fileObject.getSize() > 0L) {
                raf.read();
            }
            FileChannel channel = raf.getChannel();
            lock = channel.tryLock();
            result = true;
        }
        catch (Exception e) {
            result = false;
        }
        finally {
            if (lock != null) {
                try {
                    lock.release();
                }
                catch (IOException exception) {
                    this.logException(exception, true);
                }
            }
            if (raf != null) {
                try {
                    raf.close();
                }
                catch (IOException exception) {
                    this.logException(exception, true);
                }
            }
        }
        return result;
    }

    protected synchronized void raiseEvent(FileMetaInfo state) throws Exception {
        for (FileChangeListener listener : this.fileChangeListeners) {
            listener.onChange(state);
        }
    }

    protected synchronized void raiseAdvisory(FileMetaInfo state) throws DatagramFactoryException, SecurityViolationException, SDOException {
    }

    public String getRootDirectory() {
        return this.rootDirectory.getAbsolutePath();
    }

    public String getMonitorName() {
        return this.monitorName;
    }

    public void setSendSnapshots(boolean sendSnapshots) {
        this.sendSnapshots = sendSnapshots;
    }

    public boolean isSendSnapshots() {
        return this.sendSnapshots;
    }

    protected void logException(Throwable exception, boolean printStackTrace) {
        Trace.logException(this, exception, printStackTrace);
    }

    protected void logError(String message) {
        Trace.logError(this, message);
    }

    protected void logInfo(String message) {
        Trace.logInfo(this, message);
    }

    protected void logDebug(String message) {
        Trace.logDebug(this, message);
    }

    protected static enum DirectoryState {
        AVAILABLE,
        UNAVAILABLE;

    }
}

