package com.streamscape.mf.utils;

import com.streamscape.Trace;
import com.streamscape.sdo.SDOException;
import com.streamscape.sdo.event.MapEvent;
import com.streamscape.sef.trace.LogEventSender;
import com.streamscape.sef.trace.LogSplitter;
import com.streamscape.sef.trace.TraceRecord;
import com.streamscape.tools.tailer.impl.FileTailerInputStream;

import java.io.BufferedInputStream;
import java.io.File;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.stream.Collectors;

 public class LogsHelper
 {
    public static List<TraceRecord> getLogTraceRecords(File file, int linesOffset, List<Trace.Level> levels, String startDate, String stopDate)
    {
       List<TraceRecord> traceRecords = new ArrayList<>();

       try (FileTailerInputStream inputStream = new FileTailerInputStream(file.toPath(), linesOffset, null))
       {
          LogSplitter splitter = new LogSplitter();
          long startDateTimestamp = convertDateToTimestamp("startDate", startDate);
          long stopDateTimestamp = convertDateToTimestamp("stopDate", stopDate);
          Set<String> levelsString = levels.stream().map(level -> level.toString().toUpperCase()).collect(Collectors.toSet());

          byte [] buffer = new byte[8*1024];
          BufferedInputStream bufferedInputStream = new BufferedInputStream(inputStream);

          int startIndex = 0;
          try
          {
             int count = 0;
             while ((count = bufferedInputStream.read(buffer, 0, buffer.length)) > 0)
             {
                splitter.split(new String(buffer, 0, count), traceRecords);

                boolean needToBreak = traceRecords.size() > 0 && stopDateTimestamp > 0 && traceRecords.get(traceRecords.size() -1).getTimestamp() > stopDateTimestamp;
                traceRecords = filterByLevel(traceRecords, startIndex, levelsString);
                traceRecords = filterByTimestamp(traceRecords, startIndex, startDateTimestamp, stopDateTimestamp);
                startIndex = traceRecords.size();

                if (needToBreak)
                   break;
             }
          }
          catch(Exception exception)
          {
          }

          splitter.split(null, traceRecords);
          traceRecords = filterByLevel(traceRecords, startIndex, levelsString);
          traceRecords = filterByTimestamp(traceRecords, startIndex, startDateTimestamp, stopDateTimestamp);
       }
       catch(Exception exception)
       {
          Trace.logException(LogsHelper.class, exception, true);
       }

       return traceRecords;
    }

    private static List<TraceRecord> filterByLevel(List<TraceRecord> traceRecords, int startIndex, Set<String> levels)
    {
       if (levels.size() == 0 || (levels.contains(Trace.Level.DEBUG.toString()) && levels.contains(Trace.Level.ERROR.toString()) && !levels.contains(Trace.Level.INFO.toString())))
          return traceRecords;

       if (startIndex >= traceRecords.size())
          return traceRecords;

       ListIterator<TraceRecord> iterator = traceRecords.listIterator(startIndex);
       while(iterator.hasNext())
       {
          if (!levels.contains(iterator.next().getLevel().toUpperCase()))
             iterator.remove();
       }

       return traceRecords;
    }

    private static List<TraceRecord> filterByTimestamp(List<TraceRecord> traceRecords, int startIndex, long startDateTimestamp, long stopDateTimestamp)
    {
       if (startDateTimestamp > 0 || stopDateTimestamp > 0)
       {
          int startIndexNew = 0;
          int endIndexNew = traceRecords.size();

          for (int i = startIndex; i < traceRecords.size(); i++)
          {
             TraceRecord traceRecord = traceRecords.get(i);
             if (startDateTimestamp > 0 && traceRecord.getTimestamp() < startDateTimestamp)
                startIndexNew = i;
             else if (stopDateTimestamp > 0 && traceRecord.getTimestamp() > stopDateTimestamp)
             {
                endIndexNew = i;
                break;
             }
          }

          if (startIndexNew != startIndex || endIndexNew != traceRecords.size())
             traceRecords = traceRecords.subList(startIndexNew, endIndexNew);
       }

       return traceRecords;
    }

    private static long convertDateToTimestamp(String name, String date)
    {
       if (date != null && date.trim().length() > 0)
          try
          {
             return LogSplitter.createDateFormat().parse(date).getTime();
          }
          catch (ParseException exception)
          {
             Trace.logError(LogsHelper.class, "Invalid {} date format {}.", name, date);
             Trace.logException(LogsHelper.class, exception, true);
          }
       return 0;
    }

    public static TraceRecord createTraceRecordFromTraceLogMapEvent(MapEvent event)
    {
       try
       {
          TraceRecord traceRecord = new TraceRecord();

          traceRecord.setTimestamp(event.getEventLongProperty  (LogEventSender.TIMESTAMP_PROPERTY));
          traceRecord.setLevel    (event.getEventStringProperty(LogEventSender.LEVEL_PROPERTY));
          traceRecord.setMessage  (event.getEventStringProperty(LogEventSender.MESSAGE_PROPERTY));
          traceRecord.setComponent(event.getEventStringProperty(LogEventSender.COMPONENT_PROPERTY));
          traceRecord.setEntity   (event.getEventStringProperty(LogEventSender.ENTITY_PROPERTY));

          return traceRecord;
       }
       catch (SDOException exception) {}

       return null;
    }
 }
