package com.streamscape.mf.manager.sdo;

import com.streamscape.mf.agent.enums.NotificationType;
import com.streamscape.mf.agent.sdo.ProcessIdentifier;
import com.streamscape.sdo.CloneableDataObject;
import com.streamscape.sdo.IAbstractExceptionEvent;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.advisory.*;
import com.streamscape.sdo.enums.Severity;
import com.streamscape.sdo.enums.StateAdvisoryType;
import com.streamscape.sdo.excp.*;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;

/**
 * <p>Title: Red Square Framework</p>
 *
 * <p>Description: Represents process identifier (instance)</p>
 *
 * <p>Copyright: Copyright (c) 2010</p>
 *
 * <p>Company: StreamScape Technologies</p>
 *
 * @author Sergey Zakharov
 * @version 3.2
 * @since 3.2
 */
 // -------------------------------------------------------------------------------------------------------------------
 //     DATE    |      AUTHOR      |                              SUBJECT
 // -------------------------------------------------------------------------------------------------------------------
 //  28.06.2010 | Mikhail Batsyn   | Creation
 // ------------------------------------------------------------------------------------------------------------------- 
 //  15.10.2010 | Sergey Zakharov  | Updated to provide milliseconds in the time stamp of the notification
 // -------------------------------------------------------------------------------------------------------------------
 //  17.11.2010 | Sergey Zakharov  | ECR 00000151: Processing of ClientExceptions should be added to RedSquareManager.
 // -------------------------------------------------------------------------------------------------------------------
 //  08.08.2011 | Andrey Solovyev  | Added notification type
 // -------------------------------------------------------------------------------------------------------------------
 public class PortalNotification extends CloneableDataObject
 { 
   //TODO: add NotificationId which should be unique across RedSquare area
   
   private String           source;
   private String           description;
   private String           details;
   private String           severity;
   private String           action;
   private String           date;
   private String           group;
   private NotificationType type;

   public PortalNotification() {
       SimpleDateFormat dateFormatter = new SimpleDateFormat("dd MMM yyyy HH:mm:ss.S", Locale.US);
       date = dateFormatter.format(new Date());
       action = "";
   }
   
   public PortalNotification(String componentName, ImmutableEventDatagram event)
   {
      action = ""; // to avoid error in javascript
      source = componentName;
      setType(NotificationType.Default);
      SimpleDateFormat dateFormatter = new SimpleDateFormat("dd MMM yyyy HH:mm:ss.S", Locale.US);
      date = dateFormatter.format(new Date());

      if (event instanceof ConnectionStateChangeAdvisory)
      {
         ConnectionStateChangeAdvisory advisory = (ConnectionStateChangeAdvisory)event;
         switch (advisory.getState())
         {
            case CLOSED:
               severity = "Critical";
               break;
            case RECONNECTING:
            case CONNECTING:
            case LEASE_EXPIRED:
            case LEASE_RENEW_FAILED:
            case IN_FAILOVER:
               severity = "Warning";
               break;
            default:
               severity = "General Information";
         }
         description = advisory.getMessage();
         details = "Connection = " + advisory.getConnectionName()
            + "<br>\nURL = " + advisory.getConnectionUrl();
      }
      else if (event instanceof ComponentStateChangeAdvisory)
      {
         ComponentStateChangeAdvisory advisory = (ComponentStateChangeAdvisory)event;
         description = advisory.getMessage();
         details = advisory.getEventId();
         setSeverity(Severity.INFO);
      }
      else if (event instanceof HTTPClientAdvisory)
      {
         HTTPClientAdvisory advisory = (HTTPClientAdvisory)event;
         severity = "System Information";
         description = advisory.getEventText();
         details = "Type = " + advisory.getEventType()
           + "<br>\nSession id = " + advisory.getSessionId(); 
         setType(NotificationType.HTTPClientNotification);
      }
      else if (event instanceof RuntimeAdvisory)
      {
         RuntimeAdvisory advisory = (RuntimeAdvisory)event;
         description = advisory.getMessage();
         details = event.getEventId();
         setSeverity (advisory.getSeverity());
      }
      else if (event instanceof StateAdvisory)
      {         
         StateAdvisory advisory = (StateAdvisory) event;
         description = advisory.getMessage();
         StateAdvisoryType type = advisory.getType();
         switch (type)
         {
            case INFORMATION:
               setSeverity(Severity.INFO);
               break;
            case WARNING:
               setSeverity(Severity.WARNING);
               break;
            case FAILURE:
               setSeverity(Severity.SEVERE);
               break;              
            default:
               setSeverity(Severity.GENERIC);               
         }

         StringBuilder buffer = new StringBuilder();
         for (String propertyName : advisory.getPropertyNames()) 
         {
            buffer.append(propertyName).append(" = ").append(advisory.getProperty(propertyName)).append(", ");
         }

         if (buffer.length() > 0)
         {
            details = buffer.toString().substring(0, buffer.length() - 2);
         }
         //action = "<a onclick=\"alert('Retried');\">Retry</a>&nbsp;/&nbsp;<a onclick=\"alert('Discarded');\">Discard</a>";
      }
      else if (event instanceof DatabaseSQLException)
      {
         DatabaseSQLException exception = (DatabaseSQLException)event;
         description = exception.getErrorMessage();
         details = "Component  = " + exception.getComponentName() 
           + "<br>\nConnection = " + exception.getConnectionName()
           + "<br>\nSQL State  = " + exception.getSQLState(); 
         setSeverity (exception.getSeverity());
      }
      else if (event instanceof ClientException)
      {
         ClientException exception = (ClientException)event;
         description = exception.getErrorMessage();
         details = "Connection = " + exception.getConnectionName();            
         setSeverity (exception.getSeverity());
         //group = exception.getEventGroupId();
      }
      else if (event instanceof TransportException)
      {
         TransportException exception = (TransportException)event;
         description = exception.getErrorMessage();
         details = "Component  = " + exception.getComponentName() 
         + "<br>\nConnection = " + exception.getConnectionName();
         setSeverity (exception.getSeverity());
         //group = exception.getEventGroupId();
      }
      else if (event instanceof SoapRequestException)
      {
         SoapRequestException exception = (SoapRequestException)event;
         description = exception.getErrorMessage();
         details = "No additional details available.";            
         setSeverity (exception.getSeverity());
         //group = exception.getEventGroupId();
      } else if (event instanceof ProcessIdentifier) {
          setSeverity(Severity.WARNING);
          setType(NotificationType.UserAction);
      }
      else if (event instanceof SoapFaultException)
      {
         SoapFaultException exception = (SoapFaultException)event;
         description = exception.getErrorMessage();
         details = "SOAP code: " + exception.getSoapFaultCode() + ", SOAP fault: " + exception.getSoapFaultMessage();
         setSeverity (exception.getSeverity());
         //group = exception.getEventGroupId();
      }
      else if (event instanceof XMLParseException)
      {
         XMLParseException exception = (XMLParseException)event;
         description = exception.getErrorMessage();
         details = exception.getErrorMessage();            
         setSeverity (exception.getSeverity());
         //group = exception.getEventGroupId();
      }
      else
      {
         if (event instanceof IAbstractExceptionEvent)
         {
             description = ((IAbstractExceptionEvent) event).getErrorMessage();
             severity = "Critical";
             details = ((IAbstractExceptionEvent) event).getMessage();
         } else {
             severity = "Warning";
             description = event.getEventId();
             details = event.getClass().getSimpleName();
         }
         
      }
   }

   public String getSource()
   {
      return source;
   }
   
   public void setSource(String source)
   {
       this.source = source;
   }
   
   

   public String getDescription()
   {
      return description;
   }

   public String getDetails()
   {
      return details;
   }
   
   public void setDetails(String details) {
       this.details = details;
   }
   
   public void setDescription(String description) {
       this.description = description;
   }

   public String getSeverity()
   {
      return severity;
   }

   public String getAction()
   {
      return action;
   }

   public String getDate()
   {
      return date;
   }

   public String getGroup()
   {
      return group;
   }   
   
   public void setSeverity(Severity s)
   {
      switch (s)
      {
         case SEVERE:
         {
            severity = "Critical";
            break;
         }
         case WARNING:
         {
            severity = "Warning";
            break;
         }
         case INFO:
         case GENERIC:
         {
            severity = "General Information";
            break;
         }
      }      
   }
   
   public void setGroup(String group)
   {
      // group can be assigned only if it was not set initially
      if (this.group == null)
         this.group = group;
   }

  /**
   * @param type the type to set
   */
   public void setType(NotificationType type)
   {
      this.type = type;
   }

 /**
  * @return the type
  */
  public NotificationType getType()
  {
     return type;
  }
 }
