package com.streamscape.mf.manager;

import com.streamscape.Trace;
import com.streamscape.cli.ds.DataspaceAccessor;
import com.streamscape.cli.ds.DataspaceType;
import com.streamscape.cli.service.ServiceAccessor;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionException;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.cli.tlp.FabricConnectionFactoryException;
import com.streamscape.lib.concurrent.worker.MonitorWorker;
import com.streamscape.mf.agent.RedSquareAgent;
import com.streamscape.mf.agent.enums.AgentState;
import com.streamscape.mf.agent.enums.ProcessQueueState;
import com.streamscape.mf.agent.sdo.*;
import com.streamscape.mf.manager.sco.SCOProperties;
import com.streamscape.mf.manager.sdo.*;
import com.streamscape.mf.utils.ConfigurationChecker;
import com.streamscape.mf.utils.RowSetUtils;
import com.streamscape.omf.json.jackson.JSONSerializer;
import com.streamscape.omf.serializer.SerializerException;
import com.streamscape.omf.xml.XSerializer;
import com.streamscape.repository.cache.IllegalStateException;
import com.streamscape.repository.globals.UnresolvedVariableException;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.runtime.mf.admin.glv.VariableFactory;
import com.streamscape.sdo.ImmutableEventDatagram;
import com.streamscape.sdo.event.BytesEvent;
import com.streamscape.sdo.event.DataEvent;
import com.streamscape.sdo.event.EventDatagramFactory;
import com.streamscape.sdo.excp.FabricEventException;
import com.streamscape.sdo.excp.ServiceFrameworkException;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sdo.rowset.RowSet;
import com.streamscape.sdo.utils.SDOUtils;
import com.streamscape.sdo.vcard.vCard;
import com.streamscape.sef.EventAsyncConsumer;
import com.streamscape.sef.FabricEventListener;
import com.streamscape.sef.FabricException;
import com.streamscape.sef.accessor.FabricComponentAccessorException;
import com.streamscape.sef.dataspace.DataspaceManager;
import com.streamscape.sef.dispatcher.AbstractSystemService;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.exchange.FabricAddress;
import com.streamscape.sef.moderator.ComponentReference;
import com.streamscape.sef.moderator.FabricModeratorAdvisory;
import com.streamscape.sef.moderator.FabricNodeReference;
import com.streamscape.sef.network.LinkAddress;
import com.streamscape.sef.network.LinkProtocol;
import com.streamscape.sef.network.http.acceptor.HTTPAcceptor;
import com.streamscape.sef.network.http.acceptor.HTTPAcceptorConfiguration;
import com.streamscape.sef.network.http.server.utils.HTTPUtils;
import com.streamscape.sef.security.Organization;
import com.streamscape.sef.security.SecurityManager;
import com.streamscape.sef.security.SecurityManagerException;
import com.streamscape.sef.security.User;
import com.streamscape.service.osf.config.ServiceConfigurationException;
import org.apache.commons.io.FileUtils;

import java.io.*;
import java.sql.SQLException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import static com.streamscape.sef.ErrorCodes.SEF_GENERAL_ERROR;
import static com.streamscape.sef.ErrorCodes.SEF_SERVICE_INITIALIZATION_FAILED;

/**
 * <p>Title: Red Square Framework</p>
 * 
 * <p>Description: Service for agent nodes coordination</p>
 * 
 * <p>Copyright: Copyright (c) 2010</p>
 * 
 * <p>Company: Neeve Technologies</p>
 * 
 * @author Sergey Zakharov
 * @version 3.4
 * @since 3.2
 */
 // -------------------------------------------------------------------------------------------------------------------
 //     DATE    |      AUTHOR      |                              SUBJECT
 // -------------------------------------------------------------------------------------------------------------------
 //  10.06.2010 | Sergey Zakharov  | Creation
 // -------------------------------------------------------------------------------------------------------------------
 //  02.07.2010 | Sergey Zakharov  | Implemented event handler for detailed audit (audit step request) obtaining
 // -------------------------------------------------------------------------------------------------------------------
 //  15.07.2010 | Sergey Zakharov  | Added processing of auditId for getDetailedAudit to identify audit record
 // -------------------------------------------------------------------------------------------------------------------
 //  20.07.2010 | Sergey Zakharov  | Updated Retry/Discard operations to return response
 // -------------------------------------------------------------------------------------------------------------------
 //  15.10.2010 | Sergey Zakharov  | Updated to allow process properties, statistics retrieval and process removal
 // -------------------------------------------------------------------------------------------------------------------
 //  21.10.2010 | Sergey Zakharov  | Updated to include target process type and instance for outgoing responses
 // -------------------------------------------------------------------------------------------------------------------
 //  26.10.2010 | Sergey Zakharov  | Added NULL pointer errors handling
 // -------------------------------------------------------------------------------------------------------------------
 //  04.11.2010 | Sergey Zakharov  | Updated to comply with the latest dataspace framework
 // -------------------------------------------------------------------------------------------------------------------
 //  06.11.2010 | Sergey Zakharov  | Fixed problem with notifications retrieval from the process queue
 // -------------------------------------------------------------------------------------------------------------------
 //  09.11.2010 | Mikhail Filichev | ECR 00000107: Functionality of FabricConnection should be extended.
 // -------------------------------------------------------------------------------------------------------------------
 //  11.11.2010 | Sergey Zakharov  | EBF 00000141: Event group is not shown for Red Square notifications loaded from notif queue.
 // -------------------------------------------------------------------------------------------------------------------
 //  06.12.2010 | Sergey Zakharov  | Updated 'getProcessList' request processing
 // -------------------------------------------------------------------------------------------------------------------
 //  03.01.2011 | Mikhail Filichev | ECR 00000188: FabricConnectionFactory should be reworked.
 // -------------------------------------------------------------------------------------------------------------------
 //  04.02.2011 | Sergey Zakharov  | Updated 'removeProcess' operation to remove SQL audit along with general audit.
 // -------------------------------------------------------------------------------------------------------------------
 //  08.02.2011 | Mikhail Filichev | ECR 00000204: Bound Producer should be implemented for FabricConnection.
 // -------------------------------------------------------------------------------------------------------------------
 //  11.04.2011 | Andrey Solovyev  | Red Square Agent Isolation
 // -------------------------------------------------------------------------------------------------------------------
 //  07.07.2011 | Sergey Zakharov  | Rewritten to comply with the new dataspace framework.
 // -------------------------------------------------------------------------------------------------------------------
 //  12.07.2011 | Sergey Zakharov  | Updated AgentsStateProcessor and ProcessesStateProcessor to use their own dataspace accessors.
 // -------------------------------------------------------------------------------------------------------------------
 //  20.07.2011 | Sergey Zakharov  | Fixed 'filterNotification' method to properly escape queue name and to use new
 //             |                  | 'query event' dataspace operation.
 // -------------------------------------------------------------------------------------------------------------------
 //  27.07.2011 | Sergey Zakharov  | ECR 00000408: Dataspace identifiers should be case sensitive.
 //--------------------------------------------------------------------------------------------------------------------
 //  03.08.2011 | Mikhail Filichev | ECR 00000402: Method 'createDataEventAsNew' of EventDatagramFactory should be removed.
 // -------------------------------------------------------------------------------------------------------------------
 //  09.08.2011 | Sergey Zakharov  | Added dynamic check of necessary semantic types and event prototypes on startup.
 //--------------------------------------------------------------------------------------------------------------------
 //  20.12.2011 | Andrey Solovyev  | Moving to package com.streamscape.mf
 //--------------------------------------------------------------------------------------------------------------------
 //  12.08.2012 | Ruslan Boyarskiy | ECR 00000653: Logs for agent nodes downloading.
 //--------------------------------------------------------------------------------------------------------------------
 //  27.10.2012 | Ruslan Boyarskiy | ECR 00000927: Need to print exception's traces in java part of RS
 // -------------------------------------------------------------------------------------------------------------------
 //  27.10.2012 | Ruslan Boyarskiy | ECR 00000928: Implement opportunity to change User's role
 // -------------------------------------------------------------------------------------------------------------------
 //  13.11.2013 | Mikhail Filichev | EBF 00001206: Initialization of RedSquare service does not work for slow disks.
 // -------------------------------------------------------------------------------------------------------------------
 //  26.11.2014 | Nikita Kutuzov   | SAE-124 RedSquare should be enhanced.
 // -------------------------------------------------------------------------------------------------------------------
 public class RedSquareManager extends AbstractSystemService
 {
    public final static String      NOT_ENROLLED_ROLE                            = "Not Enrolled";
    public final static String      MANAGEMENT_POOL_NAME                         = "ManagementFramework";
    public final static String      SYSTEM_VARIABLE_NAME                         = "System"; 
    public static final  int        MAX_TEXT_FIELD_WIDTH                         = 80;
    public final static String      PROCESS_STATE_CHANGE_EVENT_ID                = "event.redsquare.someProcessTypeStateChange";
    public static final String      AGENT_STATE_CHANGE_EVENT_ID                  = "event.redsquare.agentStateChange";
    public static final String      NOTIFICATION_EVENT_ID                        = "event.redsquare.notification";
    public static final String      RED_SQUARE_AGENT_OPERATION_PROGRESS_EVENT_ID = "event.redsquare.agent.OperationProgress";

    private final String            USERS_ORGANIZATIONS_TABLE_NAME               = "rs$UsersOrganizations";
    private final String            USERS_TABLE_NAME                             = "rs$Users";
    private final String            PROCESS_TYPES_TABLE_NAME                     = "rs$Processes";
    private final String            DOCUMENTATION_TABLE_NAME                     = "rs$ProcessesDocumentation";
    private final String            AGENTS_TABLE_NAME                            = "rs$Agents";
    
    private final String            MAIN_DOMAIN_NAME                             = "Domain";

    private final String            FILE_PATH                                    = RedSquareManager.class.getPackage().getName().replaceAll("\\.", "/");
    private final String            INSTALL_FILE_NAME                            = "/" + FILE_PATH + "/InstallRedSquareManager.sql";
    
    private final String            AGENT_LOGS_FOLDER                            = "." + System.getProperty( "file.separator" ) + ".agentlogs"; // root of node
    private final Long              AGENT_LOGS_LIVETIME                          = 600000L; // ~ 10 minutes
    private final String            AGENT_LOGS_TMPFILE_END                       = "LogFile.zip";    

    protected FabricConnection      connection                                   = null;
    protected DataspaceAccessor     tspaceAccessor                               = null;
    protected DataspaceAccessor     qspaceAccessor                               = null;

    protected String                tableSpaceName                               = null;
    protected String                notifQueueName                               = null;
    protected String                queueSpaceName                               = null;
    protected XSerializer           serializer                                   = null;
    protected Long                  agentStateCheckInterval                      = -1L;

    EventDatagramFactory            datagramFactory                              = null;

    private Map<String, Long>       agentsLastActiveTimes                        = new ConcurrentHashMap<String, Long>();
    private Map<String, Long>       processesLastActiveTimes                     = new ConcurrentHashMap<String, Long>();
    private volatile boolean        someProcessTypeStateChanged                  = false;
    private ProcessesStateProcessor processesChecker                             = null;
    private AgentsStateProcessor    agentsChecker                                = null;
    private EventAsyncConsumer      redSquareAgentsDisconnectedConsumer          = null;
    
    private JSONSerializer jsonSerializer                                = null;

    public RedSquareManager() throws Exception
    {
       ConfigurationChecker.checkSemanticTypes();
       ConfigurationChecker.checkEventPrototypes();
    }

    public int getMinorBuild()
    {
       return Version.getBuild();
    }

    public int getMajorVersion()
    {
       return Version.getMajorVersion();
    }

    public int getMinorVersion()
    {
       return Version.getMinorVersion();
    }

    public String getVersion()
    {
       return Version.getVersionString();
    }

    public void start()
    {
       ctx.logDebug("Starting RS Manager heartbeat check threads.");
       try
       {
          agentsChecker = new AgentsStateProcessor("AgentStateProcessor", "Checks agents states and sends it to Portal in case of any changes",
                                                   agentStateCheckInterval);
          agentsChecker.start();
          ctx.logDebug("RS Agents HeartBeat check thread started.");
       }
       catch (Exception e)
       {
          ctx.logError("Unable to set " + agentStateCheckInterval + " timeout for AgentStateProcessor.");
       }

       try
       {
          processesChecker = new ProcessesStateProcessor("ProcessStateProcessor", "Checks process states and sends it to portal in case of any " +
                                                                                  "changes", agentStateCheckInterval);
          processesChecker.start();
          ctx.logDebug("RS Processes HeartBeat check thread started.");
       }
       catch (Exception e)
       {
          ctx.logError("Unable to set " + agentStateCheckInterval + " timeout for ProcessStateProcessor.");
       }

       try
       {
          if (VariableFactory.existsLiteralPool(MANAGEMENT_POOL_NAME))
          {
             if (!VariableFactory.existsVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME))
             {
                VariableFactory.addVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME);
                VariableFactory.setVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME, "Domain");
             }
          }
          else
          {
             VariableFactory.addLiteralPool(MANAGEMENT_POOL_NAME);
             VariableFactory.addVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME);
             VariableFactory.setVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME, "Domain");
          }
       }
       catch (Exception e)
       {
          ctx.logError("Unable to set " + MANAGEMENT_POOL_NAME + " System variable");
       }
       
       pingAgents();
       startAgentsDisconnectedConsumer();
       
       new File( AGENT_LOGS_FOLDER ).mkdir();
    }

    public void stop()
    {
       ctx.logDebug("Stopping RS Manager HeartBeat Threads.");
       
       stopAgentsDisconnectedConsumer();

       if (agentsChecker != null)
       {
          agentsChecker.stop();
          agentsChecker = null;
       }

       if (processesChecker != null)
       {
          processesChecker.stop();
          processesChecker = null;
       }

       try 
       {
          FileUtils.deleteDirectory(new File( AGENT_LOGS_FOLDER ));
       } 
       catch( final IOException ex ) 
       {
          ctx.logError( "Can not remove " + AGENT_LOGS_FOLDER +  " directory : " + ex.getMessage() );
       }

       try
       {
          Thread.sleep(1000); // allow consumers to finish their work
       }
       catch (InterruptedException exception)
       {
          Thread.currentThread().interrupt();
       } 

       ctx.logDebug("Stopped RS Agents HeartBeat Threads.");
    }

    public void destroy()
    {
       try
       {
          if (tspaceAccessor != null)
             tspaceAccessor.close();

          if (qspaceAccessor != null)
             qspaceAccessor.close();

          if (connection != null)
             connection.close();
       }
       catch (FabricConnectionException exception)
       {
          Trace.logException(this, exception, false);
       }
    }

    protected void doInit() throws ServiceFrameworkException
    {
       try
       {
          ConfigurationChecker.createTriggers(ctx, "/resources/RedSquareManagerTriggers.slang");

          connection = new FabricConnectionFactory().createConnection();
          connection.open();

          queueSpaceName = ctx.lookupStringProperty(SCOProperties.QUEUE_SPACE_NAME);
          ctx.logInfo("Queue space name '" + queueSpaceName + "'.");

          notifQueueName = ctx.lookupStringProperty(SCOProperties.NOTIF_QUEUE_NAME);
          ctx.logInfo("Notifications queue name '" + notifQueueName + "'.");

          tableSpaceName = ctx.lookupStringProperty(SCOProperties.TABLE_SPACE_NAME);
          ctx.logInfo("Table space name '" + tableSpaceName + "'.");

          if (sco.hasProperty(SCOProperties.AGENTS_STATUS_CHECK_INTERVAL))
          {
             agentStateCheckInterval = ctx.lookupNumericProperty(SCOProperties.AGENTS_STATUS_CHECK_INTERVAL);
             ctx.logInfo("Interval for agent and processes states check is '" + agentStateCheckInterval + "'.");
          }
          else
          {
             agentStateCheckInterval = 10000L;
             ctx.logInfo("Interval for agent and processes states check is not specified and set to default " + agentStateCheckInterval + "ms.");
          }

          serializer = context.getXSerializerFactory().createSerializer("RedSquareManagerSerializer");
          serializer.mapAttribute("SemanticType", "class");
          
          datagramFactory = context.getEventDatagramFactory();

          initializeSystemTablesAndAccessors();

          jsonSerializer = HTTPUtils.getJsonSerializerForFabric();
       }
       catch (FabricConnectionFactoryException error)
       {
          throw new ServiceFrameworkException(SEF_SERVICE_INITIALIZATION_FAILED, error.getMessage());
       }
       catch (ServiceConfigurationException error)
       {
          throw new ServiceFrameworkException(SEF_SERVICE_INITIALIZATION_FAILED, error.getMessage());
       }
       catch (Exception error)
       {
          throw new ServiceFrameworkException(SEF_SERVICE_INITIALIZATION_FAILED, error);
       }
    }
    
    public RedSquareResponseWrapper dispatchRequest(ImmutableEventDatagram event) throws Exception
    {
       Object result = null;
       RedSquareRequestWrapper wrapper = (RedSquareRequestWrapper) ((DataEvent)event).getData();
       if (wrapper == null)
          throw new Exception("Null data in redsquare request.");
       if (wrapper.getId() == null)
          throw new Exception("Null id in redsquare request data.");
       switch(wrapper.getId())
       {
          case "event.redsquare.AgentsListRequest":
             result = getAgentsList(wrapper.getJsondata());
             break;
          case "event.redsquare.globalVariableActionRequest":
             result = doGlobalVariableAction(wrapper.getJsondata());
             break;
          case "event.redsquare.GlobalVariablesRequest":
             result = getGlobalVariables(wrapper.getJsondata());
             break;
          case "event.redsquare.LoggedInUsersRequest":
             result = getLoggedInUsers(wrapper.getJsondata());
             break;
          case "event.redsquare.UserInfoRequest":
             result = getUserInfo(wrapper.getJsondata());
             break;
          case "event.redsquare.CreateUserInfoRequest":
             result = createRedSquareUser(wrapper.getJsondata());
             break;
          case "event.redsquare.enrolUserRequest":
             result = enrolUser(wrapper.getJsondata());
             break;
          case "event.redsquare.updateEnroledUserRequest":
             result = updateEnroledUser(wrapper.getJsondata());
             break;
          case "event.redsquare.expelUserRequest":
             result = expelUser(wrapper.getJsondata());
             break;
          case "event.redsquare.RedSquareUsersForOrganizationRequest":
             result = getRedSquareUsersForOrganization(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessTypesRequest":
             result = getProcessTypesList(wrapper.getJsondata());
             break;
          case "event.redsquare.documentationRequest":
             result = getDocumentationProcessTypesList(wrapper.getJsondata());
             break;
          case "event.redsquare.documentationActionRequest":
             result = doDocumentationEntryAction(wrapper.getJsondata());
             break;
          case "event.redsquare.NodeLogsRequest":
             result = getLogsForNode(wrapper.getJsondata());
             break;
          case "event.redsquare.FilterNotificationsRequest":
             result = filterNotifications(wrapper.getJsondata());
             break;
          case "event.redsquare.ListDataspacesRequest":
             result = listDataspaces(wrapper.getJsondata());
             break;
          case "event.redsquare.ListUserFunctionsRequest":
             result = listUserFunctions(wrapper.getJsondata());
             break;
          default:
             ctx.logError("Unknown redsquare request id:" + wrapper.getId());
       }

       return new RedSquareResponseWrapper(result);
    }
    
    private <T> T deserialize(String jsondata, Class<T> clazz) throws SerializerException
    {
       return (T)jsonSerializer.deserialize(clazz, jsondata);
    }

    public AgentListResponse getAgentsList(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SQLException, SerializerException
    {
       Trace.logDebug(this, "Agents list request processing.");

       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT AGENT_NAME,AGENT_STATE,AGENT_NODE FROM " +  
             escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + " WHERE AGENT_STATE <> 'STOPPED' ORDER BY AGENT_NAME");

       if (printSLResponseIfError("Unable to extract agents definitions from '" + AGENTS_TABLE_NAME + "' table.", response))
          return null;

       RedSquareRowSet rowSet = RowSetUtils.initFromRowSet(response.getRowSet());
       AgentListResponse result = new AgentListResponse();
       result.setAgentSet(rowSet);

       ArrayList<String> httpList = new ArrayList<String>();
       RowSet agentRowSet = response.getRowSet();
       if (agentRowSet.first())
       {
          agentRowSet.beforeFirst();
          while (agentRowSet.next())
          {
             String httpAddress = "";
             for (FabricNodeReference node : ctx.getModerator().getFabricNodes())
             {
                if (node.getName().equals(agentRowSet.getObject("AGENT_NODE")))
                {
                   for (LinkAddress acceptor : node.getAcceptors())
                   {
                      if (acceptor.getProtocol().equals(LinkProtocol.HTTP))
                      {
                         httpAddress = acceptor.getAddress().toString();
                         break;
                      }
                   }
                }
             }
             httpList.add(httpAddress);
          }
       }

       result.setHttpLinks(httpList);
       return result;
    }

    public RedSquareOperationResponse doGlobalVariableAction(String jsondata) throws Exception
    {
       GlobalVariablesRequest request = deserialize(jsondata, GlobalVariablesRequest.class);
       Trace.logDebug(this, "Global variables action request processing. Action: " + request.getAction());
       RedSquareOperationResponse response = new RedSquareOperationResponse();
       switch (request.getAction())
       {
          case Delete:
          {
             if (request.getVariableName().isEmpty())
             {
                VariableFactory.removeLiteralPool(request.getPoolName());
                response.setSuccess(true);
                response.setAdditionalMessage("Pool " + request.getPoolName() + " was deleted.");
                response.setProcessType("");
                return response;
             }
             VariableFactory.removeVariable(request.getPoolName(), request.getVariableName());
             response.setSuccess(true);
             response.setAdditionalMessage("Variable " + request.getVariableName() + " was deleted.");
             response.setProcessType(request.getPoolName());
             return response;
          }
          case Edit:
          {
             VariableFactory.setVariable(request.getPoolName(), request.getVariableName(), request.getValue());
             response.setSuccess(true);
             //TODO: field name should be changed from process type to something else,
             //we have enought situations where we need it different already, but there is no need in creating new object
             response.setProcessType(request.getPoolName());
             response.setAdditionalMessage("Variable " + request.getVariableName() + " was edited successfully.");
             return response;
          }
          case Create:
          {
             if (request.getVariableName().isEmpty())
             {
                VariableFactory.addLiteralPool(request.getPoolName());
                response.setSuccess(true);
                response.setAdditionalMessage("Pool " + request.getPoolName() + " was created.");
                response.setProcessType("");
                return response;
             }
             VariableFactory.addVariable(request.getPoolName(), request.getVariableName());
             VariableFactory.setVariable(request.getPoolName(), request.getVariableName(), request.getValue());
             response.setSuccess(true);
             response.setProcessType(request.getPoolName());
             response.setAdditionalMessage("Variable " + request.getVariableName() + " was created successfully.");
             return response;
          }
       }
       
       response.setSuccess(false);
       response.setErrorMessage("ERROR: Unknown action type.");
       return response;
    }

    public RedSquareRowSet getGlobalVariables(String jsondata) throws Exception
    {
       Trace.logDebug(this, "Global variables request processing.");
       GlobalVariablesRequest request = deserialize(jsondata, GlobalVariablesRequest.class);
       if (request.getPoolName() == null || request.getPoolName().isEmpty())
       {
          List<String> globalVarsPools = VariableFactory.listLiteralPools();
          RedSquareRowSet result = new RedSquareRowSet(1, globalVarsPools.size());
          for (String poolName : globalVarsPools)
             result.addRow(new Object[] { poolName });
          return result;
       }
       else
       {
          List<String> globalVarsPoolVars = VariableFactory.listLiteralPoolVariables(request.getPoolName());
          RedSquareRowSet result = new RedSquareRowSet(3, globalVarsPoolVars.size());
          for (String varName : globalVarsPoolVars)
          {
             String value = VariableFactory.getVariable(request.getPoolName(), varName);
             result.addRow(new Object[] { varName, value, request.getPoolName() });
          }
          return result;
       }
    }
    
    public RedSquareRowSet getLoggedInUsers(String jsondata)
    {
       try
       {
          Trace.logDebug(RedSquareManager.class, "Logged in users request processing.");

          List<String> compList = connection.getModerator().listComponents();
          List<String> usersList = new ArrayList<String>();
          List<String> userCompList = new ArrayList<String>();

          for (String comp : compList)
          {
             if (comp.contains("Client_HTTP") && comp.contains("@") && comp.contains("redsquare"))
             {
                String userName = comp.split("\\.")[1].split("@")[0];
                if (usersList.contains(userName))
                   continue;
                usersList.add(userName);
                userCompList.add(comp);                
             }
          }

          RedSquareRowSet result = new RedSquareRowSet(2, usersList.size());
          for (int i = 0; i < usersList.size(); ++i)          
             result.addRow(new Object[] { usersList.get(i), userCompList.get(i) });

          Trace.logDebug(RedSquareManager.class, "Logged in users request processed.");
          return result;
       }
       catch (Exception e)
       {
          Trace.logException(RedSquareManager.class, e, true);
          return null;
       }
    }
    
    public RedSquareUserInfo getUserInfo(String jsondata) throws SecurityManagerException, FabricComponentAccessorException, UnresolvedVariableException, SQLException, SerializerException
    {
       RedSquareUser redSquareUser = deserialize(jsondata, RedSquareUser.class);
       Trace.logDebug(this, "User info request processing. 'user name': " + redSquareUser.getUserName());
       
       SecurityManager secManager = context.getSecurityManager();
       RedSquareUserInfo resultingInfo = new RedSquareUserInfo(NOT_ENROLLED_ROLE, VariableFactory.getVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME));
       resultingInfo.setUserId(redSquareUser.getUserName());

       User user = secManager.lookupUser(redSquareUser.getUserName());
       if (user != null)
       {
          if (user.isAdministrator())
             resultingInfo.setRole("Admin");
          resultingInfo.setOrganization(user.getOrganization().toString());
       }

       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT USER_ROLE FROM " + escapeDataSpaceObjectName(USERS_TABLE_NAME) + " WHERE " +
             "USER_NAME='" + redSquareUser.getUserName() + "'");

       if (response != null && response.isOK() && response.getRowSet().first())
          resultingInfo.setRole(response.getRowSet().getObject("USER_ROLE").toString());

       vCard vCard = secManager.getVCard(redSquareUser.getUserName());
       if (vCard != null)
       {
          if (!vCard.getFullName().isEmpty())
             resultingInfo.setFullName(vCard.getFullName());
          else
             resultingInfo.setFullName(redSquareUser.getUserName());

          if (vCard.getPhoto().getType() != null || vCard.getPhoto().getData() != null)
             resultingInfo.setBase64photo(/*"data:" + vCard.getPhoto().getType() + ";base64," + */vCard.getPhoto().getData());
          else
             resultingInfo.setBase64photo("");
       }
       else
       {
          resultingInfo.setFullName(redSquareUser.getUserName());
          resultingInfo.setBase64photo("");
       }

       return resultingInfo;
    }

    public RedSquareUserInfo createRedSquareUser(String jsondata) throws SecurityManagerException, FabricComponentAccessorException, SerializerException
    {
       Trace.logDebug(this, "Create RS User request processing.");
       RedSquareUserInfo request = deserialize(jsondata, RedSquareUserInfo.class);
       SecurityManager secManager = context.getSecurityManager();
       User user = secManager.createUser(request.getUserId(), request.getPassword(), request.getDescription());
       secManager.setUserOrganization(user.getName().toString(), request.getOrganization());
       secManager.addUserToGroup(request.getUserId(), SecurityManager.OPERATORS_GROUP);

       if (!request.isEnrol())
          return request;

       SLResponse response = tspaceAccessor.invokeLanguageRequest("INSERT INTO " + escapeDataSpaceObjectName(USERS_TABLE_NAME) +
             " VALUES ('" + request.getUserId() + "','" + request.getDescription() +
             "','" + request.getRole() + "','')");
       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to insert values into " + USERS_TABLE_NAME + " table.", response);
          throw new SecurityManagerException(SEF_GENERAL_ERROR, "Unable to update RedSquare Inner table.");
       }

       return request;
    }
    
    public RedSquareUserInfo enrolUser(String jsondata) throws Exception 
    {
       RedSquareUserInfo request = deserialize(jsondata, RedSquareUserInfo.class);
       String system = "Domain";
       try
       {
          system = VariableFactory.getVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME);
       } 
       catch (UnresolvedVariableException e)
       {
          Trace.logError(this, "Unable to get current system!");
          Trace.logException(this, e, true);
       }

       Trace.logDebug(this, "Enrol User request processing. 'user name': " + request.getUserId());

       SecurityManager secManager = context.getSecurityManager();
       User user = secManager.lookupUser(request.getUserId());
       String description = request.getDescription();

       SLResponse response = tspaceAccessor.invokeLanguageRequest("INSERT INTO "
             + escapeDataSpaceObjectName(USERS_TABLE_NAME) + " VALUES ('" + request.getUserId() + "','" + description 
             + "','" + request.getRole() + "','" + system + "')");

       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to extract user info from '" + USERS_TABLE_NAME + "' table.", response);
          return null;
       }
       request.setOrganization(user.getOrganization().toString());
       return request;
    }

    public RedSquareUserInfo updateEnroledUser(String jsondata) throws Exception
    {
       RedSquareUserInfo request = deserialize(jsondata, RedSquareUserInfo.class);
       Trace.logDebug(this, "Update Enroled user request processing. 'user name': " + request.getUserId());

       SecurityManager secManager = context.getSecurityManager();
       User user = secManager.lookupUser(request.getUserId());

       Map<String,String> updateFields = new HashMap<String, String>();
       if (null != request.getDescription())
          updateFields.put( "DESCRIPTION", request.getDescription() );

       if (null != request.getRole())
          updateFields.put( "USER_ROLE", request.getRole() );

       if (null != request.getSystem())
          updateFields.put( "SYSTEM", request.getSystem() );

       if (updateFields.size() > 0) 
       {
          String trequest = "UPDATE " + escapeDataSpaceObjectName(USERS_TABLE_NAME) + " SET ";

          for (String field: updateFields.keySet())
             trequest += field + "='" + updateFields.get( field )  +"', ";

          trequest = trequest.replaceFirst(", $", "");
          trequest += " WHERE USER_NAME='" + request.getUserId() + "'";

          SLResponse tresponse = tspaceAccessor.invokeLanguageRequest( trequest );
          if (tresponse == null || !tresponse.isOK())
          {
             printSLResponseIfError("Unable to insert values into " + USERS_TABLE_NAME + " table.", tresponse);
             return null;
          }
       }
       else 
          Trace.logError(this, "There are no fields to change");

       request.setOrganization(user.getOrganization().toString());
       return request;
    }        

    public RedSquareUserInfo expelUser(String jsondata) throws Exception 
    {
       RedSquareUserInfo request = deserialize(jsondata, RedSquareUserInfo.class);
       Trace.logDebug(this, "Expel User request processing. 'user name': " + request.getUserId());

       SLResponse response = tspaceAccessor.invokeLanguageRequest("DELETE FROM "
             + escapeDataSpaceObjectName(USERS_TABLE_NAME) + " WHERE USER_NAME='" + request.getUserId() + "'");

       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to delete values from " + USERS_TABLE_NAME + " table!", response);
          return null;
       }

       SecurityManager secManager = context.getSecurityManager();
       User user = secManager.lookupUser(request.getUserId());
       request.setOrganization(user.getOrganization().toString());

       return request;
    }

    public RedSquareRowSet getRedSquareUsersForOrganization(String jsondata) throws Exception
    {
       UserInfosForOrganizationRequest request = deserialize(jsondata, UserInfosForOrganizationRequest.class);
       Trace.logDebug(this, "Organization User request processing. 'organization name': " + request.getOrganization());

       SecurityManager secManager = context.getSecurityManager();
       Organization org = secManager.lookupOrganization(request.getOrganization());
       List<String> orgMembers = org.listMembers();
       List<String> usersOnly = new LinkedList<String>();

       for (String userOrGroup : orgMembers)
          if (secManager.lookupUser(userOrGroup) != null) 
             usersOnly.add(userOrGroup);

       RedSquareRowSet result = new RedSquareRowSet(1, usersOnly.size());
       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM " + escapeDataSpaceObjectName(USERS_TABLE_NAME));
       if (response == null || !response.isOK() || response.getRowSet() == null)
       {   
          printSLResponseIfError("Unable to extract user info from '" + USERS_TABLE_NAME + "' table.", response);
          for (String userId : usersOnly)
          {   
             RedSquareUserInfo info = new RedSquareUserInfo();
             info.setUserId(userId);
             info.setOrganization(request.getOrganization());
             result.addRow(new Object[] { info });
          }
          return result;
       }

       RowSet rowSet = response.getRowSet();
       rowSet.beforeFirst();
       String system = "Domain";
       try
       {
          system = VariableFactory.getVariable(MANAGEMENT_POOL_NAME, SYSTEM_VARIABLE_NAME);
       } 
       catch (UnresolvedVariableException e)
       {
          Trace.logError(this, "Unable to get current system.");
          Trace.logException(this, e, true);
       }

       while (rowSet.next())
       {
          String userId      = rowSet.getObject("USER_NAME").toString();
          String userRole    = rowSet.getObject("USER_ROLE").toString();
          String description = rowSet.getObject("DESCRIPTION").toString();

          String fullName = userId;
          
          try
          {
             vCard vCard = secManager.getVCard(userId);
             if (vCard != null) 
                fullName = vCard.getFullName();
          }
          catch (Exception exception)
          {
          }
          
          if (usersOnly.contains(userId)) 
          {
             usersOnly.remove(userId);

             RedSquareUserInfo info = new RedSquareUserInfo();
             info.setOrganization(request.getOrganization());
             info.setUserId(userId);
             info.setEnrol(true);
             info.setFullName(fullName);
             info.setSystem(system);
             info.setDescription(description);
             info.setRole(userRole);

             result.addRow(new Object[]{ info });
          }
       }

       for (String userId : usersOnly)
       {
          RedSquareUserInfo info = new RedSquareUserInfo();
          info.setUserId(userId);
          info.setOrganization(request.getOrganization());

          String fullName = userId;

          try
          {
             vCard vCard = secManager.getVCard(userId);
             if (vCard != null) 
                fullName = vCard.getFullName();
          }
          catch (Exception exception)
          {
          }
          info.setFullName(fullName);

          result.addRow(new Object[]{ info });
       }

       return result;
    }

    public RedSquareRowSet getProcessTypesList(String jsondata) throws Exception
    {
       Trace.logDebug(this, "Process types request processing.");
       ProcessTypesRequest request = deserialize(jsondata, ProcessTypesRequest.class);
       
       SLResponse response;
       if (request.getOrganization() == null || request.getOrganization().equalsIgnoreCase("") || request.getOrganization().equals(MAIN_DOMAIN_NAME))
       {
          response = tspaceAccessor.invokeLanguageRequest("SELECT PROCESS_TYPE,PROCESS_NODE,PORTAL_SERVICE,PROCESS_STATE, ID FROM " +  
                      escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + " ORDER BY PROCESS_TYPE");
       }
       else
       {
          response = tspaceAccessor.invokeLanguageRequest("SELECT PROCESS_TYPE,PROCESS_NODE,PORTAL_SERVICE,PROCESS_STATE, ID FROM " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + " WHERE " +
                     "PROCESS_ORGANIZATION='" + request.getOrganization() + "' ORDER BY PROCESS_TYPE");
       }

       if (printSLResponseIfError("Unable to extract process type definitions from '" + PROCESS_TYPES_TABLE_NAME + "' table.", response))
          return null;

       return RowSetUtils.initFromRowSet(response.getRowSet());
    }

    public RedSquareRowSet getDocumentationProcessTypesList(String jsondata) throws Exception
    {
       Trace.logDebug(this, "Documentation request processing.");
       ProcessTypesRequest request = deserialize(jsondata, ProcessTypesRequest.class);
       
       SLResponse response;
       if (request.getFilterProcessTypeName() == null)
       {
          //serializer can't serialize arrays with different elements, so we use String instead of Boolean here to return rowSet correctly
          response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM " + escapeDataSpaceObjectName(DOCUMENTATION_TABLE_NAME) + " WHERE " +
                                                          "MAIN_LINK='true' ORDER BY PROCESS_TYPE");
       }
       else
       {
          response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM " + escapeDataSpaceObjectName(DOCUMENTATION_TABLE_NAME) + " WHERE " +
                                                          "PROCESS_TYPE='" + request.getFilterProcessTypeName() +
                                                          "' ORDER BY MAIN_LINK");
       }

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          printSLResponseIfError("Unable to extract documentation definitions from '" + DOCUMENTATION_TABLE_NAME + "' table.", response);
          return null;
       }

       return RowSetUtils.initFromRowSet(response.getRowSet());
    }
    

    public RedSquareOperationResponse doDocumentationEntryAction(String jsondata) throws Exception
    {
       DocumentationEntryActionRequest request = deserialize(jsondata, DocumentationEntryActionRequest.class);
       Trace.logDebug(this, "Documentation action request processing. Action: " + request.getAction());

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       SLResponse response = null;
       String postAction = null;
       switch (request.getAction())
       {
          case Delete:
          {
             response = tspaceAccessor.invokeLanguageRequest("DELETE FROM " + escapeDataSpaceObjectName(DOCUMENTATION_TABLE_NAME) + " WHERE " +
                   "ENTRY_NAME='" + request.getEntryName() + "' AND PROCESS_TYPE='" +
                   request.getProcessType() + "'");
             postAction = "deleted";
             break;
          }
          case Create:
          {
             //serializer can't serialize arrays with different elements, so we use String instead of Boolean here to return rowSet correctly
             response = tspaceAccessor.invokeLanguageRequest("INSERT INTO " + escapeDataSpaceObjectName(DOCUMENTATION_TABLE_NAME) + "(PROCESS_TYPE," +
                   " DOCUMENTATION_ADDRESS, ENTRY_NAME, MAIN_LINK) VALUES('" + request.getProcessType() +
                   "','" + request.getEntryAddress() + "','" + request.getEntryName() + "'," +
                   "'" + request.getMainEntry().toString() + "')");
             postAction = "created";
             break;
          }
          case Edit:
          {
             response = tspaceAccessor.invokeLanguageRequest("UPDATE " + escapeDataSpaceObjectName(DOCUMENTATION_TABLE_NAME) + " SET " +
                   "DOCUMENTATION_ADDRESS='" + request.getEntryAddress() + "', " +
                   "ENTRY_NAME='" + request.getEntryName() + "' WHERE ID='" + request.getId() + "'");
             postAction = "edited";
             break;
          }
       }

       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to " + request.getAction() + " documentation entry for " + request.getProcessType(), response);
          opResponse.setSuccess(false);
          opResponse.setErrorMessage("Unable to " + request.getAction() + " " + request.getEntryName() + " entry!");
          return opResponse;
       }

       opResponse.setSuccess(true);
       opResponse.setAdditionalMessage("Document entry " + request.getEntryName() + " was " + postAction + " successfully.");

       return opResponse;
    }

    /*
     * Handles event, to get log file from target node
     * @param NodeLogsRequest contains information about target node
     * @return NodeLogsResponse contains path(aliase) for url
     */
    public NodeLogsResponse getLogsForNode(String jsondata)
    {
        // clear old packed log files
        clearOldLogsArchives();

        String errorMessage = "Unknown error";
        String localLogName = "";        
        Boolean success = false;
        File responseLog = null;
        try 
        {
           // init information about requested node
           NodeLogsRequest request = deserialize(jsondata, NodeLogsRequest.class);
           String targetNode = request.getNodeName();
           String targetServiceType = request.getServiceType();
           String targetServiceName = request.getServiceName();

           if (!connection.isBoundEventId(RedSquareAgent.AGENT_GET_LOG_FILE_REQUEST_EVENT_ID ))
              connection.bindProducerFor(RedSquareAgent.AGENT_GET_LOG_FILE_REQUEST_EVENT_ID );
           
           ctx.logDebug( "Preparing request for " + targetNode + " target node" );
           final BytesEvent logRequest = (BytesEvent)datagramFactory.createEvent(RedSquareAgent.AGENT_GET_LOG_FILE_REQUEST_EVENT_ID );
                      
           final ServiceAccessor targetAccessor = connection.createServiceAccessor( targetNode, targetServiceType, targetServiceName );
           final BytesEvent logResponse = (BytesEvent)targetAccessor.invokeServiceRequest( logRequest );
           
           ctx.logDebug( "Response has been received from " + targetNode + " target node" );

           if (logResponse.getBytesLength() > 0) 
           {
              localLogName = targetNode + "_" + ((Number)(new Date()).getTime()).toString() + AGENT_LOGS_TMPFILE_END;
        
              final String localLogPath = AGENT_LOGS_FOLDER + System.getProperty( "file.separator" ) + localLogName;
        
              responseLog = new File( localLogPath );
              final FileOutputStream fos = new FileOutputStream(responseLog);
              fos.write( logResponse.getBytes() );
              fos.close();
              
              ctx.logDebug( "Add aliase for new file into HTTPAcceptor" );
              
              HTTPAcceptorConfiguration HTTPConfiguration = ((HTTPAcceptor)RuntimeContext.getInstance().getAcceptorManager().lookupFactory( "HTTPAcceptorFactory" ).lookupAcceptor( "Default" )).getConfiguration();

              // sets http alias for file
              HTTPConfiguration.getUrlToPathAliases().put(localLogName, responseLog.getAbsolutePath());
              success = true;
           }
           else 
           {
              // invokes only if packed log file exceeds 5 mb.
              errorMessage = "Size of packed log file from " + targetNode + " node exceeds 5 mb." ;
           }
           
        }
        catch( final Exception ex ) 
        {
            ctx.logDebug( "The following error is happened: " + ex.toString() );
            Trace.logException(this, ex, true);
            if (responseLog != null)
               responseLog.delete();
            localLogName = "";
        }

        // initiate response        
        NodeLogsResponse response = new NodeLogsResponse();
        response.setFilePath( localLogName );
        response.setSuccess( success );
        response.setErrorMessage( errorMessage );
        
        return response;        
    }    
    
    private void clearOldLogsArchives()
    {
       ctx.logDebug( "Clearing Old Log files" );
       HTTPAcceptor ac = (HTTPAcceptor)RuntimeContext.getInstance().getAcceptorManager().lookupFactory( "HTTPAcceptorFactory" ).lookupAcceptor( "Default" );
       HTTPAcceptorConfiguration HTTPConfiguration = ac.getConfiguration();
       HashMap<String, String> HTTPAccessorURLs = HTTPConfiguration.getUrlToPathAliases();
       
       // Gets list of files that live more RS_AGENT_LOGS_LIVETIME milliseconds
       File [] listFiles = new File( AGENT_LOGS_FOLDER ).listFiles(new AgentLogsFileFilter( (new Date()).getTime(),AGENT_LOGS_LIVETIME));
       
       // Deletes file and corresponding alliases
       for (File listFile : listFiles)
       {
          ctx.logDebug(listFile.getName() + " log file has been removed");
          HTTPAccessorURLs.remove(listFile.getName());
          listFile.delete();
       }
    }
    

    public PortalNotificationArray filterNotifications(String jsondata) throws Exception
    {
       Trace.logDebug(this, "Filter notification request processing.");
       RedSquareFilter redSquareFilter = deserialize(jsondata, RedSquareFilter.class);
       
       List<PortalNotification> result = new ArrayList<PortalNotification>();
       String selectorString = " where (CorrelationId is not null)";
       String selector = redSquareFilter.getSelector();
       if ((selector != null) && (selector.length() > 0))
          selectorString = selectorString + " AND (" + selector + ")";

       SLResponse response = qspaceAccessor.invokeLanguageRequest("select top " + redSquareFilter.getMaxRecordsNumber() + " SeqId from " +
                                                                  escapeDataSpaceObjectName(notifQueueName) + selectorString + " ORDER BY SeqId " +
                                                                  "DESC");

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          printSLResponseIfError("Unable to extract notifications from the specified '" + notifQueueName + "' notifications queue.", response);
          return null;
       }

       RowSet rowSet = response.getRowSet();
       if (!rowSet.first())
          return new PortalNotificationArray(redSquareFilter.getMaxRecordsNumber());

       rowSet.beforeFirst();
       try
       {
          while (rowSet.next())
          {
             String notificationId = rowSet.getObject("SeqId").toString();
             response = qspaceAccessor.invokeLanguageRequest("QUERY EVENT \"" + notificationId + "\" FROM " + escapeDataSpaceObjectName
                                                                                                                 (notifQueueName));
             if (response == null || !response.isOK() || response.getRowSet() == null || !response.getRowSet().first())
             {
                printSLResponseIfError("Unable to extract notification from the specified '" + notifQueueName + "' notif queue.", response);
                return null;
             }

             Object data = serializer.deserialize(ImmutableEventDatagram.SEMANTIC_TYPE_NAME, (String)response.getRowSet().getObject(1));
             if (data != null || data instanceof DataEvent)
             {
                DataEvent event = (DataEvent)data;
                Object notificationData = event.getData();
                if (notificationData != null && notificationData instanceof PortalNotification)
                {
                   PortalNotification notification = (PortalNotification)notificationData;
                   if (notification.getGroup() == null)
                   {
                      if (event.getEventGroupId() != null)
                         notification.setGroup(event.getEventGroupId());
                      else
                         notification.setGroup(""); // to avoid errors in javascript
                   }
                   result.add(notification);
                }
             }
          }
       }
       catch (Exception error)
       {
          ctx.logError(error.getMessage());
          Trace.logException(this, error, true);
       }

       PortalNotificationArray array = new PortalNotificationArray(redSquareFilter.getMaxRecordsNumber());
       for (int i = (result.size() - 1); i >= 0; --i)
          array.addNotification(result.get(i));
       return array;
    }

    ////////////////////////////

    public ImmutableEventDatagram processAdvisory(ImmutableEventDatagram event)
    {
       try
       {
          // If not initialized yet
          if ((ctx == null) || (ctx.getModerator() == null))
             return null;
                    
          FabricAddress source = new FabricAddress(event.getEventSource());
          ComponentReference component = ctx.getModerator().lookupComponent(source);
          
          String sourceName;
          if (component != null)
             sourceName = component.getName();
          else if (ctx.getModerator().lookupFabricNode(source) != null)
             sourceName = ctx.getModerator().lookupFabricNode(source).getName();
          else
             sourceName = "Unknown";

          PortalNotification notification = new PortalNotification(sourceName, event);
          return SDOUtils.createDataEvent(NOTIFICATION_EVENT_ID, notification);
       }
       catch (Exception error)
       {
          Trace.logException(this, error, true);
          return null;
       }
    }

    public void setProcessTypeStatus(ProcessStateChangeResponse stateObject) throws FabricComponentAccessorException, SQLException
    {
       if (stateObject.getProcessState() == null)
          return;

       if (stateObject.getProcessState().equals(ProcessQueueState.DELETED))
       {
          Trace.logDebug(this, "Removing process " + stateObject.getProcessType());
          SLResponse response = tspaceAccessor.invokeLanguageRequest("DELETE FROM " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + " WHERE  PROCESS_TYPE='" + stateObject.getProcessType() + "'");
          printSLResponseIfError("Failed to delete process " + stateObject.getProcessType(), response);
          processesLastActiveTimes.remove(stateObject.getProcessType());
          someProcessTypeStateChanged = true;
          sendOperationStatusEvent(new ProgressNotification("updatingManagerTable", "confirmingChanges", "Updating Processes...", "OK", ""));
          sendOperationStatusEvent(new ProgressNotification("confirmingChanges", "confirmingChanges", "Applying Changes...", "RUNNING", ""));
          return;
       }

       processesLastActiveTimes.put(stateObject.getProcessType(), System.currentTimeMillis());

       SLResponse response = tspaceAccessor.invokeLanguageRequest(
             "SELECT PROCESS_STATE FROM " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + " WHERE  PROCESS_TYPE='" + stateObject.getProcessType() + "'");
       printSLResponseIfError("Failed to select for process " + stateObject.getProcessType(), response);
       
       if (!response.isOK() || !response.getRowSet().next())
       {
          String org = "Domain";
          if (stateObject.getOrganization() != null)
             org = stateObject.getOrganization();

          ctx.logDebug("Registering new RS Process. 'name': " + stateObject.getProcessType());
          String insertRequest = "INSERT INTO " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + "(" + "   PROCESS_TYPE, " +
                                 "" + "   PROCESS_NODE, " + "   PROCESS_ORGANIZATION, " + "   PORTAL_SERVICE, " + "   FLOW_TYPE, " +
                                 "" + "   PROCESS_STATE, " + "   LAST_ACTIVE_TIME" + ") VALUES " + "(" + "'" + stateObject.getProcessType() + "'," +
                                 "" + "'" + stateObject.getNodeName() + "'," + "'" + org + "'," + "'" + stateObject.getAgentInfo() + "'," +
                                 "" + "'" + stateObject.getFlowType() + "'," + "'" + stateObject.getProcessState() + "'," +
                                 "" + "'" + System.currentTimeMillis() + "'" + ")";
          response = tspaceAccessor.invokeLanguageRequest(insertRequest);
          printSLResponseIfError("Failed to insert new process " + stateObject.getProcessType(), response);

          ctx.logDebug("Registered new RS Process. 'name': " + stateObject.getProcessType());
          someProcessTypeStateChanged = true;
          sendOperationStatusEvent(new ProgressNotification("updatingManagerTable", "confirmingChanges", "Updating Processes...", "OK", ""));
          sendOperationStatusEvent(new ProgressNotification("confirmingChanges", "confirmingChanges", "Applying Changes...", "RUNNING", ""));
          return;
       }

       response = tspaceAccessor.invokeLanguageRequest("SELECT PROCESS_STATE FROM " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) +
                                  " WHERE PROCESS_TYPE='" + stateObject.getProcessType() + "'");

       if (printSLResponseIfError("Failed to select process " + stateObject.getProcessType(), response))
          return;
       if (!response.getRowSet().next())
          return;

       if (!stateObject.getProcessState().name().equals(response.getRowSet().getObject("PROCESS_STATE").toString()))
       {
          Trace.logDebug(this, "State changed to " + stateObject.getProcessState() + " from " + response.getRowSet().getObject("PROCESS_STATE"));
          sendOperationStatusEvent(new ProgressNotification("updatingManagerTable", "confirmingChanges", "Updating Processes...", "OK", ""));
          sendOperationStatusEvent(new ProgressNotification("confirmingChanges", "confirmingChanges", "Applying Changes...", "RUNNING", ""));

          String request = "UPDATE " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + " SET PROCESS_STATE='" + stateObject.getProcessState() +
                "', LAST_ACTIVE_TIME='" + System.currentTimeMillis() + "' WHERE PROCESS_TYPE='" + stateObject.getProcessType() + "'";
          if (stateObject.getOrganization() != null)
             request = "UPDATE " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + " SET PROCESS_ORGANIZATION='" + stateObject.getOrganization()
                   + "', LAST_ACTIVE_TIME='" + System.currentTimeMillis() + "' WHERE PROCESS_TYPE='" + stateObject.getProcessType() + "'";

          response = tspaceAccessor.invokeLanguageRequest(request);
          printSLResponseIfError("Unable to update process '" + stateObject.getProcessType() + "' organization or state.", response);
          someProcessTypeStateChanged = true;
       }
    }

    
    public void registerAgent(AgentRegisterEvent event) throws FabricComponentAccessorException, SQLException
    {
       agentsLastActiveTimes.put(event.getNodeName() + "." + event.getAgentName(),  System.currentTimeMillis());

       SLResponse response = tspaceAccessor.invokeLanguageRequest(
             "SELECT * FROM " + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + " WHERE AGENT_NAME='" + event.getAgentName() + "'");
       
       if (printSLResponseIfError("Unable to select from agents table.", response))
          return;
       
       if (response.getRowSet() == null || !response.getRowSet().next())
       {
          response = tspaceAccessor.invokeLanguageRequest("INSERT INTO " + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + 
                " VALUES('" + event.getAgentName() + "','" + event.getState() + "','" + event.getNodeName() + "','" + System.currentTimeMillis() + "')");
          sendAgentStateChange(event.getAgentName(), event.getNodeName(), event.getState());

          if (printSLResponseIfError("New RS Agent registration failed. Agent name: " + event.getAgentName(), response))
             return;
       }
       else
       {
          RowSet rowSet = response.getRowSet();
          String agentState = rowSet.getObject("AGENT_STATE").toString();
          if (!agentState.equalsIgnoreCase(event.getState().toString()))
          {
             response = tspaceAccessor.invokeLanguageRequest(
                   "UPDATE " + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + " SET LAST_ACTIVE_TIME='" + System.currentTimeMillis() + 
                            "', AGENT_STATE='" + event.getState() + "' WHERE AGENT_NAME='" + event.getAgentName() + "'");
             sendAgentStateChange(event.getAgentName(), event.getNodeName(), event.getState());
             printSLResponseIfError("Unable to update agents state", response);
          }
       }
       

       // stop all agent processes
       if (event.getState() == AgentState.STOPPED)
          stopAllAgentProcesses(tspaceAccessor, event.getNodeName());
    }

    private Object listDataspaces(String jsondata) throws Exception
    {
       ListDataspacesRequest request = deserialize(jsondata, ListDataspacesRequest.class);
       Trace.logDebug(this, "Getting list dataspaces of event scope {}, include mnode dataspaces: {}, include agents: {}.",
                      request.getEventScopes(), request.isIncludeMnodeDataspaces(), request.isIncludeAgents());

       final ListDataspacesResponse response = new ListDataspacesResponse();

       if (request.isIncludeMnodeDataspaces())
          response.addDataspaces(RedSquareAgent.listDataspaces(request, context).getDataspaces());

       if (request.isIncludeAgents())
       {
          forEachAgent((agentResponse) -> {
             response.addDataspaces(
                ((ListDataspacesResponse)((RedSquareResponseWrapper)agentResponse.getData()).getData()).getDataspaces());
             return null;
          }, "event.redsquare.ListDataspacesRequest", jsondata);
       }

       return response;
    }

    private ListUserFunctionsResponse listUserFunctions(String jsondata) throws Exception
    {
       Trace.logDebug(this, "List user functions request processing.");
       ListUserFunctionsRequest request = deserialize(jsondata, ListUserFunctionsRequest.class);

       ListUserFunctionsResponse response = new ListUserFunctionsResponse();
       if (request.isIncludeAgents())
       {
          forEachAgent((agentResponse) -> {
             response.addFunctions(((ListUserFunctionsResponse)((RedSquareResponseWrapper)agentResponse.getData()).getData()).getFunctions());
             return null;
          }, "event.redsquare.ListUserFunctionsRequest", jsondata);
       }

       return response;
    }

    private void forEachAgent(RedSquareAgent.FunctionWithException<DataEvent> function, String eventId, String jsondata) throws Exception
    {
       SLResponse slResponse =
          tspaceAccessor.invokeLanguageRequest(
             "SELECT AGENT_NODE, AGENT_NAME FROM " + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + " WHERE AGENT_STATE <> 'STOPPED'");

       if (printSLResponseIfError("Unable to extract agents definitions from '" + AGENTS_TABLE_NAME + "' table.", slResponse))
          return;

       RowSet rowSet = slResponse.getRowSet();
       while(rowSet.next())
       {
          String agentNodeName = rowSet.getString(1);
          String agentName = rowSet.getString(2);

          ServiceAccessor agentServiceAccessor = null;
          try
          {
             agentServiceAccessor = connection.createServiceAccessor(agentNodeName, "RedSquareAgent", agentName);

             DataEvent event = (DataEvent)datagramFactory.createEvent("event.redsquare.request");
             event.setData(new RedSquareRequestWrapper(eventId, jsondata));

             DataEvent serviceResponse = (DataEvent)agentServiceAccessor.invokeServiceRequest(event);
             function.execute(serviceResponse);
          }
          catch(Exception exception)
          {
             Trace.logError(this, "Failed to get dataspaces list for agent {}.", agentNodeName);
             Trace.logException(this, exception, true);
          }
          finally
          {
             if (agentServiceAccessor != null)
             {
                try
                {
                   agentServiceAccessor.close();
                }
                catch(Exception exception)
                {
                }
             }
          }
       }
    }

    private void stopAllAgentProcesses(DataspaceAccessor accessor, String nodeName) throws FabricComponentAccessorException
    {
       SLResponse response = accessor.invokeLanguageRequest("UPDATE " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME) + 
             " SET PROCESS_STATE='STOPPED' WHERE PROCESS_NODE='" + nodeName + "'");
       printSLResponseIfError("Unable to update process types table.", response);
       someProcessTypeStateChanged = true;
    }



    
    ///////////////////////////////////
    
    protected void initializeSystemTablesAndAccessors() throws Exception
    {
       ctx.logInfo("Initialization of system tables and accessors...");

       DataspaceManager dsManager = context.getDataspaceManager();
       if (dsManager.lookup(tableSpaceName) == null)
       {
          ctx.logInfo("Table space '" + tableSpaceName + "' does not exist. Recreation...");
          dsManager.createDataspace(DataspaceType.TSPACE, tableSpaceName, EventScope.OBSERVABLE);
          ctx.logInfo("Table space '" + tableSpaceName + "' created.");
       }

       if (dsManager.lookup(queueSpaceName) == null)
       {
          ctx.logInfo("Queue space '" + queueSpaceName + "' does not exist. Recreation...");
          dsManager.createDataspace(DataspaceType.QSPACE, queueSpaceName, EventScope.GLOBAL);
          ctx.logInfo("Queue space '" + queueSpaceName + "' created.");
       }

       qspaceAccessor = connection.createDataspaceAccessor(DataspaceType.QSPACE, queueSpaceName);
       tspaceAccessor = connection.createDataspaceAccessor(DataspaceType.TSPACE, tableSpaceName);

       InputStream in = RedSquareManager.class.getResourceAsStream(INSTALL_FILE_NAME);
       BufferedReader br = new BufferedReader(new InputStreamReader(in));

       String strLine;

       while ((strLine = br.readLine()) != null)
       {
          Trace.logDebug(this, "Processing SQL line from file: " + strLine);
          printSLResponseIfError("Failed to execute " + strLine, tspaceAccessor.invokeLanguageRequest(strLine));
       }
       in.close();

       if (qspaceAccessor.lookupCollection(notifQueueName) == null)
       {
          String req = "CREATE PERSISTENT EVENT QUEUE " + escapeDataSpaceObjectName(notifQueueName) + " CONSTRAINED BY " +
                       escapeDataSpaceObjectName(NOTIFICATION_EVENT_ID) + " WITH SOURCE EVENT AS BLOB CONSUMER EVENT SCOPE GLOBAL";
          Trace.logDebug(this, req);
          printSLResponseIfError("Failed to execute " + req, qspaceAccessor.invokeLanguageRequest(req));
       }

       checkSystemTable(PROCESS_TYPES_TABLE_NAME);
       checkSystemTable(AGENTS_TABLE_NAME);
       checkSystemTable(USERS_TABLE_NAME);
       checkSystemTable(USERS_ORGANIZATIONS_TABLE_NAME);

       checkSystemQueue(notifQueueName);
       ctx.logInfo("System tables initialized and accessors opened.");
    }

    protected void checkSystemTable(String name) throws Exception
    {
       if (tspaceAccessor.lookupCollection(name) == null)
       {
          ctx.logError("Table '" + name + "' does not exist. Please check the installation scripts.");
          throw new RuntimeException("System table '" + name + "' does not exist.");
       }
    }

    protected void checkSystemQueue(String queueName) throws Exception
    {
       if (qspaceAccessor.lookupCollection(queueName) == null)
       {
          ctx.logError("Queue '" + queueName + "' does not exist. Please check initialization.");
          throw new RuntimeException("System queue '" + queueName + "' does not exist.");
       }
    }
    
    private void pingAgents()
    {
       ctx.logInfo("Pinging red square agents.");
       try
       {
          DataEvent statusEvent = (DataEvent) datagramFactory.createEvent("event.redsquare.agent.ping");
          statusEvent.setData("ping");
          ctx.raiseEvent(statusEvent, 0);
       }
       catch (Exception exception)
       {
          ctx.logError("Unable to raise RedSquare Agent ping event");
          Trace.logException(RedSquareManager.class, exception, true);
       }
    }

    private void startAgentsDisconnectedConsumer()
    {
       FabricEventListener managerConnectedListener = new FabricEventListener()
       {
          public void onEvent(ImmutableEventDatagram event) throws FabricEventException
          {
             if (!(event instanceof FabricModeratorAdvisory))
                return;

             String nodeName = ((FabricModeratorAdvisory)event).getEntity();
             DataspaceAccessor accessor = null;
             try
             {
                ctx.logInfo("Node " + nodeName + " disconnected. Stopping its agents and processes.");

                accessor = connection.createDataspaceAccessor(DataspaceType.TSPACE, tableSpaceName);

                SLResponse response = accessor.invokeLanguageRequest(
                      "UPDATE " + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + " SET LAST_ACTIVE_TIME='" + System.currentTimeMillis() + 
                               "', AGENT_STATE='STOPPED' WHERE AGENT_NODE='" + nodeName + "'");
                printSLResponseIfError("Failed to update agents table for node " + nodeName, response);

                response = accessor.invokeLanguageRequest("SELECT AGENT_NAME FROM "
                      + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) + " WHERE AGENT_NODE='" + nodeName + "'");
                if (printSLResponseIfError("Failed to select from agents table for node " + nodeName, response))
                   return;

                RowSet rowSet = response.getRowSet(); 
                rowSet.beforeFirst();
                while(rowSet.next())
                {
                   String agentName = rowSet.getString(1);
                   sendAgentStateChange(agentName, nodeName, AgentState.STOPPED);
                   stopAllAgentProcesses(accessor, agentName);
                }
             }
             catch (Exception exception)
             {
                ctx.logError("Failed to change agents and processes to STOPPED for node " + nodeName + ". Cause: " + exception.getMessage());
             }
             finally
             {
                if (accessor != null)
                   accessor.close();
             }
          }
       };
       
       try
       {
          redSquareAgentsDisconnectedConsumer = connection.createEventAsyncConsumer("RedSquareAgentsDisconnectedConsumer", managerConnectedListener, FabricModeratorAdvisory.EVENT_ID,
                "(type = 'NODE_DISCONNECTED' OR type = 'NODE_DISCONNECTED_FORCIBLY')", EventScope.OBSERVABLE, true);
          redSquareAgentsDisconnectedConsumer.start();
       }
       catch(Exception exception)
       {
          ctx.logError("Failed to create RedSquareAgentsDisconnectedConsumer. Cause: " + exception.getMessage());
       }

    }
    
    private void stopAgentsDisconnectedConsumer()
    {
       try
       {
          if (redSquareAgentsDisconnectedConsumer != null)
          {
             redSquareAgentsDisconnectedConsumer.stop();
             connection.dropEventAsyncConsumer(redSquareAgentsDisconnectedConsumer.getName());
             redSquareAgentsDisconnectedConsumer = null;
          }
       }
       catch (Exception exception)
       {
          ctx.logError("Failed to remove RedSquareAgentsDisconnectedConsumer. Cause: " + exception.getMessage());
       }
    }
    
    /////////////////////////////////////////////

    private class AgentsStateProcessor extends MonitorWorker
    {
       protected DataspaceAccessor accessor = null;

       protected AgentsStateProcessor(String name, String description, long timeout) throws FabricException, FabricConnectionException, FabricComponentAccessorException
       {
          super(name, description, timeout);
          accessor = connection.createDataspaceAccessor(DataspaceType.TSPACE, tableSpaceName);
       }

       protected void doStop()
       {
          super.doStop();

          if (accessor != null)
          {
             accessor.close();
             accessor = null;
          }
       }
       
       protected void doExecute() throws FabricException
       {
          try
          {
             SLResponse response;
             try
             {
                response = accessor.invokeLanguageRequest("SELECT AGENT_NAME, AGENT_STATE, AGENT_NODE FROM "
                      + escapeDataSpaceObjectName(AGENTS_TABLE_NAME));
             }
             catch (FabricComponentAccessorException e)
             {
                ctx.logError("Unable to extract agent definitions from '" + AGENTS_TABLE_NAME + "' table.");
                Trace.logException(this, e, true);
                return;
             }

             if (printSLResponseIfError("Unable to extract agent definitions from '" + AGENTS_TABLE_NAME + "' table.", response))
                return;

             RowSet currentRowSet = response.getRowSet();
             while (currentRowSet.next())
             {
                long currentTime = System.currentTimeMillis();
                String agentName = (String) currentRowSet.getObject("AGENT_NAME");
                String agentNodeName = (String) currentRowSet.getObject("AGENT_NODE");
                String agentState = (String) currentRowSet.getObject("AGENT_STATE");

                Long lastActiveTime = agentsLastActiveTimes.get(agentNodeName + "." + agentName);
                if ((lastActiveTime == null || currentTime - lastActiveTime > 120000) && !agentState.equals(AgentState.STOPPED.name()))
                {
                   try
                   {
                      response = accessor.invokeLanguageRequest("UPDATE " + escapeDataSpaceObjectName(AGENTS_TABLE_NAME) +
                            " SET AGENT_STATE='" + AgentState.STOPPED + "' WHERE AGENT_NAME='" + agentName + "'");
                      printSLResponseIfError("Failed to update agents table for " + agentName + " to stopped state.", response);
                   }
                   catch (FabricComponentAccessorException e)
                   {
                      ctx.logError("Unable to update agent definitions in '" + AGENTS_TABLE_NAME + "' table.");
                      Trace.logException(this, e, true);
                   }

                   sendAgentStateChange(agentName, agentNodeName, AgentState.STOPPED);
                }
             }
          }
          catch (Throwable error)
          {
             Trace.logException(this, error, true);
          }
       }
    }

    private void sendAgentStateChange(String agentName, String agentNodeName, AgentState state)
    {
       try
       {
          DataEvent statusEvent = (DataEvent) datagramFactory.createEvent(AGENT_STATE_CHANGE_EVENT_ID);
          AgentStateEvent stateEvent = new AgentStateEvent(agentName, state);
          stateEvent.setNodeName(agentNodeName);
          statusEvent.setData(stateEvent);
          ctx.raiseEvent(statusEvent, 0);
       }
       catch (Exception stateEx)
       {
          ctx.logError("Unable to send agent state change event!");
          Trace.logException(RedSquareManager.class, stateEx, true);
       }
    }
    
    
    private class ProcessesStateProcessor extends MonitorWorker
    {
       protected DataspaceAccessor accessor = null;

       protected ProcessesStateProcessor(String name, String description, long timeout) throws FabricException, FabricConnectionException, FabricComponentAccessorException
       {
          super(name, description, timeout);
          accessor = connection.createDataspaceAccessor(DataspaceType.TSPACE, tableSpaceName);
       }

       protected void doStop()
       {
          super.doStop();

          if (accessor != null)
          {
             accessor.close();
             accessor = null;
          }
       }


       protected void doExecute() throws FabricException
       {
          try
          {
             SLResponse response;
             try
             {
                response = accessor.invokeLanguageRequest("SELECT PROCESS_TYPE, PROCESS_STATE FROM " +
                      escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME));
             }
             catch (FabricComponentAccessorException e1)
             {
                ctx.logError("Unable to extract process type definitions from '" + PROCESS_TYPES_TABLE_NAME + "' table.");
                Trace.logException(this, e1, true);
                return;
             }

             if (printSLResponseIfError("Unable to extract process type definitions from '" + PROCESS_TYPES_TABLE_NAME + "' table.", response))
                return;

             RowSet currentRowSet = response.getRowSet();
             while (currentRowSet.next())
             {
                long currentTime = System.currentTimeMillis();

                String processType = (String) currentRowSet.getObject("PROCESS_TYPE");
                String processState = (String) currentRowSet.getObject("PROCESS_STATE");
                Long lastActiveTime = processesLastActiveTimes.get(processType);
                if ((lastActiveTime == null || currentTime - lastActiveTime > 120000) && !processState.equals(ProcessQueueState.STOPPED.name()))
                {
                   try
                   {
                      accessor.invokeLanguageRequest("UPDATE " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME)
                            + " SET PROCESS_STATE='" + ProcessQueueState.STOPPED + "' WHERE PROCESS_TYPE='" + processType + "'");
                      printSLResponseIfError("Unable to update process type definitions in '" + PROCESS_TYPES_TABLE_NAME+ "' table.", response);

                   }
                   catch (FabricComponentAccessorException e)
                   {
                      ctx.logError("Unable to update process type definitions in '" + PROCESS_TYPES_TABLE_NAME+ "' table.");
                      Trace.logException(this, e, true);
                      return;
                   }
                   someProcessTypeStateChanged = true;
                }
             }

             if (someProcessTypeStateChanged)
             {
                sendOperationStatusEvent(new ProgressNotification("confirmingChanges", "", "Applying Changes...", "OK",""));
                Trace.logDebug(RedSquareManager.class, "Some process type status changed, sending out status object!");
                try
                {
                   response = accessor.invokeLanguageRequest("SELECT ID, PROCESS_STATE, PROCESS_ORGANIZATION FROM " + escapeDataSpaceObjectName(PROCESS_TYPES_TABLE_NAME));
                }
                catch (FabricComponentAccessorException e1)
                {
                   Trace.logDebug(this, "Unable to extract process type definitions from '" + PROCESS_TYPES_TABLE_NAME+ "' table.");
                   Trace.logException(this, e1, true);
                   return;
                }

                if (printSLResponseIfError("Unable to extract process type definitions from '" + PROCESS_TYPES_TABLE_NAME + "' table.", response))
                   return;

                currentRowSet = response.getRowSet();
                RedSquareRowSet resultSet;
                try
                {
                   resultSet = RowSetUtils.initFromRowSet(currentRowSet);
                }
                catch (IllegalStateException e)
                {
                   Trace.logException(RedSquareManager.class, e, true);
                   return;
                }

                try
                {
                   DataEvent statusEvent = (DataEvent) datagramFactory.createEvent(PROCESS_STATE_CHANGE_EVENT_ID);
                   statusEvent.setData(resultSet);
                   ctx.raiseEvent(statusEvent, 0);
                }
                catch (Exception stateEx)
                {
                   ctx.logError("Unable to send process state change event!");
                   Trace.logException(RedSquareManager.class, stateEx, true);
                }
                someProcessTypeStateChanged = false;
             }
          }
          catch (Throwable error)
          {
             Trace.logException(this, error, true);
          }
       }
    }

    private void sendOperationStatusEvent(ProgressNotification notification)
    {
       try
       {
          DataEvent event = (DataEvent)datagramFactory.createEvent(RED_SQUARE_AGENT_OPERATION_PROGRESS_EVENT_ID);
          event.setData(notification);
          ctx.raiseEvent(event, 0);
       }
       catch (Exception e)
       {
          Trace.logDebug(this, "Unable to send operation status event!");
          e.printStackTrace();
       }
    }

    private static String escapeDataSpaceObjectName(String toEscape)
    {
       return "\"" + toEscape + "\"";
    }   
    
    private class AgentLogsFileFilter implements FileFilter 
    {
       private Long currentTime = null;
       private Long timeDifference = null;

       public AgentLogsFileFilter( final Long currentTime, 
             final Long timeDifference )
       {
          this.currentTime = currentTime;
          this.timeDifference = timeDifference;
       }

       public boolean accept(File file)
       {
          boolean result = false;          
             
          if ((currentTime - file.lastModified()) > timeDifference)
               result = true;

          return result;
       }
    }
    
    private boolean printSLResponseIfError(String message, SLResponse response)
    {
       if (response != null && !response.isOK())
       {
          String cause = null;
          if (response.getException() != null)
             cause = response.getException().toString();
          else if (response.getText() != null)
             cause = response.getText();
          
          ctx.logError(message + " Cause: " + cause);
          
          return true;
       }
       return false;
    }

 }
