package com.streamscape.mf.agent;

import com.streamscape.Trace;
import com.streamscape.Trace.Level;
import com.streamscape.cli.ds.DataspaceAccessor;
import com.streamscape.cli.ds.DataspaceType;
import com.streamscape.cli.ds.collection.AuditQueue;
import com.streamscape.cli.ds.collection.ProcessQueue;
import com.streamscape.cli.http.HTTPClientException;
import com.streamscape.cli.tlp.FabricConnection;
import com.streamscape.cli.tlp.FabricConnectionException;
import com.streamscape.cli.tlp.FabricConnectionFactory;
import com.streamscape.cli.tlp.FabricConnectionFactoryException;
import com.streamscape.cli.xmpp.XMPPClientException;
import com.streamscape.ds.SqlInvariants;
import com.streamscape.ds.schema.collection.qspace.pqueue.ProcessQueueException;
import com.streamscape.ds.schema.collection.qspace.pqueue.ProcessQueueRecipientsTableManager.RecipientInfo;
import com.streamscape.ds.schema.collection.qspace.pqueue.ProcessState;
import com.streamscape.ds.schema.collection.qspace.pqueue.ProcessStateChange;
import com.streamscape.lib.concurrent.worker.MonitorWorker;
import com.streamscape.lib.selector.SelectorExpression;
import com.streamscape.lib.selector.SelectorFormatException;
import com.streamscape.lib.selector.parser.SelectorParser;
import com.streamscape.lib.utils.Pair;
import com.streamscape.lib.utils.StringUtils;
import com.streamscape.lib.zip.ZipArchiveEntry;
import com.streamscape.lib.zip.ZipArchiveOutputStream;
import com.streamscape.mf.agent.enums.*;
import com.streamscape.mf.agent.sco.SCOProperties;
import com.streamscape.mf.agent.sdo.*;
import com.streamscape.mf.manager.sdo.NotificationConfigEntryRequest;
import com.streamscape.mf.manager.sdo.PortalNotification;
import com.streamscape.mf.utils.ConfigurationChecker;
import com.streamscape.mf.utils.LogsHelper;
import com.streamscape.mf.utils.RowSetUtils;
import com.streamscape.omf.FactoryManagerException;
import com.streamscape.omf.ObjectMediationException;
import com.streamscape.omf.SerialVersionMismatchException;
import com.streamscape.omf.java.JSerializerException;
import com.streamscape.omf.json.JSONSerializerException;
import com.streamscape.omf.json.jackson.JSONSerializer;
import com.streamscape.omf.mapper.SemanticMapperException;
import com.streamscape.omf.serializer.SerializerException;
import com.streamscape.omf.xml.XSerializer;
import com.streamscape.omf.xml.XSerializerException;
import com.streamscape.repository.RepositoryException;
import com.streamscape.repository.cache.IllegalStateException;
import com.streamscape.repository.cli.RepositoryAccessorException;
import com.streamscape.runtime.RuntimeContext;
import com.streamscape.sdo.*;
import com.streamscape.sdo.advisory.*;
import com.streamscape.sdo.enums.Severity;
import com.streamscape.sdo.event.*;
import com.streamscape.sdo.event.AuditEvent.ContentType;
import com.streamscape.sdo.excp.*;
import com.streamscape.sdo.mf.admin.DatagramFactoryException;
import com.streamscape.sdo.mf.admin.DatagramFactoryManagerException;
import com.streamscape.sdo.mf.admin.PrototypeFactory;
import com.streamscape.sdo.mf.admin.SemanticTypeFactoryException;
import com.streamscape.sdo.operation.SLResponse;
import com.streamscape.sdo.rowset.RowSet;
import com.streamscape.sdo.utils.SDOUtils;
import com.streamscape.sef.*;
import com.streamscape.sef.accessor.FabricComponentAccessorException;
import com.streamscape.sef.dataspace.DataspaceComponentException;
import com.streamscape.sef.dataspace.DataspaceManager;
import com.streamscape.sef.dataspace.DataspaceManagerException;
import com.streamscape.sef.dispatcher.AbstractSystemService;
import com.streamscape.sef.enums.ComponentState;
import com.streamscape.sef.enums.EventScope;
import com.streamscape.sef.exchange.FabricAddress;
import com.streamscape.sef.moderator.ComponentModel;
import com.streamscape.sef.moderator.ComponentReference;
import com.streamscape.sef.moderator.Moderator;
import com.streamscape.sef.network.http.server.utils.HTTPUtils;
import com.streamscape.sef.network.mf.admin.AcceptorException;
import com.streamscape.sef.network.mf.admin.AcceptorFactoryException;
import com.streamscape.sef.network.mf.admin.AcceptorFactoryManagerException;
import com.streamscape.sef.security.SecurityManagerException;
import com.streamscape.sef.service.ServiceContext;
import com.streamscape.sef.service.ServiceContextException;
import com.streamscape.sef.service.ServiceManagerException;
import com.streamscape.sef.trace.LogEventSender;
import com.streamscape.sef.trace.TraceConfigurator;
import com.streamscape.sef.trace.TraceRecord;
import com.streamscape.service.osf.eim.mf.EventIdentityPluginManagerException;
import com.streamscape.slex.SemanticLexiconException;
import com.streamscape.slex.slang.SLSessionException;
import org.apache.commons.io.FileUtils;

import java.io.*;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import static com.streamscape.sef.ErrorCodes.SEF_SERVICE_INITIALIZATION_FAILED;

/**
 * <p>Title: Red Square True View Console Agent Service</p>
 * 
 * <p>Description: Service which feeds portal from node side</p>
 * 
 * <p>Copyright: Copyright (c) 2010</p>  
 * 
 * <p>Company: StreamScape Technologies</p>
 * 
 * @author Andrey Solovyev
 * @version 3.4
 * @since 3.2
 */
 // -------------------------------------------------------------------------------------------------------------------
 //     DATE     |      AUTHOR      |                              SUBJECT
 // -------------------------------------------------------------------------------------------------------------------
 //  11.04.2011  | Andrey Solovyev  | Creation from RedSquareAgent
 // -------------------------------------------------------------------------------------------------------------------
 //  07.07.2011  | Sergey Zakharov  | Rewritten to comply with the latest dataspace.
 // -------------------------------------------------------------------------------------------------------------------
 //  12.07.2011  | Sergey Zakharov  | Updated AgentRegisterNotificationWorker and ServiceStateCheckWorker to use
 //              |                  | their own dataspace accessors.
 // -------------------------------------------------------------------------------------------------------------------
 //  15.07.2011  | Sergey Zakharov  | Fixed consumer creation for the process queue.
 // -------------------------------------------------------------------------------------------------------------------
 //  15.07.2011  | Sergey Zakharov  | Fixed recipient creation for the process queue.
 // -------------------------------------------------------------------------------------------------------------------
 //  20.07.2011  | Sergey Zakharov  | Updated 'getProcessMessage' method implementation to use new 'query event'
 //              |                  | dataspace operation.
 // -------------------------------------------------------------------------------------------------------------------
 //  27.07.2011  | Sergey Zakharov  | ECR 00000408: Dataspace identifiers should be case sensitive.
 // -------------------------------------------------------------------------------------------------------------------
 //  03.08.2011  | Mikhail Filichev | ECR 00000402: Method 'createDataEventAsNew' of EventDatagramFactory should be removed.
 // -------------------------------------------------------------------------------------------------------------------
 //  09.08.2011  | Sergey Zakharov  | Added dynamic check of necessary semantic types and event prototypes on startup.
 // -------------------------------------------------------------------------------------------------------------------
 //  18.10.2011  | Sergey Zakharov  | Fixed issue with SQL auditing.
 // -------------------------------------------------------------------------------------------------------------------
 //  21.11.2011  | Sergey Zakharov  | Fixed 'processAdvisory' method implementation.
 // -------------------------------------------------------------------------------------------------------------------
 //  23.11.2011  | Sergey Zakharov  | Fixed 'getProcessesAuditInfo' method implementation.
 // -------------------------------------------------------------------------------------------------------------------
 //  20.12.2011  | Andrey Solovyev  | Moving to package com.streamscape.mf
 // -------------------------------------------------------------------------------------------------------------------
 //  02.02.2012  | Sergey Zakharov  | Fixed process instance change event trigger on the process queue.
 // -------------------------------------------------------------------------------------------------------------------
 //  02.03.2012  | Mikhail Filichev | ECR 00000490: Error Codes should be used in Runtime API.
 // -------------------------------------------------------------------------------------------------------------------
 //  15.06.2012  | Sergey Zakharov  | EBF 00000764: RedSquareAgent causes dataspace deadlock by incorrect using of accessor.
 // -------------------------------------------------------------------------------------------------------------------
 //  16.07.2012  | Ruslan Boyarskiy | EBF 00000803: "java.lang.ClassCastException: java.lang.Long cannot be cast to 
 //              |                  | java.lang.Integer" exception.
 // -------------------------------------------------------------------------------------------------------------------
 //  12.08.2012  | Ruslan Boyarskiy | ECR 00000653: Logs for agent nodes downloading.
 // -------------------------------------------------------------------------------------------------------------------
 //  21.10.2012  | Ruslan Boyarskiy | ECR 00000911: Need to implement purge process queue functionality
 //              |                  | ECR 00000574: Reoffer mechanism implementation in portal.
 // -------------------------------------------------------------------------------------------------------------------
 //  27.10.2012  | Ruslan Boyarskiy | ECR 00000927: Need to print exception's traces in java part of RS
 // -------------------------------------------------------------------------------------------------------------------
 //  26.10.2012  | Sergey Zakharov  | Added check for LOCKED_FOR_OFFER process state on removal to prevent putting 
 //              |                  | the process queue into inconsistent state.
 // -------------------------------------------------------------------------------------------------------------------
 //  01.11.2012  | Ruslan Boyarskiy | EBF 00000739: Error during notification configuration in RedSquare: 
 //              |                  | Unable to Create entry!'.
 // ------------------------------------------------------------------------------------------------------------------- 
 //  10.11.2012  | Ruslan Boyarskiy | ECR 00000951: Implement Suspend/Resume functionality for process queue
 // -------------------------------------------------------------------------------------------------------------------
 //  11.12.2012  | Ruslan Boyarskiy | EBF 00001001: RS4.2: retryDiscard page should show error instead of exception
 // -------------------------------------------------------------------------------------------------------------------
 //  24.12.2012  | Sergey Zakharov  | Optimized 'getProcessState' event handler implementation.
 // -------------------------------------------------------------------------------------------------------------------
 //  28.01.2013  | Sergey Zakharov  | Added closing of notification consumers on service stop.
 // -------------------------------------------------------------------------------------------------------------------
 //  18.11.2013 | Mikhail Filichev | ECR 00001208: Predefined Audit and Exception event prototypes should be added.
 // -------------------------------------------------------------------------------------------------------------------
 //  21.01.2014 | Mikhail Filichev | ECR 00000387: ExceptionEventDatagram should be extended.
 // -------------------------------------------------------------------------------------------------------------------
 //  26.11.2014 | Nikita Kutuzov   | SAE-124 RedSquare should be enhanced.
 // -------------------------------------------------------------------------------------------------------------------
 //  07.10.2016 | Mikhail Filichev | ECR SAE-584: Log Monitor should be integrated to Activator.
 // -------------------------------------------------------------------------------------------------------------------
 public class RedSquareAgent extends AbstractSystemService
 {
    public static final int        MAX_TEXT_FIELD_WIDTH                               = 80;
    private static final int       ZIP_LOG_FILE_MAX_SIZE                              = 5 * 1024 * 1024; // ~ 5mb
    
    // event ids
    public final static String     PROCESS_STATE_CHANGE_EVENT_ID                      = "event.redsquare.agent.ProcessStateChange";
    public final static String     NOTIFICATION_EVENT_ID                              = "event.redsquare.notification";
    public final static String     PROCESS_INSTANCE_CHANGE_EVENT_ID                   = "event.redsquare.agent.ProcessListChange";
    public final static String     RED_SQUARE_AGENT_REGISTER_EVENT_ID                 = "event.redsquare.agent.Register";
    public final static String     RED_SQUARE_AGENT_OPERATION_PROGRESS_EVENT_ID       = "event.redsquare.agent.OperationProgress";
    public final static String     RED_SQUARE_SERVICE_STATE_CHANGE_EVENT_ID           = "event.redsquare.agent.ServiceStateChange";
    public final static String     RED_SQUARE_LOG_LINE_EVENT_ID                       = "event.redsquare.agent.LogLine";
    public final static String     PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID = "event.instance.change";
    public final static String     AGENT_GET_LOG_FILE_REQUEST_EVENT_ID                = "event.redsquare.agent.GetLogFileRequest";
    public final static String     AGENT_GET_LOG_FILE_RESPONSE_EVENT_ID               = "event.redsquare.agent.GetLogFileResponse";    

    // table names
    private final String           NOTIFICATIONS_TABLE_NAME                           = "rs$Notifications";
    private final String           SERVICE_TYPE_TABLE_NAME                            = "rs$ServiceTypes";
    private final String           PROCESS_CONFIGURATION_TABLE_NAME                   = "rs$ProcessConfiguration";

    //first names for operations using progress notifications
    private final String           INSTALL_FIRST_OPERATION_NAME                       = "installAgentStart";
    private final String           DEPLOY_FIRST_OPERATION_NAME                        = "deployProcessStart";
    private final String           UNDEPLOY_FIRST_OPERATION_NAME                      = "undeployProcessStart";
    private final String           PROVISION_FIRST_OPERATION_NAME                     = "doProvisionStart";
    private final String           CREATE_QUEUE_FIRST_OPERATION_NAME                  = "createQueueStart";
    private final String           DROP_QUEUE_FIRST_OPERATION_NAME                    = "dropQueueStart";
    private final String           REMOVE_PROCESS_TYPE_FIRST_OPERATION_NAME           = "removeProcessStart";

    private final static String    SERVICE_GROUP_PROPERTY                             = "service.group";
    
    private static EventConsumer  logConsumer                                         = null;

    private final String           FILE_PATH                                          = RedSquareAgent.class.getPackage().getName().replaceAll("\\" +
                                                                                                                                               ".", "/");
    private final String           INSTALL_FILE_NAME                                  = "/" + FILE_PATH + "/InstallRedSquareAgent.sql";

    protected String               processQueueSuffix                                 = ".Process";
    protected String               auditQueueName                                     = "";
    protected String               sqlAuditQueueName                                  = "";
    protected String               userOperationAuditQueueName                             = "";

    protected FabricConnection     connection                                         = null;
    protected DataspaceAccessor    tspaceAccessor                                     = null;
    protected DataspaceAccessor    qspaceAccessor                                     = null;
    protected DataspaceAccessor    tspaceAccessorForNotifications                     = null;
    protected final Object         tspaceAccessorForNotificationsMutex                = new Object();
    
    protected EventDatagramFactory datagramFactory                                    = null;
    protected Moderator            moderator                                          = null;

    protected String               tableSpaceName                                     = null;
    protected String               queueSpaceName                                     = null;

    protected Long                 infoSendInterval                                   = null;
    protected Long                 serviceStateCheckInterval                          = null;
    protected Long                 agentStateCheckInterval                            = null;
    protected Long                 agentStateDuplicateSendInterval                    = null;

    protected XSerializer          serializer                                         = null;

    private volatile boolean       needToNotifyAboutProcessInstanceChanges            = false;

    private String                 thisNode                                           = null;

    AgentRegisterNotificationWorker agentRegisterNotificationWorker                    = null;
    ServiceStateCheckWorker         serviceStateCheckWorker                            = null;
    AuditQueue                      userOperationQueue                                 = null;

    private JSONSerializer jsonSerializer                                     = null;
    
    private String                          defaultNotificationConsumerName           = "RedSquare_Notification_Consumer_";
    private GroupNotificationsListener      groupNotificationsListener;
    private EventAsyncConsumer              groupNotificationsConsumer;
    private Map<String, EventAsyncConsumer> notificationsConsumers;
    
    private ProcessQueueStateChangeSender   processQueueStateChangeSender;
    private RedSquareAgentRegisterSender    redSquareAgentRegisterSender;
    static
    {
       try
       {
          ConfigurationChecker.checkSemanticTypes();
          ConfigurationChecker.checkEventPrototypes();
       }
       catch (Exception ex)
       {
          Trace.logException(RedSquareAgent.class, ex, true);
       }
    }

    public RedSquareAgent()
    {
    }

    public int getMinorBuild()
    {
       return Version.getBuild();
    }

    public int getMajorVersion()
    {
       return Version.getMajorVersion();
    }

    public int getMinorVersion()
    {
       return Version.getMinorVersion();
    }

    public String getVersion()
    {
       return Version.getVersionString();
    }

    protected void doInit() throws ServiceFrameworkException
    {
       try
       {
          ConfigurationChecker.createTriggers(ctx, "/resources/RedSquareAgentTriggers.slang");

          connection = new FabricConnectionFactory().createConnection();
          connection.open();

          tableSpaceName = ctx.lookupStringProperty(SCOProperties.TABLE_SPACE_NAME);
          ctx.logInfo("Table space name '" + tableSpaceName + "'.");

          queueSpaceName = ctx.lookupStringProperty(SCOProperties.QUEUE_SPACE_NAME);
          ctx.logInfo("Queue space name '" + queueSpaceName + "'.");

          if (sco.hasProperty(SCOProperties.AGENT_STATE_CHECK_INTERVAL))
          {
             agentStateCheckInterval = ctx.lookupNumericProperty(SCOProperties.AGENT_STATE_CHECK_INTERVAL);
             ctx.logInfo("Interval for agent state check is '" + agentStateCheckInterval + "'.");
          }
          else
          {
             agentStateCheckInterval = 10000L;
             ctx.logInfo("Interval for agent state check is not specified, setting to default " + agentStateCheckInterval + "ms.");
          }

          if (sco.hasProperty(SCOProperties.AGENT_STATE_DUPLICATE_SEND_INTERVAL))
          {
             agentStateDuplicateSendInterval = ctx.lookupNumericProperty(SCOProperties.AGENT_STATE_DUPLICATE_SEND_INTERVAL);
             ctx.logInfo("Interval for agent state duplicate send is '" + agentStateDuplicateSendInterval + "'.");
          }
          else
          {
             agentStateDuplicateSendInterval = 30000L;
             ctx.logInfo("Interval for agent state duplicate send is not specified, setting to default " + agentStateDuplicateSendInterval + "ms.");
          }

          if (sco.hasProperty(SCOProperties.SERVICE_STATE_CHECK_INTERVAL))
          {
             serviceStateCheckInterval = ctx.lookupNumericProperty(SCOProperties.SERVICE_STATE_CHECK_INTERVAL);
             ctx.logInfo("Interval for service state checks is '" + serviceStateCheckInterval + "'.");
          }
          else
          {
             serviceStateCheckInterval = 5000L;
             ctx.logInfo("Interval for service state check is not specified setting to default " + serviceStateCheckInterval + "ms.");
          }

          serializer = context.getXSerializerFactory().createSerializer("RedSquareAgentSerializer");
          serializer.mapAttribute("SemanticType", "class");

          thisNode = connection.getModerator().getFabricNode().getName();
          Trace.logDebug(this, "Red Square agent is installed on '" + thisNode + "' node");

          connection.bindProducerFor(AuditEvent.USER_EVENT_ID);

          datagramFactory = context.getEventDatagramFactory();
          moderator = context.getModerator();
          auditQueueName = context.getName() + ".Audit";
          sqlAuditQueueName = context.getName() + ".SQLAudit";
          userOperationAuditQueueName = context.getName() + ".UserAudit";
          initializeSystemTablesAndAccessors();

          jsonSerializer = HTTPUtils.getJsonSerializerForFabric();
       }
       catch (Exception error)
       {
          throw new ServiceFrameworkException(SEF_SERVICE_INITIALIZATION_FAILED, error.getMessage());
       }
    }

    public void destroy()
    {
       try
       {
          if (tspaceAccessor != null)
             tspaceAccessor.close();

          if (tspaceAccessorForNotifications != null)
             tspaceAccessorForNotifications.close();
          
          if (qspaceAccessor != null)
             qspaceAccessor.close();

          if (connection != null)
             connection.close();
       } 
       catch (FabricConnectionException exception)
       {
          Trace.logException(this, exception, true);
       }
    }

    public void start()
    {
       ctx.logDebug("Starting RS Agent " + thisNode + " heartbeat threads.");
       try
       {
          ctx.logDebug("Starting service state check heartbeat thread.");
          serviceStateCheckWorker = new ServiceStateCheckWorker("ServiceStateCheckThread",
                "Checks the state of all services and updates service table", serviceStateCheckInterval);

          serviceStateCheckWorker.start();
       } 
       catch (Exception e)
       {
          ctx.logError("Unable to set " + serviceStateCheckInterval
                + " timeout for ServiceStateCheckThread.");
       }

       try
       {
          processQueueStateChangeSender = new ProcessQueueStateChangeSender();
          redSquareAgentRegisterSender = new RedSquareAgentRegisterSender();
          
          ctx.logDebug("Starting agent heartbeat thread.");
          agentRegisterNotificationWorker = new AgentRegisterNotificationWorker( "AgentRegistrationsAndProcessNotifications",
                "Send agent state each time interval, send processes state every time interval", agentStateCheckInterval);
          agentRegisterNotificationWorker.start();
       }
       catch (Exception e)
       {
          ctx.logError("Unable to set " + agentStateCheckInterval + " timeout for AgentRegisterNotifictionWorker.");
       }

       try
       {
          ctx.logDebug("Opening listeners for registered notifications...");
          startGroupNotificationsConsumer();            
       } 
       catch (Exception e)
       {
          ctx.logError("Unable to open listeners for notifications. " + e.getMessage());
       }        
    }

    public void stop()
    {
       ctx.logDebug("Stopping notifications listeners...");
       try
       {
          stopGroupNotificationsConsumer();
          stopNotificationConsumers();
       }
       catch (Exception error)
       {
          Trace.logException(this, error, false);
       }

       ctx.logDebug("Stopping RS Agent heartbeat threads.");
       if (serviceStateCheckWorker != null)
       {
          serviceStateCheckWorker.stop();
          serviceStateCheckWorker = null;
       }

       if (agentRegisterNotificationWorker != null)
       {
          agentRegisterNotificationWorker.stop();
          agentRegisterNotificationWorker = null;
       }

       try
       {
          Thread.sleep(1000); // allow consumers to finish their work
       }
       catch (InterruptedException exception)
       {
          Thread.currentThread().interrupt();
       } 
       ctx.logDebug("RS Agent heartbeat threads stopped.");
    }

    public RedSquareResponseWrapper dispatchRequest(ImmutableEventDatagram event) throws Exception
    {
       Object result = null;
       RedSquareRequestWrapper wrapper = (RedSquareRequestWrapper) getDataFromImmutable(event);
       if (wrapper == null)
          throw new Exception("Null data in redsquare request.");
       if (wrapper.getId() == null)
          throw new Exception("Null id in redsquare request data.");
       switch(wrapper.getId())
       {
          case "event.redsquare.CreateQueueRequest":
             result = createQueue(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.UpdateQueueConsumerRequest":
             result = updateQueueConsumer(wrapper.getJsondata());
             break;
          case "event.redsquare.RemoveRecipientRequest":
             result = removeRecipient(wrapper.getJsondata());
             break;
          case "event.redsquare.DropQueueRequest":
             result = dropQueue(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.CreateRecipientRequest":
             result = createRecipient(wrapper.getJsondata());
             break;
          case "event.redsquare.SuspendProcessQueueRequest":
             result = suspendQueue(wrapper.getJsondata());
             break;
          case "event.redsquare.ResumeProcessQueueRequest":
             result = resumeQueue(wrapper.getJsondata());
             break;
          case "event.redsquare.PurgeProcessQueueRequest":
             result = purgeQueue(wrapper.getJsondata());
             break;
          case "event.redsquare.RemoveProcessTypeRequest":
             result = removeProcessType(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.ProcessMessageRequest":
             result = getProcessMessage(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessMessagePayloadRequest":
             result = getProcessMessagePayload(wrapper.getJsondata());
             break;
          case "event.redsquare.RemoveProcessRequest":
             result = removeProcess(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.DiscardProcessRequest":
             result = discardProcess(wrapper.getJsondata());
             break;
          case "event.redsquare.RetryProcessRequest":
             result = retryProcess(wrapper.getJsondata());
             break;
          case "event.redsquare.ReofferProcessRequest":
             result = reofferProcess(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessPropertiesRequest":
             result = getProcessProperties(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessStatisticsRequest":
             result = getProcessStatistics(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessListRequest":
             result = getProcessesList(wrapper.getJsondata());
             break;
          case "event.redsquare.AuditStepRequest":
             result = getDetailedAudit(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessAuditRequest":
             result = getProcessesAuditInfo(wrapper.getJsondata());
             break;
          case "event.redsquare.getServicesRequest":
             result = getServices(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessProvisionRequest":
             result = doProvision(getUserName(event));
             break;
          case "event.redsquare.getServiceGroupsRequest":
             result = getServiceGroups(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessInstallRequest":
             result = installAgent(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.DeployProcessRequest":
             result = deployProcess(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.UndeployProcessRequest":
             result = unDeployProcess(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.logSubscriptionRequest":
             result = logsSubscriptionHandler(wrapper.getJsondata());
             break;
          case "event.redsquare.errorLogsRequest":
             result = getLatestLogs(wrapper.getJsondata());
             break;
          case "event.redsquare.EventListRequest":
             result = getEventsList(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessConfigurationCreate":
             result = createProcessConfiguration(wrapper.getJsondata(), getUserName(event));
             break;
          case "event.redsquare.ProcessConfigurationUpdate":
             result = updateProcessConfigurationObject(wrapper.getJsondata());
             break;
          case "event.redsquare.ProcessConfigurationRequest":
             result = getProcessConfigurationObject(wrapper.getJsondata());
             break;
          case "event.redsquare.NotificationActionRequest":
             result = doNotificationConfigAction(wrapper.getJsondata());
             break;
          case "event.redsquare.NotificationConfigurationRequest":
             result = getNotificationConfigList(wrapper.getJsondata());
             break;
          case "event.redsquare.ListDataspacesRequest":
             result = listDataspaces(wrapper.getJsondata());
             break;
          case "event.redsquare.ListUserFunctionsRequest":
             result = listUserFunctions(wrapper.getJsondata());
             break;
          case "event.redsquare.CallUserFunction":
             result = callUserFunction(wrapper.getJsondata());
             break;
          case "event.redsquare.InvokeQueryRequest":
             result = invokeQuery(wrapper.getJsondata());
             break;
          default:
             ctx.logError("Unknown redsquare request id:" + wrapper.getId());
       }

       return new RedSquareResponseWrapper(result);
    }

    private <T> T deserialize(String jsondata, Class<T> clazz) throws SerializerException
    {
       return (T)jsonSerializer.deserialize(clazz, jsondata);
    }
    
    public RedSquareOperationResponse createQueue(String jsondata, String userName) throws Exception
    {
       CreateQueueRequest request = deserialize(jsondata, CreateQueueRequest.class);
       sendOperationStateEvent(new ProgressNotification(CREATE_QUEUE_FIRST_OPERATION_NAME, 
             "createQueueBeginDataBaseUpdate", "Starting queue creation...", "OK", userName));

       RedSquareOperationResponse response = new RedSquareOperationResponse();
       sendOperationStateEvent(new ProgressNotification("createQueueBeginDataBaseUpdate",
             "createQueueBeginDataBaseUpdate", "Updating queue configuration...", "RUNNING",userName));

       String queueType;
       //Only Process queue creation will be used now
       switch (request.getQueueType())
       {
          case AUDIT_QUEUE:
          {
             queueType = "AUDIT";
             break;
          }
          case PROCESS_QUEUE:
          {
             queueType = "PROCESS";
             break;
          }
          default:
          {
             Trace.logDebug(this, "Unknown queue type " + request.getQueueType() + " specified.");
             response.setErrorMessage("Unable to create queue.");
             response.setAdditionalMessage("Unknown queue type " + request.getQueueType() + " specified.");
             response.setSuccess(false);
             sendOperationStateEvent(new ProgressNotification("createQueueBeginDataBaseUpdate",
                   "", "Unknown Queue Type. Queue Creation Failed!", "NOK", userName));
             return response;
          }
       }

       String req = "CREATE PERSISTENT " + queueType + " QUEUE "
             + escape(request.getQueueName()) + " CONSTRAINED BY "
             + escape(request.getEventId()) + " CONSUMER";

       if (request.getQueueType() == QueueType.PROCESS_QUEUE)
       {
          req += " max attempts " + request.getDeliveryAttempts() + " offer interval " + request.getOfferInterval() +
                " recipient timeout " + request.getTimeout() + " suspend on fail " + request.isSuspendOnFailure() + 
                " parallel degree " + request.getNumberOfConsumers();
       }

       String queueName = request.getQueueName().split("\\.")[0];
       SLResponse slResponse = qspaceAccessor.invokeLanguageRequest(req);

       if (slResponse == null || !slResponse.isOK())
       {
          printSLResponseIfError("Unable to create collection " + request.getQueueName() + " of " + request.getQueueType() + " type.", slResponse);   

          response.setErrorMessage("Unable to create queue");
          response.setAdditionalMessage("Unable to create queue");
          response.setSuccess(false);
          sendOperationStateEvent(new ProgressNotification("createQueueBeginDataBaseUpdate", "",
                "Unable to create " + request.getQueueName() + " of " + request.getQueueType() + " type. Cause: " + getSLResponseError(slResponse), "NOK", userName));
          return response;
       }

       if (request.getQueueType() == QueueType.PROCESS_QUEUE)
       {
          try
          {
             if (!PrototypeFactory.existsPrototype(PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID))
             {
                Trace.logDebug(this, PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID + " event does not exist. Adding.");
                addProcessStateChangePrototype(PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID);
                Trace.logDebug(this, PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID + " added");
             }
          } 
          catch (Exception e)
          {
             Trace.logError(this, "Can not create prototype for " + PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID);
             Trace.logException(this, e, true);
          }

          String createEventTriggerRequest = "create event trigger " 
                + "\"InstanceChangeTrigger\" on [" + request.getQueueName() + 
                "] on update as { raise event this.getCurrentEvent() on [" + PROCESS_INSTANCE_CHANGE_EVENT_FOR_TRIGGER_EVENT_ID  +"]; }";

          SLResponse triggerResponse = qspaceAccessor.invokeLanguageRequest(createEventTriggerRequest);
          printSLResponseIfError("Unable to create trigger for " + request.getQueueName()
                + " of " + request.getQueueType() + " type, request: " + createEventTriggerRequest, triggerResponse);   
       }

       sendOperationStateEvent(new ProgressNotification("createQueueBeginDataBaseUpdate", "", 
             "Updating Process Queue configuration...", "OK", userName));

       SLResponse slTableUpdateResponse = tspaceAccessor.invokeLanguageRequest("UPDATE \""
             + PROCESS_CONFIGURATION_TABLE_NAME + "\" SET CONSTRAINT_EVENT='"
             + request.getEventId() + "' WHERE PROCESS_NAME='" + queueName + "'");
       printSLResponseIfError("Unable to update configuration table!", slTableUpdateResponse);

       return response;
    }
    
    public RedSquareOperationResponse updateQueueConsumer(String jsondata) throws Exception
    {
       CreateQueueRequest request = deserialize(jsondata, CreateQueueRequest.class);
        ProcessQueue processQueue = null;
        try
        {
           processQueue = getProcessQueue(request.getQueueName());
        }
        catch(Exception exception)
        {
           return buildErrorResponse("Failed to get process queue.", exception, request.getQueueName());
        }
        
        try
        {
           if (processQueue.getParallelDegree() != request.getNumberOfConsumers())
           {
              processQueue.stop();
              processQueue.setParallelDegree(request.getNumberOfConsumers());
              processQueue.start();
           }

           processQueue.setPollerConfiguration(request.getOfferInterval(), request.getTimeout(), request.getDeliveryAttempts(), request.isSuspendOnFailure());
        }
        catch(Exception exception)
        {
           return buildErrorResponse("Failed to update process queue offer rules.", exception, request.getQueueName());
        }
        
        RedSquareOperationResponse response = new RedSquareOperationResponse();
        response.setProcessType(request.getQueueName());
        return response;
    }

    public RedSquareOperationResponse removeRecipient(String jsondata) throws FabricComponentAccessorException, SerializerException
    {
       RecipientRemovalRequest request = deserialize(jsondata, RecipientRemovalRequest.class);
       RedSquareOperationResponse response = new RedSquareOperationResponse();
       String req = "DROP RECIPIENT " + request.getName() + " ON " + escape(request.getQueueName());
       Trace.logDebug(this, req);
       
       SLResponse slResponse = qspaceAccessor.invokeLanguageRequest(req);
       if (slResponse == null || !slResponse.isOK())
       {
          printSLResponseIfError("Unable drop recipient.", slResponse);   

          response.setErrorMessage("Unable to remove recipient");
          response.setAdditionalMessage("Unable to remove recipient: " + slResponse.getException().getMessage());
          response.setSuccess(false);
          return response;
       }

       response.setAdditionalMessage("Recipient " + request.getName() + " has been removed from " + request.getQueueName());
       return response;
    }

    public RedSquareOperationResponse createRecipient(String jsondata) throws FabricComponentAccessorException, SerializerException
    {
       RecipientCreationRequest request = deserialize(jsondata, RecipientCreationRequest.class);
       RedSquareOperationResponse response = new RedSquareOperationResponse();

       if (request.isRecreate())
       {
          //TODO: if name changed, no drop will be done -> error.
          String req = "DROP RECIPIENT " + request.getName() + " ON " + escape(request.getQueueName());
          SLResponse slResponse = qspaceAccessor.invokeLanguageRequest(req);
          if (slResponse == null || !slResponse.isOK())
          {
             response.setErrorMessage("Unable to remove recipient");
             response.setAdditionalMessage("Unable to remove recipient: " + slResponse.getException().getMessage());
          }
       }

       String req = "CREATE " + (request.isCertified() ? "CERTIFIED" : "") + " RECIPIENT "
             + request.getName() + " ON QUEUE "
             + escape(request.getQueueName());

       if (request.getRoutingRules() != null && request.getRoutingRules().length() > 0)
          req += " WHEN (" + request.getRoutingRules() + ") ";
       req += " RAISE EVENT ON " + escape(request.getEventID());

       SLResponse slResponse = qspaceAccessor.invokeLanguageRequest(req);
       if (slResponse == null || !slResponse.isOK())
       {
          printSLResponseIfError("Unable to create recipient.", slResponse);   
          if (slResponse.getException() !=null && slResponse.getException().getMessage() != null) 
          {
             response.setErrorMessage("Unable to create recipient " + slResponse.getException().getMessage());
             response.setAdditionalMessage("Unable to create recipient! " + slResponse.getException().getMessage());
          }
          else if (slResponse.getText() != null)
          {
             response.setErrorMessage("Unable to create recipient " + slResponse.getText());
             response.setAdditionalMessage("Unable to create recipient! " + slResponse.getText());
          } 
          else 
          {
             response.setErrorMessage("Unable to create recipient: unknown error.");
             response.setAdditionalMessage("Unable to create recipient: unknown error.");
          }

          response.setSuccess(false);
          return response;
       }

       if (!request.isRecreate())
          response.setAdditionalMessage("Recipient has been created!");
       else
          response.setAdditionalMessage("Recipient definition has been updated!");

       return response;
    }

    public RedSquareOperationResponse dropQueue(String jsondata, String userName) throws FabricComponentAccessorException, SerializerException
    {
       DropQueueRequest request = deserialize(jsondata, DropQueueRequest.class);
       sendOperationStateEvent(new ProgressNotification(DROP_QUEUE_FIRST_OPERATION_NAME,
             "dropQueueBeginDataBaseUpdate", "Starting queue removing...", "OK", userName));

       RedSquareOperationResponse response = new RedSquareOperationResponse();
       response.setSuccess(true);

       sendOperationStateEvent(new ProgressNotification("dropQueueBeginDataBaseUpdate",
             "dropQueueBeginDataBaseUpdate", "Updating queue configuration...", "RUNNING", userName));

       SLResponse slResponse = qspaceAccessor.invokeLanguageRequest("DROP COLLECTION " + escape(request.getQueueName()));

       if (slResponse == null || !slResponse.isOK())
       {
          printSLResponseIfError("Unable to drop " + request.getQueueName() + " queue.", slResponse);   
          response.setAdditionalMessage("Unable to drop queue");
          response.setSuccess(false);
          sendOperationStateEvent(new ProgressNotification("dropQueueBeginDataBaseUpdate", "", "Unable to drop queue.", "NOK", userName));
          return response;
       }

       String queueName = request.getQueueName().split("\\.")[0];
       sendOperationStateEvent(new ProgressNotification("dropQueueBeginDataBaseUpdate", "", "Updating queue configuration...", "OK", userName));
       response.setAdditionalMessage("Queue " + request.getQueueName() + "' was dropped.");

       SLResponse slTableUpdateResponse = tspaceAccessor.invokeLanguageRequest("UPDATE \""
             + PROCESS_CONFIGURATION_TABLE_NAME
             + "\" SET CONSTRAINT_EVENT='' WHERE PROCESS_NAME='" + queueName + "'");

       printSLResponseIfError("Unable to update configuration table!", slTableUpdateResponse);   
       return response;
    }

    public RedSquareOperationResponse suspendQueue(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       ProcessQueue processQueue = null;
       try
       {
          processQueue = getProcessQueue(getProcessQueueName(process));
       }
       catch(Exception exception)
       {
          return buildErrorResponse("Failed to get process queue.", exception, process.getProcessType());
       }

       try
       {
          processQueue.suspend();
       }
       catch(Exception exception)
       {
          return buildErrorResponse("Failed to suspend process queue.", exception, process.getProcessType());
       }

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       opResponse.setProcessType(process.getProcessType());
       opResponse.setAdditionalMessage("Queue of " + process.getProcessType() + " has been suspended.");
       return opResponse;
    }

    public RedSquareOperationResponse resumeQueue(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       ProcessQueue processQueue = null;
       try
       {
          processQueue = getProcessQueue(getProcessQueueName(process));
       }
       catch(Exception exception)
       {
          return buildErrorResponse("Failed to get process queue.", exception, process.getProcessType());
       }

       try
       {
          processQueue.resume();
       }
       catch(Exception exception)
       {
          return buildErrorResponse("Failed to resume process queue.", exception, process.getProcessType());
       }

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       opResponse.setProcessType(process.getProcessType());
       opResponse.setAdditionalMessage("Queue of " + process.getProcessType() + " has been resumed.");
       return opResponse;
    }    

    public RedSquareOperationResponse purgeQueue(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException, DataspaceComponentException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);

       List<String> processesList = new ArrayList<String>();
       try
       {
          processesList = getProcessIds(process);
       }
       catch (Exception exception)
       {
          return buildErrorResponse("Failed to get process ids.", exception, process.getProcessType());
       }

       try
       {
          qspaceAccessor.executeQuery("purge queue " + escape(getProcessQueueName(process)));
       }
       catch(Exception exception)
       {
          return buildErrorResponse("Failed to purge process queue.", exception, process.getProcessType());
       }
       
       purgeAuditQueues(processesList);
       
       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       opResponse.setProcessType(process.getProcessType());
       opResponse.setAdditionalMessage("Queue of " + process.getProcessType() + " has been purged.");
       return opResponse;
    }

    private List<String> getProcessIds(ProcessIdentifier process) throws Exception
    {
       List<String> processesList = new ArrayList<String>();
       int oldFetchSize = qspaceAccessor.getFetchSize();
       try
       {
          qspaceAccessor.setFetchSize(10000);
          RowSet rowSet = qspaceAccessor.executeQuery("select ProcessId from " + escape(getProcessQueueName(process)));
          while(rowSet.next())
             processesList.add(rowSet.getString(1));
          return processesList;
       }
       finally
       {
          qspaceAccessor.setFetchSize(oldFetchSize);
       }
    }
    
    private void purgeAuditQueues(List<String> processesList)
    {
       Iterator<String> iterator = processesList.iterator();
       StringBuilder builder = new StringBuilder();
       while(iterator.hasNext())
       {
          if (builder.length() == 0)
             builder.append("(");
          else
             builder.append(",");
          builder.append("'").append(iterator.next()).append("'");
          if (builder.length() > 1024*1024*10)
          {
             purgeAuditQueues(builder);
             builder = new StringBuilder(); 
          }
       }
       purgeAuditQueues(builder);
    }    
    
    private void purgeAuditQueues(StringBuilder builder)
    {
       if (builder.length() == 0)
          return;
       builder.append(")");
       String inValues = builder.toString();
       purgeAuditQueue(inValues, auditQueueName);
       purgeAuditQueue(inValues, sqlAuditQueueName);
       purgeAuditQueue(inValues, userOperationAuditQueueName);
    }
    
    private void purgeAuditQueue(String inValues, String queueName)
    {
       try
       {
          qspaceAccessor.executeQuery("delete from " + escape(queueName) + " where CorrelationId in " + inValues);
       }
       catch (DataspaceComponentException exception)
       {
          ctx.logError("Failed to delete from '" + queueName + "'. Cause: " + exception.getMessage());
       }
    }    
    
    public RedSquareOperationResponse removeProcessType(String jsondata, String userName) throws FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier request = deserialize(jsondata, ProcessIdentifier.class);

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();

       boolean processQueueExists = true;
       List<String> processesList = new ArrayList<String>();
       try
       {
          processesList = getProcessIds(request);
       }
       catch (Exception exception)
       {
          if (exception.getMessage() != null && exception.getMessage().contains("not found: " + getProcessQueueName(request)))
             processQueueExists = false;
          else
          {
             sendOperationStateEvent(new ProgressNotification(REMOVE_PROCESS_TYPE_FIRST_OPERATION_NAME, "",
                                                              "Getting process ids...", "NOK", userName));
             return buildErrorResponse("Failed to get process ids.", exception, request.getProcessType());
          }
       }

       if (processQueueExists)
       {
          sendOperationStateEvent(new ProgressNotification(REMOVE_PROCESS_TYPE_FIRST_OPERATION_NAME, "dropProcessQueue",
                                                           "Starting process removal...", "OK", userName));

          sendOperationStateEvent(new ProgressNotification("dropProcessQueue",
                                                           "dropProcessQueue", "Dropping process queue...", "RUNNING",""));
          SLResponse response = qspaceAccessor.invokeLanguageRequest("DROP COLLECTION " + escape(getProcessQueueName(request)));
          if (response == null || !response.isOK())
          {
             Trace.logError(this, "Unable to drop process queue " + request.getProcessType());
             sendOperationStateEvent(new ProgressNotification("dropProcessQueue",
                                                              "", "Dropping process queue...", "NOK",""));
             opResponse.setSuccess(false);
             return opResponse;
          }
          sendOperationStateEvent(new ProgressNotification("dropProcessQueue",
                                                           "purgingAuditQueues", "Dropping process queue...", "OK",""));

          sendOperationStateEvent(new ProgressNotification("purgingAuditQueues", "purgingAuditQueues", "Purging audit queues...", "RUNNING", userName));
          purgeAuditQueues(processesList);
          sendOperationStateEvent(new ProgressNotification("purgingAuditQueues", "removeAgentConfiguration", "Purging audit queues...", "OK", userName));
       }
       else
       {
          sendOperationStateEvent(new ProgressNotification(REMOVE_PROCESS_TYPE_FIRST_OPERATION_NAME, "removeAgentConfiguration",
                                                           "Starting process removal...", "OK", userName));
       }


       sendOperationStateEvent(new ProgressNotification("removeAgentConfiguration",
                                                        "removeAgentConfiguration", "Removing agent configuration...", "RUNNING",""));

       SLResponse response = tspaceAccessor.invokeLanguageRequest("DELETE FROM \""
                                                                  + PROCESS_CONFIGURATION_TABLE_NAME + "\" WHERE PROCESS_NAME='" + request.getProcessType() + "'");

       if (response == null || !response.isOK())
       {
          Trace.logDebug(this, "Unable to remove process from configuration in'" + PROCESS_CONFIGURATION_TABLE_NAME + "' table.");
          opResponse.setSuccess(false);
          sendOperationStateEvent(new ProgressNotification("removeAgentConfiguration", "", "Removing agent configuration...", "NOK", userName));
          return opResponse;
       }

       sendOperationStateEvent(new ProgressNotification("removeAgentConfiguration",
                                                        "sendingEventToManager", "Removing agent configuration...", "OK", userName));

       sendOperationStateEvent(new ProgressNotification("sendingEventToManager", "sendingEventToManager", "Sending Manager event...", "RUNNING", userName));
       processQueueStateChangeSender.send(request.getProcessType(), ProcessQueueState.DELETED);
       sendOperationStateEvent(new ProgressNotification("sendingEventToManager", "", "Sending Manager event...", "OK", userName));


       opResponse.setSuccess(true);
       opResponse.setAdditionalMessage(request.getProcessType() + " is Removed");

       return opResponse;
    }

    public RedSquareRowSet getProcessMessage(String jsondata) throws IAbstractExceptionEvent, SQLException, FabricException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);

       SLResponse response = qspaceAccessor.invokeLanguageRequest("select SeqId,Created,EventGroupId,EventKey,ProcessId,State,Acknowledged from "
             + escape(process.getProcessType() + processQueueSuffix)
             + " where ProcessId='" + process.getProcessId() + "'");

       RedSquareRowSet result = new RedSquareRowSet(3, 30);
       RowSet rowSet = response.getRowSet();
       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          printSLResponseIfError("Unable to extract process message from the specified '"
                + escape(process.getProcessType() + processQueueSuffix) + "' process queue.", response);

          result.addRow(new Object[] { "Error", "error", process.getProcessId()  + " process isn't found in " + process.getProcessType() });
          return result;
       }

       if (!rowSet.first())
       {
          Trace.logDebug(this, "Unable to get request message for '" + process.getProcessId() + "' process.");
          return null;
       }

       String seqId = rowSet.getObject("SeqId").toString();
       Object timestamp = rowSet.getObject("Created");
       String processGroup =  rowSet.getString("EventGroupId");
       String eventKey =  rowSet.getString("EventKey");
       String processId =  rowSet.getString("ProcessId");
       String state = rowSet.getObject("State").toString();
       Object timestampAcknowledged = rowSet.getObject("Acknowledged");
       String eventId = null;
       long payloadSize = -1;
       
       {
          response = qspaceAccessor.invokeLanguageRequest("QUERY EVENT \"" + seqId + "\" FROM "
                + escape(process.getProcessType() + processQueueSuffix));

          if (!printSLResponseIfError("Unable to extract process message from the specified '"
                + processQueueSuffix + "' process queue.", response))
          {
             response.getRowSet().first();
             String xmlData = (String) response.getRowSet().getObject(1);
             ImmutableEventDatagram event = (ImmutableEventDatagram)serializer.deserialize(ImmutableEventDatagram.SEMANTIC_TYPE_NAME, xmlData);
             if (event != null)
                eventId = event.getEventId();
             
             String payload = getSerializedPayload(event);
             if (payload != null)
                payloadSize = payload.length();
          } 
       }

       result.addRow(new Object[] { "Timestamp",     "label",     timestamp != null ? timestamp.toString() : "unknown" });
       result.addRow(new Object[] { "",              "label",     "" });
       result.addRow(new Object[] { "Process Group", "label",     processGroup});
       result.addRow(new Object[] { "Event Id",      "textfield", eventId });
       result.addRow(new Object[] { "Event Key",     "textfield", eventKey });
       result.addRow(new Object[] { "Correlation Id","textfield", processId });
       result.addRow(new Object[] { "State",         "textfield", state });
       result.addRow(new Object[] { "Acknowledged",  "textfield", timestampAcknowledged != null ? timestampAcknowledged.toString() : null});
       result.addRow(new Object[] { "Payload Size",  "textfield", payloadSize == -1 ? "undefined" : payloadSize + " bytes"});
       
       result.compact();
       result.setTargetProcessType(process.getProcessType());
       result.setTargetProcessId(process.getProcessId());

       return result;
    }

    public RedSquareRowSet getProcessMessagePayload(String jsondata) throws IAbstractExceptionEvent, SQLException, FabricException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       RedSquareRowSet result = new RedSquareRowSet(3, 1);

       SLResponse response = qspaceAccessor.invokeLanguageRequest("select SeqId from "
             + escape(process.getProcessType() + processQueueSuffix)
             + " where ProcessId='" + process.getProcessId() + "'");

       RowSet rowSet = response.getRowSet();
       if (printSLResponseIfError("Unable to extract process message from the specified '"
             + escape(process.getProcessType() + processQueueSuffix)
             + "' process queue.", response))
       {
          result.addRow(new Object[] { "Error", "error", process.getProcessId()  + " process isn't found in " + process.getProcessType() });
          return result;
       }

       if (!rowSet.first())
       {
          Trace.logError(this, "Unable to get request message for '" + process.getProcessId() + "' process.");
          return null;
       }

       String seqId = rowSet.getObject("SeqId").toString();
       response = qspaceAccessor.invokeLanguageRequest("QUERY EVENT \"" + seqId + "\" FROM " + 
             escape(process.getProcessType() + processQueueSuffix));

       if (response == null || !response.isOK() || response.getRowSet() == null || !response.getRowSet().next())
       {
          printSLResponseIfError("Unable to extract process message from the specified '"
                + processQueueSuffix + "' process queue.", response);
          return null;
       }

       String xmlData = (String) response.getRowSet().getObject(1);
       ImmutableEventDatagram event = (ImmutableEventDatagram)serializer.deserialize(ImmutableEventDatagram.SEMANTIC_TYPE_NAME, xmlData);
       if (event == null || !(event instanceof EventDatagram))
       {
          Trace.logError(this, "Unsupported process request type.");
          return null;
       }

       result.addRow(new Object[] { "Payload", "textarea", getSerializedPayload(event) });

       result.compact();
       result.setTargetProcessType(process.getProcessType());
       result.setTargetProcessId(process.getProcessId());

       return result;
    }

    private String getSerializedPayload(ImmutableEventDatagram event) throws SecurityViolationException, SDOException, SerializerException
    {
       String serializedPayload = null;
       if (event instanceof DataEvent)
       {
          Object payload = ((DataEvent) event).getData();
          if (payload != null)
             serializedPayload = serializer.serialize(payload);
       } 
       else if (event instanceof XMLEvent)
          serializedPayload = ((XMLEvent) event).getXML();
       else
          serializedPayload = serializer.serialize(event);
       return serializedPayload;
    }

    public RedSquareOperationResponse retryProcess(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       String operation = "RETRY PROCESS " + escape(process.getProcessId())
             + " ON " + escape(process.getProcessType() + processQueueSuffix);

       if (process.getComment() != null)
       {
          //operation += " COMMENT='" + process.getComment() + "'";
       }

       SLResponse response = qspaceAccessor.invokeLanguageRequest(operation);
       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to retry process '" + process.getProcessId()
                + "' on the specified '" + process.getProcessType() + processQueueSuffix + "' process queue.", response);
          return processFailedOperation(response, process.getProcessId());
       } 

       try
       {
          sendUserAuditAndNotification("retried", process);
       }
       catch (Exception e)
       {
          Trace.logError(this, "Can not send audit event and notification for process operation. " + e.getMessage());
          Trace.logException(this, e, true);
       }

       response = qspaceAccessor.invokeLanguageRequest("RESUME QUEUE "
             + escape(process.getProcessType() + processQueueSuffix));

       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to resume '" + process.getProcessType()
                + processQueueSuffix + "' process queue.", response);
          return processFailedOperation(response, process.getProcessId());
       }

       RedSquareOperationResponse operationResponse = new RedSquareOperationResponse(process.getProcessId());
       operationResponse.setAdditionalMessage("Process " + process.getProcessId() + " was retried.");

       return operationResponse;
    }

    public RedSquareOperationResponse reofferProcess(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       String operation = "REOFFER PROCESS " + escape(process.getProcessId())
             + " ON " + escape(process.getProcessType() + processQueueSuffix);

       if (process.getComment() != null)
       {
          //operation += " COMMENT='" + process.getComment() + "'";
       }

       SLResponse response = qspaceAccessor.invokeLanguageRequest(operation);
       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to retry process '" + process.getProcessId()
                + "' on the specified '" + process.getProcessType() + processQueueSuffix
                + "' process queue.", response);   
          return processFailedOperation(response, process.getProcessId());
       } 
       try
       {
          sendUserAuditAndNotification("reoffered", process);
       }
       catch (Exception e)
       {
          Trace.logError(this, "Can not send audit event and notification for process operation. " + e.getMessage());
          Trace.logException(this, e, true);
       }

       RedSquareOperationResponse operationResponse = new RedSquareOperationResponse(process.getProcessId());
       operationResponse.setAdditionalMessage("Process " + process.getProcessId() + " was reoffered.");

       return operationResponse;
    }

    public RedSquareOperationResponse discardProcess(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       String operation = "DISCARD PROCESS " + escape(process.getProcessId())
             + " ON " + escape(process.getProcessType() + processQueueSuffix);

       SLResponse response = qspaceAccessor.invokeLanguageRequest(operation);
       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to discard process '" + process.getProcessId()
                + "' on the specified '" + process.getProcessType() + processQueueSuffix
                + "' process queue.", response);   
          return processFailedOperation(response, process.getProcessId());
       } 

       try
       {
             sendUserAuditAndNotification("discarded", process);
       } 
       catch (Exception e)
       {
          Trace.logError(this, "Can not send audit event and notification for process removal. " + e.getMessage());
          Trace.logException(this, e, true);
       }

       response = qspaceAccessor.invokeLanguageRequest("RESUME QUEUE "
             + escape(process.getProcessType() + processQueueSuffix));

       if (response == null || !response.isOK())
       {
          printSLResponseIfError("Unable to resume '" + process.getProcessType()
                + processQueueSuffix + "' process queue.", response);   
          return processFailedOperation(response, process.getProcessId());
       }

       RedSquareOperationResponse operationResponse = new RedSquareOperationResponse(process.getProcessId());
       operationResponse.setAdditionalMessage("Process " + process.getProcessId() + " was discarded successfully.");
       return operationResponse;
    }

    public RedSquareOperationResponse removeProcess(String jsondata, String userName) throws IllegalStateException, FabricComponentAccessorException, SerializerException, SQLException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       try
       {
          RowSet rowSet = qspaceAccessor.executeQuery("delete from " + escape(process.getProcessType() + processQueueSuffix)
                + " where ProcessId=? and State <> ?", process.getProcessId(), ProcessState.LOCKED_FOR_OFFER.name());
          
          if (!rowSet.next())
             return processFailedOperation("Failed to remove process.", process.getProcessId());
          
          if (rowSet.getInt(1) < 1)
          {
             rowSet = qspaceAccessor.executeQuery("select State from " + escape(process.getProcessType() + processQueueSuffix) + " where ProcessId=?", process.getProcessId());
             if (rowSet.next())
             {
                String state = rowSet.getString(1);
                if (state != null && state.equals(ProcessState.LOCKED_FOR_OFFER.name()))
                   return processFailedOperation("Failed to remove process in state '" + ProcessState.LOCKED_FOR_OFFER.name() + "'.", process.getProcessId());
                else
                   return processFailedOperation("Please try to process again.", process.getProcessId());
             }
          }  
       }
       catch(Exception exception)
       {
          String error =  "Failed to delete process '" + process.getProcessId() +
                "' from the specified '" + process.getProcessType() + processQueueSuffix + "' process queue.";
          Trace.logError(this, error + " Cause: " + exception.getMessage());
          return processFailedOperation(error, process.getProcessId());
       }
       
       try
       {
          process.setComment("Process Removal.");
          sendUserAuditAndNotification("removed", process);
       }
       catch (Exception e)
       {
          Trace.logError(this, "Can not send audit event and notification for process removal." + e.getMessage());
          Trace.logException(this, e, true);
       }

       sendOperationStateEvent(new ProgressNotification("purgingAuditQueues", "purgingAuditQueues", "Purging audit queues...", "RUNNING", userName));
       List<String> processesList = new ArrayList<String>();
       processesList.add(process.getProcessId());
       purgeAuditQueues(processesList);

       RedSquareOperationResponse operationResponse = new RedSquareOperationResponse(process.getProcessId());
       operationResponse.setProcessType(process.getProcessType());
       return operationResponse;
    }


    // TODO: rework to properties map
    public ProcessPropertiesResponse getProcessProperties(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SerializerException, SQLException
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       SLResponse response = qspaceAccessor.invokeLanguageRequest("DESCRIBE COLLECTION "
             + escape(process.getProcessType() + processQueueSuffix));

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          Trace.logDebug(this, "Unable to get queue state for " + process.getProcessType() + processQueueSuffix);
          return null;
       }

       ProcessPropertiesResponse processProperties = new ProcessPropertiesResponse();
       processProperties.setProcessType(process.getProcessType());
       RegisteredConsumerProperties consumerProperties = new RegisteredConsumerProperties();
       consumerProperties.setName("consumer");
       processProperties.addConsumer(consumerProperties);

       RowSet rowSet = response.getRowSet();
       rowSet.beforeFirst();
       while (rowSet.next())
       {
          String propName = (String) rowSet.getObject(1);
          if (propName.equalsIgnoreCase("state"))
             processProperties.setProcessState((String) rowSet.getObject(2));
          else if (propName.equalsIgnoreCase("Event"))
             processProperties.setConstraint((String) rowSet.getObject(2));
          else if (propName.equalsIgnoreCase("queue state"))
             consumerProperties.setState((String)rowSet.getObject(2));
          else if (propName.equalsIgnoreCase("Poller Offer Interval"))
             consumerProperties.setOfferInterval(Integer.parseInt(rowSet.getObject(2).toString()));
          else if (propName.equalsIgnoreCase("Poller Max Attempts"))
             consumerProperties.setNumberOfAttempts(Integer.parseInt(rowSet.getObject(2).toString()));
          else if (propName.equalsIgnoreCase("Poller Recipient Timeout"))
             consumerProperties.setTimeout(Integer.parseInt(rowSet.getObject(2).toString()));
          else if (propName.equalsIgnoreCase("Poller Suspend On Failure"))
             consumerProperties.setSuspendOnFailure((Boolean) rowSet.getObject(2));
       }

       return processProperties;
    }

    public RedSquareRowSet getProcessStatistics(String jsondata) throws SQLException, FabricComponentAccessorException, SerializerException, IllegalStateException 
    {
       ProcessIdentifier process = deserialize(jsondata, ProcessIdentifier.class);
       String procType = process.getProcessType();

       SLResponse response = qspaceAccessor.invokeLanguageRequest("SHOW QUEUE STATS FOR "
             + escape(procType + processQueueSuffix));

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          Trace.logDebug(this, "Unable to extract statistics for '" + processQueueSuffix + "' process queue.");
          return null;
       }

       Map<String, Integer> messageStates = new HashMap<String, Integer>();
       messageStates.put(ProcessState.ACKNOWLEDGED.name(), 0);
       messageStates.put(ProcessState.DISCARDED.name(), 0);
       messageStates.put(ProcessState.ENQUEUED.name(), 0);
       messageStates.put(ProcessState.EXPIRED.name(), 0);
       messageStates.put(ProcessState.LOCKED_FOR_OFFER.name(), 0);
       messageStates.put(ProcessState.PENDING.name(), 0);
       messageStates.put(ProcessState.SKIPPED.name(), 0);
       messageStates.put(ProcessState.UNACKNOWLEDGED.name(), 0);
       messageStates.put(ProcessState.UNDELIVERED.name(), 0);
       messageStates.put(ProcessState.UNKNOWN.name(), 0);

       RowSet rowSet = response.getRowSet();
       rowSet.beforeFirst();

       while (rowSet.next())
          messageStates.put((String) rowSet.getObject(1), ((Number) rowSet.getObject(2)).intValue());

       RowSet temp = new RowSet(rowSet.getRowMetaData());
       for (Map.Entry<String, Integer> entry : messageStates.entrySet())
          temp.addRow(new Object[] { entry.getKey(), entry.getValue() });

       RedSquareRowSet result = RowSetUtils.initFromRowSet(temp);
       result.setTargetProcessType(process.getProcessType());

       return result;
    }

    public ProcessListResponse getProcessesList(String jsondata) throws IllegalStateException, SQLException, FabricComponentAccessorException, SerializerException
    {
       ProcessListRequest request = deserialize(jsondata, ProcessListRequest.class);
       String procType = request.getProcessType();
       String sortField = "SeqId";

       String order = "DESC";
       if (request.getOrder() != null)
          order = request.getOrder();

       String filter = "SeqId >= 0";
       if (request.getFilter() != null)
          filter = "(" + request.getFilter() + ")";

       String req = "select count(SeqId) AS TOTAL from " + escape(procType + processQueueSuffix) + " where " + filter;
       SLResponse response = qspaceAccessor.invokeLanguageRequest(req);
       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          Trace.logError(this, "Unable to extract process instances from the specified '" + procType + processQueueSuffix + "' process queue.");
          return null;
       }

       RowSet temp = response.getRowSet();
       if (!temp.first())
       {
          Trace.logDebug(this, "Unable to extract process instances from the specified '" + procType + processQueueSuffix + "' process queue.");
          return null;
       }

       int processNumber = ((Number) temp.getObject(1)).intValue();
       if (request.getStrategy() == ProcessesRetrievalStrategy.TOP)
       {
          response = qspaceAccessor.invokeLanguageRequest("select top "
                + request.getProcessNumber() + " ProcessId,State,Created,Acknowledged from "
                + escape(procType + processQueueSuffix)
                + " ORDER BY SeqId DESC");
       }
       else if (request.getStrategy() == ProcessesRetrievalStrategy.WINDOW)
       {
          String middleSeqIdReqyest = "(SELECT MIN(SeqId) FROM "
                + escape(procType + processQueueSuffix)
                + " WHERE State='LOCKED_FOR_OFFER' OR State='ENQUEUED')";
          response = qspaceAccessor.invokeLanguageRequest(middleSeqIdReqyest);

          int seqId = 0;
          String middleSeqId = "UNKNOWN";

          if (response != null && response.getRowSet() != null)
          {
             if (response.getRowSet().first())
                if (response.getRowSet().getObject(1) != null)
                   middleSeqId = response.getRowSet().getObject(1).toString();
          }

          if (!middleSeqId.isEmpty() && !middleSeqId.equals("UNKNOWN"))
             seqId = Integer.parseInt(middleSeqId);

          //we have no processes we need, performing usual select
          if (middleSeqId.equals("UNKNOWN")) 
          {
             response = qspaceAccessor.invokeLanguageRequest("select top "
                   + request.getProcessNumber() + " ProcessId,State,Created,Acknowledged from "
                   + escape(procType + processQueueSuffix)
                   + " ORDER BY SeqId DESC");
          }
          else 
          {
             int halfFromProcessNumber = request.getProcessNumber()/2;
             //trying to get +- halfFromProcessNumber from the queue

             //to check if we are at very start of the queue
             boolean queueVeryStart = false;
             response = qspaceAccessor.invokeLanguageRequest("SELECT COUNT(*) FROM "
                   + escape(procType + processQueueSuffix) + " WHERE SeqId < " + seqId);

             if (response != null && response.getRowSet() != null)
             {
                if (response.getRowSet().first())
                {
                   String entriesCountString = response.getRowSet().getObject(1).toString();
                   int entriesCoount = Integer.parseInt(entriesCountString);
                   if (halfFromProcessNumber > entriesCoount)
                      queueVeryStart = true;
                }
             }

             //we need to check if we are at the very start (< request.getProcessNumber())
             if (queueVeryStart)
             {
                response = qspaceAccessor.invokeLanguageRequest("select ProcessId,State,Created,Acknowledged " 
                      +  " from (select * from " + escape(procType + processQueueSuffix)
                      + " ORDER BY SeqId ASC LIMIT " + request.getProcessNumber() + ") ORDER BY SeqId DESC");
             }
             else
             {
                response = qspaceAccessor.invokeLanguageRequest("select ProcessId,State,Created,Acknowledged from "
                      + escape(procType + processQueueSuffix)
                      + " where SeqId in (select SeqId from " + escape(procType + processQueueSuffix)
                      + " where SeqId>=" + seqId + " ORDER BY SeqId ASC LIMIT " + halfFromProcessNumber + ") or SeqId in (select SeqId from "
                      + escape(procType + processQueueSuffix)
                      + "where SeqId<" + seqId + " ORDER BY SeqId DESC LIMIT " + halfFromProcessNumber + ") ORDER BY SeqId DESC");
             }

             if (response != null && response.getRowSet() != null)
             {
                response.getRowSet().beforeFirst();
                //if we have not enought rows here that means
                // a) we have not enought processes total
                // b) we are near the end of the queue
                // so we do just usual top select
                if (response.getRowSet().getRowCount() < request.getProcessNumber())
                {
                   response = qspaceAccessor.invokeLanguageRequest("select top "
                         + request.getProcessNumber() + " ProcessId,State,Created,Acknowledged from "
                         + escape(procType + processQueueSuffix)
                         + " ORDER BY SeqId DESC");
                } 
                else
                   response.getRowSet().beforeFirst();
             }
          }
       } 
       else if (request.getStrategy() == ProcessesRetrievalStrategy.PAGING)
       {
          long pagesNumber = processNumber / request.getProcessNumber();
          if (processNumber % request.getProcessNumber() > 0)
             ++pagesNumber;

          if (pagesNumber == 1)
          {
             response = qspaceAccessor.invokeLanguageRequest("select ProcessId,State,Created,Acknowledged from "
                   + escape(procType + processQueueSuffix)
                   + " where " + filter + " ORDER BY " + sortField + " " + order);

          } 
          else if (request.getPageNumber() <= pagesNumber)
          {
             response = qspaceAccessor.invokeLanguageRequest("select SeqId from "
                   + escape(procType + processQueueSuffix) + " where "
                   + filter + " ORDER BY " + sortField + " " + order);

             if (response == null || !response.isOK() || response.getRowSet() == null)
             {
                Trace.logDebug(this, "Unable to extract process instances from the specified '"
                      + procType + processQueueSuffix + "' process queue.");
                return null;
             }

             // now lets extract seq_id of the first process on the page
             RowSet rowSet = response.getRowSet();
             if (rowSet.absolute((request.getPageNumber() - 1) * request.getProcessNumber()))
             {
                int startSeqId = ((Number) rowSet.getObject(1)).intValue();
                int endSeqId = startSeqId;
                if (rowSet.absolute(request.getPageNumber() * request.getProcessNumber() - 1))
                {
                   endSeqId = ((Number) rowSet.getObject(1)).intValue();
                }
                else
                {
                   rowSet.absolute(processNumber - 1);
                   endSeqId = ((Number) rowSet.getObject(1)).intValue();
                }

                if (startSeqId > endSeqId)
                {
                   response = qspaceAccessor.invokeLanguageRequest("select top "
                         + request.getProcessNumber()
                         + " ProcessId,State,Created,Acknowledged from "
                         + escape(procType + processQueueSuffix)
                         + " where SeqId <= " + startSeqId + "AND SeqId >= " + endSeqId
                         + " AND " + filter + " ORDER BY " + sortField + " " + order);
                } 
                else
                {
                   response = qspaceAccessor.invokeLanguageRequest("select top "
                         + request.getProcessNumber()
                         + " ProcessId,State,Created,Acknowledged from "
                         + escape(procType + processQueueSuffix)
                         + " where SeqId >= " + startSeqId + "AND SeqId <= " + endSeqId
                         + " AND " + filter + " ORDER BY " + sortField + " " + order);
                }
             }
          } 
          else
             response = new SLResponse(new RowSet());
       }

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          Trace.logDebug(this, "Unable to extract process instances from the specified '" + procType + processQueueSuffix + "' process queue.");
          return null;
       }

       RedSquareRowSet result = RowSetUtils.initFromRowSet(response.getRowSet());
       result.setTargetProcessType(request.getProcessType());

       ProcessListResponse processListResponse = new ProcessListResponse();
       processListResponse.setProcesses(result);
       processListResponse.setProcessNumber(processNumber);
       processListResponse.setProcessType(request.getProcessType());

       return processListResponse;
    }

    public RedSquareRowSet getDetailedAudit(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SQLException, SerializerException
    {
       ProcessAuditRequest processStep = deserialize(jsondata, ProcessAuditRequest.class);
       SLResponse response = qspaceAccessor.invokeLanguageRequest("select * from "
             + escape(auditQueueName) + " where CorrelationId='"
             + processStep.getProcessId() + "' AND EventKey='" + processStep.getStep() + "' "
             + "AND SeqId=" + processStep.getAuditId());

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          Trace.logDebug(this, "Unable to extract detailed audit for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
          return null;
       }

       RedSquareRowSet result = new RedSquareRowSet(3, 30);
       RowSet rowSet = response.getRowSet();
       if (!rowSet.first())
       {
          Trace.logDebug(this, "Unable to extract detailed audit for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
          return null;
       }

       // timestamp
       Object timestamp = rowSet.getObject("Created");
       if (timestamp == null)
       {
          Trace.logError(this, "Unable to extract 'timestamp' for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
          return null;
       }        
       result.addRow(new Object[] { "Timestamp", "label", timestamp.toString() });

       // source
       Object source = rowSet.getObject("EventSource");
       if (source == null)
       {
          Trace.logError(this, "Unable to extract 'source' for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
          return null;
       }
       result.addRow(new Object[] { "Source Component", "label", source.toString() });

       // process group
       Object processGroup = rowSet.getObject("EventGroupId");
       if (processGroup != null)
          result.addRow(new Object[] { "Process Group", "label", processGroup.toString() });            
       else
          result.addRow(new Object[] { "Process Group", "label", "" });

       // process id
       Object processId = rowSet.getObject("CorrelationId");
       if (processId == null)
       {
          Trace.logError(this, "Unable to extract 'process id' for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
          return null;
       }
       result.addRow(new Object[] { "Process Id", "textfield", processId.toString() });

       // message
       Object message = rowSet.getObject("Message");
       if (message == null)
       {
          Trace.logError(this, "Unable to extract 'message' for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
          return null;
       }
       result.addRow(new Object[] { "Audit Message", "textarea", message.toString() });

       if (((String)source).contains("DBEventSink"))
       {
          response = qspaceAccessor.invokeLanguageRequest("select * from "
                + escape(sqlAuditQueueName) + " where CorrelationId='"
                + processStep.getProcessId() + "' AND EventKey='"
                + processStep.getStep() + "'");

          if (response == null || !response.isOK() || response.getRowSet() == null)
          {
             Trace.logDebug(this, "Unable to extract SQL audit for '" + processStep.getStep() + "' step of the '" + processStep.getProcessId() + "' process.");
             return null;
          }

          rowSet = response.getRowSet();
          if (rowSet.first())
          {
             extractUserPropertyFromAuditQueue("ComponentType", rowSet, result);
             extractUserPropertyFromAuditQueue("ComponentName", rowSet, result);
             extractUserPropertyFromAuditQueue("SQLType", rowSet, result);
             extractUserPropertyFromAuditQueue("URL", rowSet, result);
             extractUserPropertyFromAuditQueue("Database", rowSet, result);

             StringBuilder sqlBatch = new StringBuilder();
             do
             {
                Object value = rowSet.getObject("SQL");
                if (value != null)
                {
                   String textValue = value.toString();
                   sqlBatch.append(textValue).append("\n");
                }
             } 
             while (rowSet.next());

             String resultSqlBatch = sqlBatch.toString();
             if (resultSqlBatch.length() > 0)
                result.addRow(new Object[] { "SQL", "textarea", resultSqlBatch });
          }
       }

       result.compact();
       result.setTargetProcessType(processStep.getProcessType());
       result.setTargetProcessId(processStep.getProcessId());
       return result;
    }
    
    private void extractUserPropertyFromAuditQueue(String propertyName, RowSet rowSet, RedSquareRowSet result) throws IllegalStateException, SQLException
    {
       Object value = rowSet.getObject(propertyName);
       if (value == null)
          return;

       String textValue = value.toString();
       if (textValue.length() < MAX_TEXT_FIELD_WIDTH)
          result.addRow(new Object[] { propertyName, "textfield", textValue });
       else
          result.addRow(new Object[] { propertyName, "textarea", textValue });
    }

    public RedSquareRowSet getProcessesAuditInfo(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SQLException, SerializerException
    {
       ProcessAuditRequest request = deserialize(jsondata, ProcessAuditRequest.class);
       SLResponse response = null;
       if (request.getType().equals(AuditTypeEnum.Process) || request.getType() == null) 
       {
          response = qspaceAccessor.invokeLanguageRequest("select SeqId,EventKey,Message,EventSource,\"Severity\"," 
                + "EventGroupId,Created from " + escape(auditQueueName)
                + " where CorrelationId='" + request.getProcessId() + "' ORDER BY EventKey,Created");            
       }
       else if (request.getType().equals(AuditTypeEnum.SQL)) 
       {
          response = qspaceAccessor.invokeLanguageRequest("select SeqId,EventKey,Message,EventSource,\"Severity\", " 
                + "EventGroupId,Created,SQL,Database,URL from " + escape(sqlAuditQueueName)
                + " where CorrelationId='" + request.getProcessId() + "' ORDER BY EventKey,Created");            
       }
       else if (request.getType().equals(AuditTypeEnum.User)) 
       {
          response = qspaceAccessor.invokeLanguageRequest("select SeqId,EventKey,Message,EventSource,\"Severity\"," 
                + "EventGroupId,Created,userName from " + escape(userOperationAuditQueueName)  
                + " where CorrelationId='"+ request.getProcessId() + "' ORDER BY EventKey,Created");  
       }

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          Trace.logDebug(this, "Unable to extract audit information from the specified '" + auditQueueName + "' audit queue.");
          return null;
       }

       RedSquareRowSet result = RowSetUtils.initFromRowSet(response.getRowSet());
       result.setTargetProcessType(request.getProcessType());
       result.setTargetProcessId(request.getProcessId());

       return result;
    }

    public ServiceListResponse getServices(String jsondata) throws Exception
    {
       ServiceListRequest request = deserialize(jsondata, ServiceListRequest.class);
       ServiceListResponse opResponse = new ServiceListResponse();
       opResponse.setAgentName(ctx.getName());

       SLResponse response;
       if (request.getServiceGroup() != null)
       {
          response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM \"" + SERVICE_TYPE_TABLE_NAME + "\" WHERE SERVICE_GROUP='"
                + request.getServiceGroup() + "'");
          opResponse.setGroupName(request.getServiceGroup());
       } 
       else
          response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM \"" + SERVICE_TYPE_TABLE_NAME + "\"");

       RowSet temp = response.getRowSet();
       if (temp == null || !temp.first())
          return opResponse;

       try
       {
          RedSquareRowSet result = RowSetUtils.initFromRowSet(temp);
          opResponse.setServices(result);
       }
       catch (IllegalStateException e)
       {
          Trace.logException(RedSquareAgent.class, e, true);
          return null;
       }

       return opResponse;
    }

    public ProcessProvisionResponse doProvision(String userName) throws FabricComponentAccessorException, SQLException
    {
       ctx.logDebug("Provision request processing.");

       sendOperationStateEvent(new ProgressNotification(PROVISION_FIRST_OPERATION_NAME,
             "doProvisionClearTable", "Starting provisioning...", "OK", userName));
       ProcessProvisionResponse opResponse = new ProcessProvisionResponse();

       sendOperationStateEvent(new ProgressNotification("doProvisionClearTable",
             "doProvisionClearTable", "Clearing service list...", "RUNNING", userName));

       Trace.logDebug(this, "Cleaning service table");
       SLResponse response = tspaceAccessor.invokeLanguageRequest("DELETE FROM \"" + SERVICE_TYPE_TABLE_NAME + "\"");
       sendOperationStateEvent(new ProgressNotification("doProvisionClearTable", "doProvisionUpdatingTable", "Clearing service list...", "OK", userName));

       Trace.logDebug(this, "Updating service table!");
       List<FabricComponent> components = context.getBoundComponents();
       String name = "";
       for (FabricComponent comp : components)
       {
          if (comp.getComponentModel() == com.streamscape.sef.enums.ComponentModel.CTX_SERVICE)
          {
             FabricManagedComponent managedComp = (FabricManagedComponent) comp;
             String group = "DefaultGroup";
             if (managedComp instanceof ServiceContext)
             {
                ServiceContext service = (ServiceContext)managedComp;
                ConfigurationProperty serviceGroupProp = service.getServiceConfiguration().getAdvancedProperty(SERVICE_GROUP_PROPERTY);
                if (serviceGroupProp != null)
                   group = serviceGroupProp.getValue().toString();

                if (!comp.getName().equals(ctx.getName()))
                {
                   sendOperationStateEvent(new ProgressNotification("doProvisionUpdatingTable"
                         + name, "doProvisionUpdatingTable" + name, "Adding "
                               + comp.getType() + "." + comp.getName() + "...", "RUNNING", userName));

                   String insertProcess = "INSERT INTO \"" + SERVICE_TYPE_TABLE_NAME
                         + "\" VALUES('" + comp.getType() + "." + comp.getName() + "', '"
                         + group + "', '" + service.getState() + "')";

                   response = tspaceAccessor.invokeLanguageRequest(insertProcess);

                   if (response.isOK())
                   {
                      Trace.logDebug(this, "Inserted Service: Name :" + comp.getName() + ". Type: " + group + ". State: " + managedComp.getState());
                      sendOperationStateEvent(new ProgressNotification( "doProvisionUpdatingTable" + name, "doProvisionUpdatingTable"
                                  + comp.getName(), "Adding " + comp.getType() + "." + comp.getName() + "...", "OK", userName));
                      name = comp.getName();
                   } 
                   else
                   {
                      opResponse.setOk(false);
                      Trace.logDebug(this, "Can not insert service " + comp.getName());
                      sendOperationStateEvent(new ProgressNotification("doProvisionUpdatingTable" + name, "doProvisionUpdatingTable"
                                  + comp.getName(), "Adding " + comp.getType() + "." + comp.getName() + "...", "WARN", userName));
                      name = comp.getName();
                   }
                }
             }
          }
       }

       sendOperationStateEvent(new ProgressNotification("doProvisionUpdatingTable" + name, "", "Updated Configuration.", "OK", userName));

       response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM \"" + SERVICE_TYPE_TABLE_NAME + "\"");
       RowSet temp = response.getRowSet();
       if (!temp.first())
          return opResponse;

       try
       {
          RedSquareRowSet result = RowSetUtils.initFromRowSet(temp);
          opResponse.setServices(result);
       } 
       catch (IllegalStateException e)
       {
          Trace.logException(this, e, true);
          return opResponse;
       }

       opResponse.setNodeName(thisNode);
       return opResponse;
    }

    public ServiceGroupResponse getServiceGroups(String jsondata) throws FabricComponentAccessorException, SQLException, SerializerException
    {
       ServiceGroupRequest request = deserialize(jsondata, ServiceGroupRequest.class);
       ServiceGroupResponse opResponse = new ServiceGroupResponse();
       opResponse.setAgentName(ctx.getName());

       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT DISTINCT SERVICE_GROUP FROM \"" + SERVICE_TYPE_TABLE_NAME + "\"");
       RowSet temp = response.getRowSet();
       if (temp == null || !temp.first())
          return opResponse;

       try
       {
          RedSquareRowSet result = RowSetUtils.initFromRowSet(temp);
          opResponse.setServiceGroups(result);
       } 
       catch (IllegalStateException e)
       {
          Trace.logException(RedSquareAgent.class, e, true);
          return null;
       }

       return opResponse;
    }

    //NOTE: not operational
    //TODO: should be either updated or removed
    public RedSquareOperationResponse installAgent(String jsondata, String userName) throws Exception
    {

       ProcessInstallRequest request = deserialize(jsondata, ProcessInstallRequest.class);
       sendOperationStateEvent(new ProgressNotification(INSTALL_FIRST_OPERATION_NAME,
             "agentCreateReadFile", "Starting agent installation...", "OK", userName));

       Trace.logDebug(this, "Installing agent.");
       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       if (request != null)
          opResponse.setProcessType(request.getProcessType());

       SLResponse response = tspaceAccessor.invokeLanguageRequest("create dataspace " + tableSpaceName + " MODEL TSPACE");
       printSLResponseIfError("TSPACE." + tableSpaceName + " already exists or error occurred while creating.", response);

       response = tspaceAccessor.invokeLanguageRequest("create dataspace " + queueSpaceName + " MODEL QSPACE");
       printSLResponseIfError("QSPACE." + queueSpaceName + " already exists or error occurred while creating.", response);

       sendOperationStateEvent(new ProgressNotification("agentCreateReadFile", "agentCreateReadFile", "Reading installation file...", "RUNNING", userName));
       InputStream in = this.getClass().getResourceAsStream(INSTALL_FILE_NAME);
       BufferedReader br = new BufferedReader(new InputStreamReader(in));
       sendOperationStateEvent(new ProgressNotification("agentCreateReadFile", "agentCreateLineProcessing", "Reading installation file...", "OK", userName));

       String strLine;
       String name = "";
       while ((strLine = br.readLine()) != null)
       {
          Trace.logDebug(this, "Processing SQL line from file: " + strLine);

          response = tspaceAccessor.invokeLanguageRequest(strLine);
          if (response == null || !response.isOK())
          {
             if (strLine.contains("DROP"))
             {
                printSLResponseIfError("Can not drop tables. They do not exist and will be created.", response);   
                continue;
             }
             printSLResponseIfError("Unable to create mandatory tables!", response);   
             opResponse.setErrorMessage("Unable to create mandatory tables!");
             opResponse.setSuccess(false);
             sendOperationStateEvent(new ProgressNotification(
                   "agentCreateLineProcessing" + name, "agentCreateLineProcessing" + strLine,
                   strLine, "WARN", userName));
             name = strLine;
          } 
          else
          {
             Trace.logDebug(this, "Line processed OK");
             sendOperationStateEvent(new ProgressNotification(
                   "agentCreateLineProcessing" + name, "agentCreateLineProcessing" + strLine,
                   strLine, "OK", userName));
             name = strLine;
          }

       }
       in.close();

       sendOperationStateEvent(new ProgressNotification("agentCreateLineProcessing" + name, "", "Installation finished", "OK", userName));

       return opResponse;
    }

    public RedSquareOperationResponse deployProcess(String jsondata, String userName) throws FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier request = deserialize(jsondata, ProcessIdentifier.class);

       sendOperationStateEvent(new ProgressNotification(DEPLOY_FIRST_OPERATION_NAME,
             "deployProcessBeginDataBaseUpdate", "Starting flow deployment...", "OK", userName));

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();

       sendOperationStateEvent(new ProgressNotification("deployProcessBeginDataBaseUpdate",
             "deployProcessBeginDataBaseUpdate", "Updating configuration tables...", "RUNNING",
             "userName"));

       SLResponse response = tspaceAccessor.invokeLanguageRequest("UPDATE \""
             + PROCESS_CONFIGURATION_TABLE_NAME + "\" SET PROCESS_STATE='"
             + ProcessDeploymentState.DEPLOYED + "' WHERE PROCESS_NAME='"
             + request.getProcessType() + "'");

       if (response == null || !response.isOK())
       {
          Trace.logDebug(this, "Unable to UPDATE configuration in'"
                + PROCESS_CONFIGURATION_TABLE_NAME + "' table.");
          opResponse.setSuccess(false);
          sendOperationStateEvent(new ProgressNotification("deployProcessBeginDataBaseUpdate",
                "", "Updating configuration tables...", "NOK", userName));
          return opResponse;
       }

       sendOperationStateEvent(new ProgressNotification("deployProcessBeginDataBaseUpdate",
             "sendingEventToManager", "Updating configuration tables...", "OK", userName));
       opResponse.setSuccess(true);

       sendOperationStateEvent(new ProgressNotification("sendingEventToManager", "updatingManagerTable", "Sending manager event...", "OK", userName));
       processQueueStateChangeSender.send(request.getProcessType(), ProcessQueueState.RUNNING);
       sendOperationStateEvent(new ProgressNotification("updatingManagerTable", "updatingManagerTable", "Updating Manager Table...", "RUNNING", userName));

       opResponse.setAdditionalMessage(request.getProcessType() + " in now Deployed");
       return opResponse;
    }

    public RedSquareOperationResponse unDeployProcess(String jsondata, String userName) throws FabricComponentAccessorException, SerializerException
    {
       ProcessIdentifier request = deserialize(jsondata, ProcessIdentifier.class);

       sendOperationStateEvent(new ProgressNotification(UNDEPLOY_FIRST_OPERATION_NAME, "undeployProcessBeginDataBaseUpdate",
             "Starting process undeployment...", "OK", userName));

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       sendOperationStateEvent(new ProgressNotification("undeployProcessBeginDataBaseUpdate", "undeployProcessBeginDataBaseUpdate", "Updating agent configuration...", "RUNNING", ""));

       SLResponse response = tspaceAccessor.invokeLanguageRequest("UPDATE \""
             + PROCESS_CONFIGURATION_TABLE_NAME + "\" SET PROCESS_STATE='"
             + ProcessDeploymentState.UNDEPLOYED + "' WHERE PROCESS_NAME='"
             + request.getProcessType() + "'");

       if (response == null || !response.isOK())
       {
          Trace.logDebug(this, "Unable to UPDATE configuration in'"
                + PROCESS_CONFIGURATION_TABLE_NAME + "' table.");
          opResponse.setSuccess(false);
          sendOperationStateEvent(new ProgressNotification("undeployProcessBeginDataBaseUpdate",
                "", "Updating agent configuration...", "NOK", userName));
          return opResponse;
       }

       sendOperationStateEvent(new ProgressNotification("undeployProcessBeginDataBaseUpdate", "sendingEventToManager",
             "Updating agent configuration...", "OK", userName));
       opResponse.setSuccess(true);

       sendOperationStateEvent(new ProgressNotification("sendingEventToManager", "updatingManagerTable", "Sending Manager event...", "OK", userName));
       processQueueStateChangeSender.send(request.getProcessType(), ProcessQueueState.UNKNOWN);
       sendOperationStateEvent(new ProgressNotification("updatingManagerTable", "updatingManagerTable", "Updating Manager Table...", "RUNNING", userName));

       opResponse.setAdditionalMessage(request.getProcessType() + " in now Undeployed");
       return opResponse;
    }

    public RedSquareOperationResponse logsSubscriptionHandler(String jsondata)  throws FabricConnectionException, FabricEventDispatcherException, EventSelectorFormatException, IllegalConsumerStateException, SerializerException
    {
       LogsSubscribeRequest request = deserialize(jsondata, LogsSubscribeRequest.class);
       RedSquareOperationResponse response = new RedSquareOperationResponse();
       response.setProcessType(thisNode);

       if (logConsumer == null) 
       {
          TraceConfigurator.getInstance().setBroadcastAll(true);
          Trace.logDebug(this, "Request for log events subscription: creating consumer");
          logConsumer = connection.createEventConsumer(thisNode + "_LogEventConsumer", new LogEventListener(), LogEventSender.EVENT_ID,
                                                       request.getSelector(), EventScope.INHERITED, true);
          response.setAdditionalMessage("Successfully started consumer " + thisNode + "_LogEventConsumer");
       } 
       else if (!request.isStart())
       {
          Trace.logDebug(this, "Request for log events subscription: dropping consumer");
          connection.dropConsumer(logConsumer.getName());
          logConsumer = null;
          TraceConfigurator.getInstance().setBroadcastAll(false);
          response.setAdditionalMessage("Successfully dropped consumer " + thisNode + "_LogEventConsumer");
       }
       else 
       { 
          Trace.logDebug(this, "Request for log events selector change: changing selector to" + request.getSelector());
          logConsumer.setEventSelector(request.getSelector());
          response.setAdditionalMessage("Successfully set consumer selector for " + thisNode + "_LogEventConsumer to " + request.getSelector());
       }

       response.setSuccess(true);
       return response;
    }
    

    public byte[] getFullLogFile( byte[] input )
    {
       ctx.logDebug( "Management RedSquare node requests log file" );
       byte[] response = null;
       ZipArchiveOutputStream zos = null;
       FileInputStream fis = null;

       File logFile = TraceConfigurator.getInstance().getLogFile();
       String tmpName = ((Number)(new Date()).getTime()).toString();
       String zipFileName = tmpName + ".zip";
       File tmpFile = new File( tmpName );
       File zipFile = null;

       try 
       {       
          // make snapshot for log file
          ctx.logDebug( "Copying log file to local copy" );
          try 
          {
             FileUtils.copyFile( logFile, tmpFile );
          } 
          catch (IOException ex) 
          {
             // if someone prints logs to log file during copying, the copyFile
             // method can raise exception.
             // To be sure that it's real issue, need to check size of tmp file
             // If it's 0 bytes then there is happend real issue 
             // (e.g. memory issue )

             ctx.logDebug( "An exception has been raised during copying." );             
             if (!tmpFile.exists() || tmpFile.length() == 0) 
                throw new IOException( ex.getMessage() );
          }

          // make local zip file
          ctx.logDebug( "Preparing zip file" );
          zipFile = new File( zipFileName );
          zos = new ZipArchiveOutputStream( zipFile );
          ZipArchiveEntry in = new ZipArchiveEntry( tmpFile, logFile.getName() );
          zos.putArchiveEntry( in );

          byte[] b = new byte[1024];
          fis = new FileInputStream( tmpFile );
          while (fis.read(b) > 0)
             zos.write(b);

          fis.close();
          fis = null;
          zos.closeArchiveEntry();
          zos.close();

          // check size of packed zip file (limitation 5 mb)
          ctx.logDebug( "Check size limitation. File's size is " + zipFile.length() );
          if (zipFile.length() > ZIP_LOG_FILE_MAX_SIZE)
          {
             // returns empty array 
             // zip file's size can't be 0, it always contains special fields
             // at least
             response = new byte[0];
          }
          else 
          {
             // prepare byte array for response
             FileInputStream fin = new FileInputStream( zipFile );
             response = new byte[(int)zipFile.length()];
             fin.read( response );
             fin.close();
          }

          zipFile.delete();
          tmpFile.delete();
       }
       catch ( IOException ex ) 
       {
          ctx.logError( "Can not prepare log file: " + ex.getMessage() );
          if (tmpFile.exists())
             tmpFile.delete();
          if (null != zipFile && zipFile.exists())
             zipFile.delete();
       }
       return response;    
    }

    public LogListResponse getLatestLogs(String jsondata) throws IllegalStateException, IllegalArgumentException, IOException, ParseException, SerializerException
    {
       LogListRequest request = deserialize(jsondata, LogListRequest.class);
       LogListResponse response = new LogListResponse();
       response.setNode(thisNode);

       if (request.getLogLevels() == null)
       {
          List<Level> levels = new ArrayList<>();
          if (request.isDebug()) 
             levels.add(Level.DEBUG);

          if (request.isError()) 
             levels.add(Level.ERROR);

          if (request.isInfo()) 
             levels.add(Level.INFO);

          if (levels.isEmpty()) 
             request.setLogLevels(Arrays.asList(Level.values()));
          else 
             request.setLogLevels(levels);
       }

       File logFile = TraceConfigurator.getInstance().getLogFile();
       if (logFile != null)
          response.setTraceRecords(
             LogsHelper.getLogTraceRecords(logFile, -request.getLogCount(), request.getLogLevels(), request.getStartDate(), request.getStopDate()));
       else
          ctx.logDebug("No trace file configured");

       return response;
    }

    /**
     * @param jsondata
     *            if this request has service name, events for this service only
     *            are returned, otherwise all events
     * @return events depending on request.serviceName
     * @throws IllegalStateException
     * @throws SerializerException 
     */
    public RedSquareRowSet getEventsList(String jsondata) throws IllegalStateException, SerializerException
    {
       EventListRequest request = deserialize(jsondata, EventListRequest.class);
       List<String> events = new LinkedList<>();
       if (request.getServiceName() == null || request.getServiceName().length() == 0)
       {
          List<String> allEvents = context.getDatagramPrototypeCache().listEventIds();
          for (String eventID : allEvents)
          {
             if (!eventID.startsWith("advisory") && !eventID.startsWith("e.") && !eventID.contains("redsquare")) 
                events.add(eventID);
          }
       }
       else
       {
          ServiceContext service = (ServiceContext)context.lookupComponentByFQName(request.getServiceName());
          if (service != null && service.listSinkEvents() != null) 
          {
             for (String eventID : service.listSinkEvents())
                if (!eventID.startsWith("advisory") && !eventID.startsWith("e.") && !eventID.contains("redsquare")) 
                   events.add(eventID);
          }
       }
       //may be we'll need to have more info from here, so RowSet is ok
       RedSquareRowSet result = new RedSquareRowSet(1, events.size());
       for (String id : events)
          result.addRow(new Object[] { id });

       return result;
    }

    public RedSquareOperationResponse createProcessConfiguration(String jsondata, String userName) throws FabricComponentAccessorException, SerializerException
    {
       ProcessConfigurationObject request = deserialize(jsondata, ProcessConfigurationObject.class);
             sendOperationStateEvent(new ProgressNotification("createConfigurationStart",
                   "createConfigurationBeginDataBaseUpdate",
                   "Starting process configuration creation...", "OK", userName));

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       opResponse.setProcessType(request.getProcessType());
       opResponse.setSuccess(true);

       sendOperationStateEvent(new ProgressNotification("createConfigurationBeginDataBaseUpdate",
             "createConfigurationBeginDataBaseUpdate", "Updating Configuration...", "RUNNING", userName));

       SLResponse response = tspaceAccessor.invokeLanguageRequest("INSERT INTO \""
             + PROCESS_CONFIGURATION_TABLE_NAME
             + "\" (PROCESS_NAME, PROCESS_DESCRIPTION, FLOW_START, CONSTRAINT_EVENT, FLOW_TYPE, ORGANIZATION, CERTIFIED_DELIVERY, SERVICE_GROUP, PROCESS_STATE, STAGED, SUSPEND_ON_FAILURE) VALUES ('"
             + request.getProcessType() + "','" + request.getProcessDescription()
             + "','" + request.getProcessFlowStart() + "','"
             + request.getConstraintEvent() + "','" + request.getProcessFlowType()
             + "','" + request.getOrganization() + "','"
             + request.getCertifiedDelivery() + "','" + request.getServiceGroup()
             + "','" + ProcessDeploymentState.UNDEPLOYED + "','" + request.getStaged()
             + "','" + request.getSuspendOnFailure() + "')");

       if (response == null || !response.isOK())
       {
          Trace.logError(this, "Unable Create configuration in'" + PROCESS_CONFIGURATION_TABLE_NAME + "' table.");
          sendOperationStateEvent(new ProgressNotification( "createConfigurationBeginDataBaseUpdate", "", "Updating Configuration...", "NOK", userName));
          return null;
       }

       sendOperationStateEvent(new ProgressNotification("createConfigurationBeginDataBaseUpdate", "sendingEventToManager", "Updating Configuration...", "OK", userName));
       sendOperationStateEvent(new ProgressNotification("sendingEventToManager", "updatingManagerTable", "Sending Manager event...", "OK", userName));
       processQueueStateChangeSender.send(request.getProcessType(), ProcessQueueState.UNKNOWN, request.getOrganization());
       sendOperationStateEvent(new ProgressNotification("updatingManagerTable", "updatingManagerTable", "Updating Processes...", "RUNNING", userName));
       return opResponse;
    }

    public RedSquareOperationResponse updateProcessConfigurationObject(String jsondata) throws FabricComponentAccessorException, SerializerException
    {
       ProcessConfigurationObject request = deserialize(jsondata, ProcessConfigurationObject.class);
       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       opResponse.setProcessType(request.getProcessType());
       opResponse.setSuccess(true);

       SLResponse response = tspaceAccessor.invokeLanguageRequest("UPDATE \""
             + PROCESS_CONFIGURATION_TABLE_NAME + "\" SET PROCESS_DESCRIPTION='"
             + request.getProcessDescription() + "', FLOW_START='"
             + request.getProcessFlowStart() + "', CONSTRAINT_EVENT='"
             + request.getConstraintEvent() + "', FLOW_TYPE='" + request.getProcessFlowType()
             + "', ORGANIZATION='" + request.getOrganization() + "', CERTIFIED_DELIVERY='"
             + request.getCertifiedDelivery() + "', STAGED='" + request.getStaged() + "', "
             + "SUSPEND_ON_FAILURE='" + request.getSuspendOnFailure() + "', "
             + "SERVICE_GROUP='" + request.getServiceGroup() + "' WHERE PROCESS_NAME='"
             + request.getProcessType() + "'");

       if (printSLResponseIfError("Unable UPDATE configuration in '" + PROCESS_CONFIGURATION_TABLE_NAME + "' table.", response))
       {
          opResponse.setErrorMessage("Unable UPDATE configuration in '" + PROCESS_CONFIGURATION_TABLE_NAME + "' table.");
          opResponse.setSuccess(false);
          return opResponse;
       }

       // update suspend on failure
       {
          ProcessQueue processQueue;
          String processQueueName = request.getProcessType() + processQueueSuffix;
          try
          {
             processQueue = getProcessQueue(processQueueName);
          }
          catch(Exception exception)
          {
             return buildErrorResponse("Failed to get process queue.", exception, request.getProcessType());
          }

          try
          {
             processQueue.setSuspendOnFailure(request.getSuspendOnFailure());
          }
          catch (Exception exception)
          {
             return buildErrorResponse("Failed to update process queue suspend on failure.", exception, request.getProcessType());
          }           
       }

       processQueueStateChangeSender.send(request.getProcessType(), ProcessQueueState.UNKNOWN, request.getOrganization());

       return opResponse;
    }

    public ProcessConfigurationObject getProcessConfigurationObject(String jsondata) throws FabricComponentAccessorException, SQLException, SerializerException
    {
       ProcessIdentifier request = deserialize(jsondata, ProcessIdentifier.class);
       ProcessConfigurationObject confObject = new ProcessConfigurationObject();

       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM \""
             + PROCESS_CONFIGURATION_TABLE_NAME + "\" WHERE PROCESS_NAME='" + request.getProcessType() + "'");

       if (printSLResponseIfError("Unable to extract configuration from '"+ PROCESS_CONFIGURATION_TABLE_NAME + "' table.", response))
          return null;

       RowSet rowSet = response.getRowSet();
       if (!rowSet.first())
       {
          Trace.logError(this, "Unable to retrieve info for '" + request.getProcessType() + "', no such process");
          return null;
       }

       //setting base params from the table
       confObject.setProcessType(request.getProcessType());
       confObject.setProcessDescription(rowSet.getObject("PROCESS_DESCRIPTION").toString());
       confObject.setConstraintEvent(rowSet.getObject("CONSTRAINT_EVENT").toString());
       confObject.setProcessFlowStart(rowSet.getObject("FLOW_START").toString());
       confObject.setProcessFlowType(rowSet.getObject("FLOW_TYPE").toString());
       confObject.setOrganization(rowSet.getObject("ORGANIZATION").toString());
       confObject.setServiceGroup(rowSet.getObject("SERVICE_GROUP").toString());
       confObject.setCertifiedDelivery(Boolean.valueOf(rowSet.getObject("CERTIFIED_DELIVERY").toString()));
       confObject.setStaged(Boolean.valueOf(rowSet.getObject("STAGED").toString()));
       confObject.setSuspendOnFailure(Boolean.valueOf(rowSet.getObject("SUSPEND_ON_FAILURE").toString()));
       confObject.setAuditQueueName("");
       confObject.setSQLAuditQueueName("");
       confObject.setProcessQueueName("");
       confObject.setConsumersSize(1);
       confObject.setConsumersTimeout(3000L);
       confObject.setConsumersOfferInterval(0L);
       confObject.setConsumersNumerOfAttempts(3);

       try
       {
          for (String queueName : qspaceAccessor.listCollections())
          {
             if (queueName.equals(request.getProcessType() + ".Audit"))
             {
                confObject.setAuditQueueName(request.getProcessType() + ".Audit");
             } 
             else if (queueName.equals(request.getProcessType() + ".Process"))
             {
                confObject.setProcessQueueName(request.getProcessType() + ".Process");
                extractProcessQueueRelatedInfo(request, confObject);
             }
             else if (queueName.equals(request.getProcessType() + ".SQLAudit"))
             {
                confObject.setSQLAuditQueueName(request.getProcessType() + ".SQLAudit");
             }
          }
       } 
       catch (Exception error)
       {
          Trace.logException(RedSquareAgent.class, error, true);
       }

       return confObject;
    }

    private void extractProcessQueueRelatedInfo(ProcessIdentifier request, ProcessConfigurationObject confObject) throws FabricComponentAccessorException, SQLException
    {
       ProcessQueue processQueue;
       try
       {
          processQueue = getProcessQueue(getProcessQueueName(request));
       }
       catch(Exception exception)
       {
          buildErrorResponse("Failed to get process queue.", exception, request.getProcessType());
          return;
       }

       try
       {
          confObject.setProcessQueueSize(processQueue.size());
          confObject.setProcessQueueStatus(ProcessQueueState.valueOf(processQueue.getQueueState().name()));

          confObject.setConsumersOfferInterval(processQueue.getOfferInterval());
          confObject.setConsumersNumerOfAttempts(processQueue.getMaxAttempts());
          confObject.setConsumersTimeout(processQueue.getRecipientTimeout());
          confObject.setSuspendOnFailure(processQueue.getSuspendOnFailure());
          confObject.setConsumersSize(processQueue.getParallelDegree());

          List<RecipientProperties> recipients = new ArrayList<>();
          for (RecipientInfo recipientInfo : processQueue.getRecipients())
          {
             RecipientProperties recipientProperties = new RecipientProperties();
             recipientProperties.setName(recipientInfo.getRecipientName());
             recipientProperties.setEventId(recipientInfo.getEventId());
             recipientProperties.setRoutingRules(recipientInfo.getSubscriptionRule());
             recipientProperties.setCertified(recipientInfo.isCertified());
             recipients.add(recipientProperties);
          }
          confObject.setRecipients(recipients);
       }
       catch (Exception exception)
       {
          Trace.logError(this, "Unable to get properties of '"+ getProcessQueueName(request) + "' process queue.");
          Trace.logException(this, exception, true);
          return;
       }
    }

    
    public RedSquareOperationResponse doNotificationConfigAction(String jsondata) throws Exception
    {

       NotificationConfigEntryRequest request = deserialize(jsondata, NotificationConfigEntryRequest.class);
       Trace.logDebug(this, "Notification config action request processing. Action: " + request.getAction());

       RedSquareOperationResponse opResponse = new RedSquareOperationResponse();
       SLResponse response = null;
       String postAction = null;
       try
       {
          switch (request.getAction()) 
          {
             case Delete:
             {
                postAction = "removed";
                removeNotificationListenerEntry(request);
                break;
             }
             case Create:
             {                
                postAction = "created";
                createNotificationListenerEntry(request);
                startNotificationConsumer(request);
                break;
             }
             case Edit:
             {
                postAction = "edited";
                updateNotificationListenerEntry(request);
                break;
             }
             case AllOff:
             {
                postAction = "turned off";
                response = tspaceAccessor.invokeLanguageRequest("UPDATE " + escape(NOTIFICATIONS_TABLE_NAME) + " SET ENABLED='false'");
                break;
             }
             case AllOn:
             {
                postAction = "turned on";
                response = tspaceAccessor.invokeLanguageRequest("UPDATE " + escape(NOTIFICATIONS_TABLE_NAME) + " SET ENABLED='true'");
                break;
             }
          }
       }
       catch(Exception exception)
       {
          response = new SLResponse(exception.getMessage(), false);
       }
       
       if (response != null && !response.isOK())
       {
          ctx.logError( "Unable to " + request.getAction() +  " notification config entry. Reason: " + response.getText() );
          opResponse.setSuccess(false);
          opResponse.setErrorMessage( "Unable to " + request.getAction() + " entry! Reason: " + response.getText() );
          return opResponse;
       }

       opResponse.setSuccess(true);
       opResponse.setAdditionalMessage("Notification config entry: " + request.getEventId() + " was " + postAction + " successfully.");

       return opResponse;

    }
    
    public RedSquareRowSet getNotificationConfigList(String jsondata) throws IllegalStateException, FabricComponentAccessorException, SQLException, SerializerException
    {
       Trace.logDebug(this, "Notification configs request processing.");

       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM "
             + escape(NOTIFICATIONS_TABLE_NAME) + " ORDER BY EVENT_ID");

       if (response == null || !response.isOK() || response.getRowSet() == null)
       {
          printSLResponseIfError("Unable to extract notifications config definitions from '" + NOTIFICATIONS_TABLE_NAME + "' table.", response);
          return null;
       }

       return RowSetUtils.initFromRowSet(response.getRowSet());
    }

    private ListUserFunctionsResponse listUserFunctions(String jsondata) throws Exception
    {
       Trace.logDebug(this, "List user functions request processing.");
       ListUserFunctionsRequest request = deserialize(jsondata, ListUserFunctionsRequest.class);

       ListUserFunctionsResponse functions = new ListUserFunctionsResponse();

       if (request.isIncludeRedSquareDataspaces())
       {
          listUserFunctions(tspaceAccessor, functions);
          listUserFunctions(qspaceAccessor, functions);
       }

       List<String> dataspaces = new ArrayList<>();
       if (request.isIncludeAllDataspaces())
       {
          ListDataspacesRequest listDataspaceRequest = new ListDataspacesRequest();
          listDataspaceRequest.setEventScopes(request.getIncludeAllDataspacesWithEventScope());
          dataspaces = listDataspaces(listDataspaceRequest, context).getDataspaces().stream().map(d -> d.getDataspace()).collect(Collectors.toList());
       }
       else if (request.getDataspaces() != null)
       {
          dataspaces = request.getDataspaces();
       }

       for (String dataspace : dataspaces)
       {
          if (dataspace.equalsIgnoreCase(tspaceAccessor.getComponentType() + "." + tspaceAccessor.getComponentName()))
          {
             if (!request.isIncludeRedSquareDataspaces())
                listUserFunctions(tspaceAccessor, functions);
          }
          else if (dataspace.equalsIgnoreCase(qspaceAccessor.getComponentType() + "." + qspaceAccessor.getComponentName()))
          {
             if (!request.isIncludeRedSquareDataspaces())
                listUserFunctions(qspaceAccessor, functions);
          }
          else
          {
             Object[] names = splitAndValidateDataspaceName(dataspace);
             openAccessorAndExecuteOperation(names, (accessor) -> {
                listUserFunctions(accessor, functions);
                return null;
             });
          }
       }

       return functions;
    }

    private void listUserFunctions(DataspaceAccessor accessor, ListUserFunctionsResponse functions) throws Exception
    {
       SLResponse response = accessor.invokeLanguageRequest("list functions");
       if (printSLResponseIfError("Unable to get list functions in dataspace " + accessor.getName() + ".", response))
          return;

       RowSet rowSet = response.getRowSet();
       while(rowSet.next())
       {
          ListUserFunctionsResponse.UserFunction function =
             new ListUserFunctionsResponse.UserFunction(context.getName(), accessor.getComponentType() + "." + accessor.getComponentName(), rowSet.getString(1));

          response = accessor.invokeLanguageRequest("describe function " + function.getFunctionName() + " parameters");
          if (printSLResponseIfError("Unable to get function parameters, " + function.getFunctionName() + " in dataspace " + function.getDataspace() + ".", response))
             continue;

          while(response.getRowSet().next())
          {
             String mode = response.getRowSet().getString(3);
             String type = response.getRowSet().getString(2);
             if ("OUT".equalsIgnoreCase(mode))
                function.setReturnType(type);
             else if ("IN".equalsIgnoreCase(mode))
                function.addArgument(response.getRowSet().getString(1), type);
          }

          functions.addFunction(function);
       }
    }

    private Object callUserFunction(String jsondata) throws Exception
    {
       CallUserFunctionRequest function = deserialize(jsondata, CallUserFunctionRequest.class);
       Trace.logDebug(this, "Calling user function " + function.getFunctionName() + " in dataspace " + function.getDataspace() + ".");

       Object[] names = splitAndValidateDataspaceName(function.getDataspace());
       if (names[0] == DataspaceType.TSPACE && names[1].equals(tableSpaceName))
          return callUserFunction(tspaceAccessor, function);
       else if (names[0] == DataspaceType.QSPACE && names[1].equals(queueSpaceName))
          return callUserFunction(qspaceAccessor, function);
       else
          return openAccessorAndExecuteOperation(names, (accessor) -> callUserFunction(accessor, function));
    }

    private Object callUserFunction(DataspaceAccessor accessor, CallUserFunctionRequest function) throws DataspaceComponentException
    {
       String query = "call " + function.getFunctionName() + "(" +
                      function.getArguments().stream().map(a -> "?").collect(Collectors.joining(",")) + ")";

       return accessor.callFunction(query, function.getArguments().toArray(new Object[0]));
    }

    private Object invokeQuery(String jsondata) throws Exception
    {
       InvokeQueryRequest request = deserialize(jsondata, InvokeQueryRequest.class);
       Trace.logDebug(this, "Invoking query " + request.getQuery() + " in dataspace " + request.getDataspace() + ".");

       Object[] names = splitAndValidateDataspaceName(request.getDataspace());
       if (names[0] == DataspaceType.TSPACE && names[1].equals(tableSpaceName))
          return invokeQuery(tspaceAccessor, request);
       else if (names[0] == DataspaceType.QSPACE && names[1].equals(queueSpaceName))
          return invokeQuery(qspaceAccessor, request);
       else
          return openAccessorAndExecuteOperation(names, (accessor) -> invokeQuery(accessor, request));
    }

    private Object invokeQuery(DataspaceAccessor accessor, InvokeQueryRequest request) throws DataspaceComponentException
    {
       return accessor.callFunction(request.getQuery(), request.getArguments().toArray(new Object[0]));
    }

    private Object listDataspaces(String jsondata) throws Exception
    {
       ListDataspacesRequest request = deserialize(jsondata, ListDataspacesRequest.class);
       Trace.logDebug(this, "Getting list dataspaces of event scope {}.", request.getEventScopes());

       return listDataspaces(request, context);
    }

    private Object[] splitAndValidateDataspaceName(String dataspace) throws Exception
    {
       List<String> tokens = StringUtils.split(dataspace, '.');
       if (tokens.size() != 2)
          throw new Exception("Invalid dataspace name '" + dataspace + "'. Should be in format <DataspaceType>.<DataspaceName>.");

       DataspaceType type = null;
       try
       {
          type = DataspaceType.valueOf(tokens.get(0).toUpperCase());
       }
       catch(Exception exception)
       {
          throw new Exception("Invalid dataspace type '" + tokens.get(0) + "' specified.");
       }

       return new Object[] {type, tokens.get(1)};
    }

    public static ListDataspacesResponse listDataspaces(ListDataspacesRequest request, RuntimeContext context)
    {
       request.setEventScopes(request.getEventScopes().stream().map(e -> e.toUpperCase()).collect(Collectors.toList()));
       ListDataspacesResponse response = new ListDataspacesResponse();
       response.setDataspaces(
          context.getDataspaceManager().getDataspaces()
             .stream()
             .filter((c) -> request.getEventScopes() == null || request.getEventScopes().size() == 0 || request.getEventScopes().contains(c.getEventScope().toString()))
             .filter((c) -> !SqlInvariants.isSystemSchemaName(c.getName()) && !SqlInvariants.isLobsSchemaName(c.getName()))
             .map((c) -> new ListDataspacesResponse.DataspaceResponse(context.getName(), c.getEventScope().toString(), c.getDataspaceType().toString() + "." + c.getName()))
             .collect(Collectors.toList())
       );

       return response;
    }

    public interface FunctionWithException<T>
    {
       Object execute(T accessor) throws Exception;
    }

    private Object openAccessorAndExecuteOperation(Object names[], FunctionWithException<DataspaceAccessor> function) throws Exception
    {
       DataspaceAccessor accessor = null;
       try
       {
          accessor = connection.createDataspaceAccessor(connection.getModerator().getFabricNode().getName(), (DataspaceType)names[0], (String)names[1]);
          if (accessor == null || !accessor.isAvailable())
             throw new Exception("Dataspace '" + names[0] + "." + names[1] + "' doesn't exist or not available.");

          return function.execute(accessor);
       }
       finally
       {
          if (accessor != null)
          {
             try
             {
                accessor.close();
             }
             catch(Exception exception)
             {
             }
          }
       }
    }

    //    public RedSquareRowSet getProcessState(ProcessStateRequest process)
//            throws IllegalStateException, FabricComponentAccessorException, IllegalArgumentException, SQLException
//    {
//        RedSquareRowSet result = new RedSquareRowSet(2, 1);
//        result.addRow(new Object[] { process.getProcessType(), getProcessState(qspaceAccessor, process.getProcessType()) });
//        result.setTargetProcessType(process.getProcessType());
//        return result;
//    }

    ////////////////////////////////////////////////////////////
    
    protected void initializeSystemTablesAndAccessors() throws Exception
    {
        ctx.logInfo("Initialization of system tables and accessors...");

        DataspaceManager dsManager = context.getDataspaceManager();
        if (dsManager.lookup(tableSpaceName) == null)
        {
            ctx.logInfo("Table space '" + tableSpaceName + "' does not exist. Recreation...");
            dsManager.createDataspace(DataspaceType.TSPACE, tableSpaceName, EventScope.OBSERVABLE);
            ctx.logInfo("Table space '" + tableSpaceName + "' created.");
        }

        if (dsManager.lookup(queueSpaceName) == null)   
        {
            ctx.logInfo("Queue space '" + queueSpaceName + "' does not exist. Recreation...");
            dsManager.createDataspace(DataspaceType.QSPACE, queueSpaceName, EventScope.OBSERVABLE);
            ctx.logInfo("Queue space '" + queueSpaceName + "' created.");
        }

        qspaceAccessor = connection.createDataspaceAccessor(RuntimeContext.getInstance().getName(), DataspaceType.QSPACE, queueSpaceName);
        tspaceAccessor = connection.createDataspaceAccessor(RuntimeContext.getInstance().getName(), DataspaceType.TSPACE, tableSpaceName);
        tspaceAccessorForNotifications = connection.createDataspaceAccessor(RuntimeContext.getInstance().getName(), DataspaceType.TSPACE, tableSpaceName);
        
        InputStream in = this.getClass().getResourceAsStream(INSTALL_FILE_NAME);
        BufferedReader br = new BufferedReader(new InputStreamReader(in));

        String strLine;

        while ((strLine = br.readLine()) != null)
        {
            Trace.logDebug(this, "Processing SQL line from file: " + strLine);
            if (strLine.startsWith("DROP"))
            {
                Trace.logDebug(this, "No need in DROP when initializing Agent.");
                continue;
            }
            printSLResponseIfError("Failed to execute SQL request '" + strLine + "'.", tspaceAccessor.invokeLanguageRequest(strLine));
        }
        in.close();

        checkSystemTable(PROCESS_CONFIGURATION_TABLE_NAME);
        checkSystemTable(SERVICE_TYPE_TABLE_NAME);
        checkSystemTable(NOTIFICATIONS_TABLE_NAME);
        
        initNotifications();
        if (qspaceAccessor.lookupCollection(auditQueueName) == null)
        {
            String req = "CREATE PERSISTENT AUDIT QUEUE "
                    + escape(auditQueueName) + " CONSTRAINED BY "
                    + escape(AuditEvent.PROCESS_EVENT_ID) + " CONSUMER";
            Trace.logDebug(this, req);
            printSLResponseIfError("Failed to create audit queue '" + auditQueueName + "'.", qspaceAccessor.invokeLanguageRequest(req));
        }
        
        if (qspaceAccessor.lookupCollection(userOperationAuditQueueName) == null)
        {
            String req = "CREATE PERSISTENT AUDIT QUEUE "
                    + escape(userOperationAuditQueueName) + " CONSTRAINED BY "
                    + escape(AuditEvent.USER_EVENT_ID) + " CONSUMER";
            
            Trace.logDebug(this, req);
            printSLResponseIfError("Failed to create persistent queue '" + userOperationAuditQueueName + "'.", qspaceAccessor.invokeLanguageRequest(req));
            
//            String trigReq = "create event trigger UserAuditEventPublisher type EventPublisher after event " + 
//               AGENT_USER_AUDIT_QUEUE_EVENT_ID + " raise event";
//            
//            Trace.logDebug(this, req);
//            printSLResponseIfError("Failed to create event trigger 'UserAuditEventPublisher'.", qspaceAccessor.invokeLanguageRequest(trigReq));
//            
//            String trigEnableReq = "enable event trigger UserAuditEventPublisher";
//            printSLResponseIfError("Failed to enable event trigger 'UserAuditEventPublisher'.", qspaceAccessor.invokeLanguageRequest(trigEnableReq));
        }

        if (qspaceAccessor.lookupCollection(sqlAuditQueueName) == null)
        {
            String req = "CREATE PERSISTENT AUDIT QUEUE "
                    + escape(sqlAuditQueueName) + " CONSTRAINED BY "
                    + escape(AuditEvent.SQL_EVENT_ID) + " CONSUMER";
            Trace.logDebug(this, req);
            printSLResponseIfError("Failed to create audit queue '" + sqlAuditQueueName + "'.", qspaceAccessor.invokeLanguageRequest(req));
        }

        checkSystemQueue(auditQueueName);
        checkSystemQueue(sqlAuditQueueName);
        ctx.logInfo("System tables initialized and accessors opened.");
        
    }
    
    private RedSquareOperationResponse buildErrorResponse(String errorMessage, Exception exception, String processType)
    {
       Trace.logError(this, errorMessage + ". Queue '" + processType + "'.");
       Trace.logException(this, exception, false);

       RedSquareOperationResponse response = new RedSquareOperationResponse();
       response.setSuccess(false);
       response.setErrorMessage(errorMessage);
       response.setAdditionalMessage(exception.getMessage());
       response.setProcessType(processType);

       return response;
    }

    private ProcessQueue getProcessQueue(String queueName) throws Exception
    {
       ProcessQueue processQueue = (ProcessQueue)qspaceAccessor.lookupCollection(queueName);
       if (processQueue == null)
          throw new Exception("Returned process queue is null.");
       return processQueue;
    }

    private AuditQueue getAuditQueue(String queueName) throws Exception
    {
       AuditQueue auditQueue = (AuditQueue)qspaceAccessor.lookupCollection(queueName);
       if (auditQueue == null)
          throw new Exception("Returned audit queue is null.");
       return auditQueue;
    }

    private String getProcessQueueName(ProcessIdentifier process)
    {
       return process.getProcessType()  + processQueueSuffix;
    }

    private RedSquareOperationResponse processFailedOperation(SLResponse response, String processId)
    {
       return processFailedOperation(response.getText(), processId);
    }

    private RedSquareOperationResponse processFailedOperation(String errorMessage, String processId)
    {
       RedSquareOperationResponse operationResponse = new RedSquareOperationResponse(processId);
       operationResponse.setSuccess(false);
       if (errorMessage != null)
          operationResponse.setErrorMessage(errorMessage);

       return operationResponse;
    }

    private boolean printSLResponseIfError(String message, SLResponse response)
    {
       if (response != null && !response.isOK())
       {
          Trace.logError(this, message + " Cause: " + getSLResponseError(response));
          return true;
       }
       return false;
    }

    private String getSLResponseError(SLResponse response)
    {
       if (response != null && !response.isOK())
       {
          String cause = null;
          if (response.getException() != null)
             return response.getException().toString();
          else if (response.getText() != null)
             return response.getText();

       }
       return null;
    }

    protected void checkSystemTable(String name) throws Exception
    {
       if (tspaceAccessor.lookupCollection(name) == null)
       {
          ctx.logError("Table '" + name + "' does not exist. Please check initialization.");
          throw new RuntimeException("System table '" + name + "' does not exist.");
       }
    }


    protected void checkSystemQueue(String queueName) throws Exception
    {
       if (qspaceAccessor.lookupCollection(queueName) == null)
       {
          ctx.logError("Queue '" + queueName + "' does not exist. Please check initialization.");
          throw new RuntimeException("System queue '" + queueName + "' does not exist.");
       }
    }

    public void processProcessInstanceChange(ProcessStateChange event)
    {
       needToNotifyAboutProcessInstanceChanges = true;
    }

    private String getUserName(ImmutableEventDatagram event)
    {
       FabricAddress address = new FabricAddress(event.getEventSource());
       ComponentReference component = ctx.getModerator().lookupComponent(address);

       String sourceName;
       String userName = null;
       if (component != null)
       {
          sourceName = component.getName();
          userName = sourceName.split("\\.")[1].split("@")[0];
       }
       return userName;
    }

    private String escape(String toEscape)
    {
       return "\"" + toEscape + "\"";
    }

    private void sendOperationStateEvent(ProgressNotification notification)
    {
       try
       {
          DataEvent event = (DataEvent) datagramFactory
                .createEvent(RED_SQUARE_AGENT_OPERATION_PROGRESS_EVENT_ID);
          event.setData(notification);
          ctx.raiseEvent(event, 0);
       } 
       catch (Exception e)
       {
          Trace.logDebug(this, "Unable to send operation status event!");
          Trace.logException(this, e, true);
       }
    }


    private Object getDataFromImmutable(ImmutableEventDatagram event) throws Exception
    {
       if (event instanceof DataEvent)
          return ((DataEvent)event).getData();
       else
          throw new Exception("Unsupported event type.");
    }

    private static void addProcessStateChangePrototype(String eventId) throws Exception
    {
       DataEvent event = (DataEvent) EventDatagramFactory.getInstance().newEventInstance("DataEvent");
       event.setData(new ProcessStateChange());
       event.addAnnotation("State", "//data/state");
       event.addAnnotation("ProcessId", "//data/processId");
       event.addAnnotation("ProcessGroup", "//data/processGroupId");
       event.addAnnotation("Step", "//data/processStep");
       event.addAnnotation("Attempt", "//data/attempt");
       SDOUtils.addDataEventPrototype(eventId, event);
    }

    private void sendUserAuditAndNotification(String action, ProcessIdentifier process) throws Exception 
    {        
       //sending user audit
       AuditEvent event = (AuditEvent)EventDatagramFactory.getInstance().createEvent(AuditEvent.USER_EVENT_ID);
       event.setAuditData("User " + process.getUserName() + " " + action + " " + process.getProcessId() + ".");
       event.setContentType(ContentType.TEXT);
       if (process.getUserName() != null)
          event.setEventStringProperty("userName", process.getUserName());
       if (process.getComment() != null)
          event.setEventStringProperty("userComment", process.getComment());
       event.setSeverity(Severity.WARNING);
       event.setCorrelationId(process.getProcessId());
       event.setEventGroupId(process.getProcessType());
       ctx.raiseEvent(event, 0);

       //sending notification
       PortalNotification notification = new PortalNotification();
       notification.setSource(thisNode);
       notification.setSeverity(Severity.GENERIC);
       notification.setType(NotificationType.UserAction);
       notification.setDescription("User " + process.getUserName() + " " + action + " " + process.getProcessId() + ".");
       notification.setDetails(process.getComment() != null ? process.getComment() : "");
       notification.setGroup(thisNode);
       DataEvent ev = SDOUtils.createDataEvent(NOTIFICATION_EVENT_ID, notification);
       ev.setCorrelationId(process.getProcessId());
       ctx.raiseEvent(ev, 0);
    }
    
    ////////////////////////////////////
    
    class LogEventListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          try
          {
             TraceRecord traceRecord = LogsHelper.createTraceRecordFromTraceLogMapEvent((MapEvent) event);
             if (traceRecord != null)
             {
                // we do not need logs about logs for sure
                if (traceRecord.getMessage() != null &&
                    (traceRecord.getMessage().contains(RED_SQUARE_LOG_LINE_EVENT_ID) ||
                     traceRecord.getMessage().contains("event.log") ||
                     traceRecord.getMessage().contains(LogEventListener.class.getName())))
                   return;

                LogLineEvent logLineEvent = new LogLineEvent();
                logLineEvent.setNodeName(thisNode);
                logLineEvent.setTraceRecord(traceRecord);

                DataEvent dataEvent = (DataEvent)datagramFactory.createEvent(RED_SQUARE_LOG_LINE_EVENT_ID);
                dataEvent.setData(logLineEvent);
                ctx.raiseEvent(dataEvent, 0);
             }
          }
          catch (Exception e)
          {
             Trace.logError(this, "Unable to send log event!");
             Trace.logException(this, e, true);
          }
       }
    }

    ///////////////////////////////////////////

    private void initNotifications() throws Exception
    {
       if (checkNotificationsConfiguration())
          return;
       
       ctx.logInfo("Notification config table is empty. Filling in with new values.");
       ctx.logInfo("Filling in advisories definitions.");

       createNotificationListenerEntry(new NotificationConfigEntryRequest(ConnectionStateChangeAdvisory.EVENT_ID, "ADVISORY", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(ComponentStateChangeAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(EventTriggerAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(HTTPClientAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(HTTPRestClientAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(RepositoryArtifactStateChangeAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(RepositoryStateChangeAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(RuntimeAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(XMPPClientAdvisory.EVENT_ID, "ADVISORY", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(XMPPConnectionStateChangeAdvisory.EVENT_ID, "ADVISORY", "", false));
       //state advisories
       createNotificationListenerEntry(new NotificationConfigEntryRequest(DataspaceStateAdvisory.EVENT_ID, "ADVISORY", "", true));

       ctx.logInfo("Filling in exception definitions.");

       createNotificationListenerEntry(new NotificationConfigEntryRequest(AcceptorException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(AcceptorFactoryException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(AcceptorFactoryManagerException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(ClientException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(DatabaseSQLException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(DatagramFactoryException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(DatagramFactoryManagerException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(DataspaceComponentException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(DataspaceManagerException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(EventIdentityPluginManagerException.EVENT_ID, "EXCEPTION", "", true));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricComponentAccessorException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricConnectionException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricConnectionFactoryException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricEventDispatcherException.EVENT_ID, "EXCEPTION", "", true));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricEventException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricEventSinkException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricEventSourceException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricEventStreamException.EVENT_ID, "EXCEPTION", "", false));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricEventTriggerException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricRequestException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FabricUnboundEventException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(FactoryManagerException.EVENT_ID, "EXCEPTION", "", false));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(HTTPClientException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(JSONSerializerException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(JSerializerException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(ObjectMediationException.EVENT_ID, "EXCEPTION", "", false));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(ProcessQueueException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(RepositoryAccessorException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(RepositoryException.EVENT_ID, "EXCEPTION", "", true));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(SLSessionException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SecurityManagerException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SemanticLexiconException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SemanticMapperException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SemanticTypeFactoryException.EVENT_ID, "EXCEPTION", "", false));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(ServiceManagerException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SoapRequestException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(XSerializerException.EVENT_ID, "EXCEPTION", "", false));

       createNotificationListenerEntry(new NotificationConfigEntryRequest(SecurityViolationException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SerialVersionMismatchException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(ServiceContextException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(ServiceFrameworkException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SoapFaultException.EVENT_ID, "EXCEPTION", "", false));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SQLQueryException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SQLQueryParseException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(SQLQueryValidationException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(TransportException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(XMLParseException.EVENT_ID, "EXCEPTION", "", true));
       createNotificationListenerEntry(new NotificationConfigEntryRequest(XMPPClientException.EVENT_ID, "EXCEPTION", "", true));
    }

    
    private boolean checkNotificationsConfiguration() throws FabricComponentAccessorException, SQLException 
    {
       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM " + escape(NOTIFICATIONS_TABLE_NAME) + "");
       RowSet result = response.getRowSet();
       return !(result == null || result.isEmpty() || !result.first());
    }

    private void startGroupNotificationsConsumer() throws Exception 
    {
       SLResponse response = tspaceAccessor.invokeLanguageRequest("SELECT * FROM " + escape(NOTIFICATIONS_TABLE_NAME) + " ORDER BY EVENT_ID");

       if (this.printSLResponseIfError("Unable to extract notifications config definitions from '" + NOTIFICATIONS_TABLE_NAME + "' table.", response))
          throw new FabricException("Unable to get consumer data from table!");

       stopGroupNotificationsConsumer();
       groupNotificationsListener = new GroupNotificationsListener();
       
       StringBuilder eventFilter = new StringBuilder();
       RowSet rowSet = response.getRowSet();
       rowSet.beforeFirst();
       while (rowSet.next())
       {
          String eventId = rowSet.getObject("EVENT_ID").toString();
          String selector = (String) rowSet.getObject("EVENT_SELECTOR");
          Boolean isEnabled = (Boolean) rowSet.getObject("ENABLED");
          groupNotificationsListener.addNotification(new NotificationConfigEntryRequest(eventId, "", selector, isEnabled));
          if (eventFilter.length() > 0)
             eventFilter.append("&");
          eventFilter.append(eventId);
       }
       
       groupNotificationsConsumer = createSystemEventAsyncConsumer(connection, defaultNotificationConsumerName + "Group", groupNotificationsListener, eventFilter.toString(), null, EventScope.OBSERVABLE, false);
       groupNotificationsConsumer.start();
    }

    private void stopGroupNotificationsConsumer() throws Exception 
    {
       if (groupNotificationsConsumer != null)
       {
          groupNotificationsConsumer.stop();
          dropSystemConsumer(connection, groupNotificationsConsumer.getName());
          groupNotificationsConsumer = null;          
       }
       
       groupNotificationsListener = null;
    }
    
    private void stopNotificationConsumers()  
    {
       if (notificationsConsumers == null)
          return;
       
       for (String eventId : notificationsConsumers.keySet())
          stopNotificationConsumer(eventId);
       notificationsConsumers.clear();
       notificationsConsumers = null;
    }

    private void stopNotificationConsumer(String eventId) 
    {
       if (groupNotificationsListener != null && groupNotificationsListener.existsNofitication(eventId))
          groupNotificationsListener.removeNotification(eventId);

       if (notificationsConsumers == null)
          return;
       
       EventAsyncConsumer consumer = notificationsConsumers.remove(eventId);
       if (consumer != null)
       {
          consumer.stop();
          try
          {
             dropSystemConsumer(connection, consumer.getName());
          }
          catch (Exception exception)
          {
             ctx.logError("Failed to drop notification consumer " + consumer.getName() + "'. Cause: " + exception.getMessage());
          }
       }
    }

    private void startNotificationConsumer(NotificationConfigEntryRequest entry) throws Exception
    {
       if (groupNotificationsListener != null && groupNotificationsListener.existsNofitication(entry.getEventId()))
       {
          groupNotificationsListener.addNotification(entry);
       }
       else if (notificationsConsumers != null && notificationsConsumers.containsKey(entry.getEventId()))
       {
          notificationsConsumers.get(entry.getEventId()).setEventSelector(entry.getSelector());
       }
       else
       {
          EventAsyncConsumer consumer = createSystemEventAsyncConsumer(connection, defaultNotificationConsumerName + entry.getEventId(), new NotificationListener(), entry.getEventId(), entry.getSelector(), EventScope.OBSERVABLE, false);
          consumer.start();

          if (notificationsConsumers == null)
             notificationsConsumers = new HashMap<String, EventAsyncConsumer>();
          notificationsConsumers.put(entry.getEventId(), consumer);
       }
    }
    
    private void createNotificationListenerEntry(NotificationConfigEntryRequest entry) throws Exception 
    {
       if (entry.getEventId().equals("advisory.ds.notif")) 
          entry.setSelector("name='qspace.ProcessQueue.Suspended' or name='qspace.ProcessQueue.Resumed'");

       try
       {
          tspaceAccessor.executeQuery("INSERT INTO " + escape(NOTIFICATIONS_TABLE_NAME) + " VALUES (?,?,?,?)",
                entry.getEventId(), entry.getEnabled(), entry.getType(), entry.getSelector());
       }
       catch(Exception exception)
       {
          ctx.logError("Error filling in notification config values!. Cause: " + exception.getMessage());
       }
    }

    private void removeNotificationListenerEntry(NotificationConfigEntryRequest entry) throws FabricComponentAccessorException
    {
       SLResponse response = tspaceAccessor.invokeLanguageRequest("DELETE FROM " + escape(NOTIFICATIONS_TABLE_NAME) + " WHERE EVENT_ID='"  + entry.getEventId() + "'");
       printSLResponseIfError("Failed to remove notification entry for " + entry.getEventId(), response);
       stopNotificationConsumer(entry.getEventId());
    }
    
    private void updateNotificationListenerEntry(NotificationConfigEntryRequest entry) throws Exception
    {
       if (entry.getEventId().equals("advisory.ds.notif")) 
          entry.setSelector("name='qspace.ProcessQueue.Suspended' or name='qspace.ProcessQueue.Resumed'");

       try
       {
          tspaceAccessor.executeQuery("UPDATE " + escape(NOTIFICATIONS_TABLE_NAME) + 
                " SET ENABLED=?, EVENT_SELECTOR=? WHERE EVENT_ID=?", entry.getEnabled(), entry.getSelector(),  entry.getEventId());
       }
       catch(Exception exception)
       {
          ctx.logError("Failed to update notification entry for " + entry.getEventId() + ". Cause: " + exception.getMessage());
       }
       startNotificationConsumer(entry);
    }
    
    class GroupNotificationsListener implements FabricEventListener
    {
       private Map<String, Pair<NotificationConfigEntryRequest, SelectorExpression>> notifications = 
             new ConcurrentHashMap<String, Pair<NotificationConfigEntryRequest, SelectorExpression>>();
       private NotificationListener listener = new NotificationListener();
       
       public void addNotification(NotificationConfigEntryRequest notification) throws SelectorFormatException
       {
          SelectorExpression selectorExpression = null;
          if (notification.getSelector() != null && notification.getSelector().trim().length() > 0)
          {
             try
             {
                selectorExpression = SelectorParser.parse(notification.getSelector());
             }
             catch (SelectorFormatException exception)
             {
                Trace.logError(this, "Failed to compile selector expression for notification '" + notification.getEventId() + "'.");
                throw exception;
             }
          }
          notifications.put(notification.getEventId(), 
                new Pair<NotificationConfigEntryRequest, SelectorExpression>(notification, selectorExpression));
       }

       public void removeNotification(String eventId)
       {
          notifications.remove(eventId);
       }

       public boolean existsNofitication(String eventId)
       {
          return notifications.containsKey(eventId);
       }
       
       public void onEvent(ImmutableEventDatagram event) throws FabricEventException
       {
          Pair<NotificationConfigEntryRequest, SelectorExpression> pair = notifications.get(event.getEventId());
          if (pair == null)
             return;
          if (pair.second != null && !pair.second.matches(event))
             return;
          
          listener.onEvent(event);
       }
       
    }
    
    class NotificationListener implements FabricEventListener
    {
       public void onEvent(ImmutableEventDatagram event)
       {
          SLResponse response;
          try
          {
             synchronized(tspaceAccessorForNotificationsMutex)
             {
                response = tspaceAccessorForNotifications.invokeLanguageRequest("SELECT ENABLED FROM "
                   + escape(NOTIFICATIONS_TABLE_NAME) + " WHERE EVENT_ID='" + event.getEventId() + "'");
             }
             if (printSLResponseIfError("Failed to select enabled from notifications table.", response))
                return;
          }
          catch (Exception exception)
          {
             Trace.logError(this, "Unable to process event " + event + ". Cuase: " + exception.getMessage());
             Trace.logException(this, exception, true);
             return;
          }

          RowSet rowSet = response.getRowSet();
          if (rowSet.isEmpty())
             return;
          
          try
          {
             rowSet.beforeFirst();
             boolean isEnabled = false;
             while (rowSet.next())
                isEnabled = (Boolean) rowSet.getObject("ENABLED");

             Trace.logDebug(this, "New event came: " + event.getEventId() + ". Enabled: " + isEnabled);
             if (isEnabled)
                processAdvisory(event);
          }
          catch(SQLException exception)
          {
             Trace.logError(this, "Unable process rowSet from event " + event + ". Cause: " + exception);
             Trace.logException(this, exception, true);
          }
       }
    }
    
    private ImmutableEventDatagram processAdvisory(ImmutableEventDatagram event)
    {
       try
       {
          // If not initialized yet
          if ((ctx == null) || (ctx.getModerator() == null))
             return null;

          String correlationId = null;
          if (event instanceof AdvisoryEvent)
             correlationId = ((AdvisoryEvent)event).getCorrelationId();

          if (event instanceof IAbstractExceptionEvent)
             correlationId = ((IAbstractExceptionEvent)event).getCorrelationId();

          FabricAddress source = new FabricAddress(event.getEventSource());
          ComponentReference component = ctx.getModerator().lookupComponent(source);
          String eventGroup = this.thisNode;

          if (component != null && component.getModel() != null && component.getModel().equals(ComponentModel.SERVICE))
          {
             String serviceName = component.getName().split("//")[1];
             String serviceGroup;
             SLResponse response;
             synchronized(tspaceAccessorForNotificationsMutex)
             {
                response = tspaceAccessorForNotifications.invokeLanguageRequest("SELECT SERVICE_GROUP FROM \""
                   + SERVICE_TYPE_TABLE_NAME + "\" WHERE SERVICE_NAME='" + serviceName + "'");
             }
             if (!response.isOK() || response.getRowSet() == null || !response.getRowSet().first())
             {
                ctx.logError("Unknown service " + serviceName + " while processing advisory. Setting eventGroup to " + this.thisNode);
             } 
             else
             {                    
                serviceGroup = response.getRowSet().getObject(1).toString();
                synchronized(tspaceAccessorForNotificationsMutex)
                {
                   response = tspaceAccessorForNotifications.invokeLanguageRequest("SELECT PROCESS_NAME FROM \""
                      + PROCESS_CONFIGURATION_TABLE_NAME + "\" WHERE SERVICE_GROUP='" + serviceGroup + "'");
                }
                if (!response.isOK() || response.getRowSet() == null || !response.getRowSet().first())
                   ctx.logError("Unknown service group " + serviceGroup + " while processing advisory. Setting eventGroup to " + this.thisNode);
                else 
                   eventGroup = response.getRowSet().getObject(1).toString();
             }

          }
          else if (component != null && component.getModel() != null && component.getModel().equals(ComponentModel.DATASPACE))
          {
             if (event instanceof StateAdvisory) 
             {
                StateAdvisory advisory = (StateAdvisory) event;
                for (String propertyName : advisory.getPropertyNames()) 
                {
                   if (propertyName.equalsIgnoreCase("collection")) 
                   {
                      eventGroup = advisory.getProperty("collection").split("\\.")[0];
                      break;
                   }
                }
             }
          }

          String sourceName;
          if (component != null)
             sourceName = component.getName();
          else if (ctx.getModerator().lookupFabricNode(source) != null)
             sourceName = ctx.getModerator().lookupFabricNode(source).getName();
          else
             sourceName = "Unknown";

          PortalNotification notification = new PortalNotification(sourceName, event);
          notification.setGroup(eventGroup);

          DataEvent ev = SDOUtils.createDataEvent(NOTIFICATION_EVENT_ID, notification);
          if (correlationId == null)
             correlationId = "RsPortalCorrelationId_" + thisNode;

          ev.setCorrelationId(correlationId);
          Trace.logDebug(this, "Notification sent! Date: " + notification.getDate() + ". Details: " + notification.getDetails() + ". Source: " + notification.getSource());
          //Sending notification event
          ctx.raiseEvent(ev, 0);
          return ev;
       } 
       catch (Exception error)
       {
          Trace.logException(RedSquareAgent.class, error, true);
          return null;
       }
    }
    
    public void pingAgent(String request)
    {
       ctx.logInfo("Ping recieved from redsquare manager. Sending agents and processes states.");

       redSquareAgentRegisterSender.needToResend();
       processQueueStateChangeSender.needToResend();

       // don't call not thread safe methods, also they can cause deadlock
       //agentRegisterNotificationWorker.doExecute();
       //serviceStateCheckWorker.doExecute();
    }
    
    //////////////////////////////////
    
    private class ServiceStateCheckWorker extends MonitorWorker
    {
       protected DataspaceAccessor accessor = null;

       protected ServiceStateCheckWorker(String name, String description, long timeout)
             throws FabricException, FabricConnectionException, FabricComponentAccessorException
       {
          super(name, description, timeout);
          accessor = connection.createDataspaceAccessor(DataspaceType.TSPACE, tableSpaceName);
       }

       protected void doStop()
       {
          super.doStop();
          if (accessor != null)
          {
             accessor.close();
             accessor = null;
          }
       }
       
       public void doExecute()
       {
          try
          {
             SLResponse response = accessor.invokeLanguageRequest(
                   "SELECT SERVICE_NAME,  SERVICE_GROUP, SERVICE_STATE FROM " + escape(SERVICE_TYPE_TABLE_NAME));
             if (printSLResponseIfError("Unable to get services from service table", response))
                return;

             RowSet rowSet = response.getRowSet();
             while (rowSet.next())
             {
                String serviceName =      (String) rowSet.getObject("SERVICE_NAME");
                String prevServiceState = (String) rowSet.getObject("SERVICE_STATE");
                String serviceGroup =     (String) rowSet.getObject("SERVICE_GROUP");

                ComponentState currentServiceState = context.getServiceManager().getServiceState(serviceName);
                if (currentServiceState != null)
                {
                   if (!currentServiceState.name().equals(prevServiceState))
                   {
                      response = accessor.invokeLanguageRequest("UPDATE " + escape(SERVICE_TYPE_TABLE_NAME) + " SET SERVICE_STATE='"
                            + currentServiceState + "'" + "WHERE SERVICE_NAME='" + serviceName + "' AND SERVICE_GROUP='" + serviceGroup + "'");

                      Trace.logDebug(this, "Updating service state for " + serviceName + " to " + currentServiceState);
                      try
                      {
                         sendServiceStateChangeEvent(serviceGroup, serviceName, currentServiceState.name());
                      }
                      catch (Exception e)
                      {
                         Trace.logError(this, "Unable to send ServiceStateChangeEvent " + e.getMessage());
                         Trace.logException(this, e, true);
                      }
                   }
                }
                else
                {
                   if (!prevServiceState.equalsIgnoreCase("STOPPED"))
                   {
                      ctx.logDebug("Can not get information about " + serviceName + " service, setting state to STOPPED");
                      response = accessor.invokeLanguageRequest("UPDATE "
                            + escape(SERVICE_TYPE_TABLE_NAME) + "    SET SERVICE_STATE='STOPPED' "
                            + "WHERE SERVICE_NAME='" + serviceName + "' AND " + "SERVICE_GROUP='" + serviceGroup + "'");
                      Trace.logDebug(this, "Updated service state for " + serviceName + " to STOPPED");
                      try
                      {
                         sendServiceStateChangeEvent(serviceGroup, serviceName, "UNKNOWN");
                      }
                      catch (Exception e)
                      {
                         Trace.logError(this, "Unable to send ServiceStateChangeEvent " + e.getMessage());
                         Trace.logException(this, e, true);
                      }
                   }
                }
             }
          }
          catch (Throwable ex)
          {
             Trace.logException(this, ex, true);
          }
       }
       
       private void sendServiceStateChangeEvent(String groupName, String serviceName, String state) throws Exception
       {
          Trace.logDebug(this, "Sending service state change event!");
          DataEvent statusEvent = (DataEvent) datagramFactory .createEvent(RED_SQUARE_SERVICE_STATE_CHANGE_EVENT_ID);
          ServiceStateChange event = new ServiceStateChange();
          event.setAgentName(thisNode);
          event.setServiceGroup(groupName);
          event.setServiceName(serviceName);
          event.setServiceState(state);
          statusEvent.setData(event);

          ctx.raiseEvent(statusEvent, 0);
       }
    }
    
    //////////////////////////////////

    private class AgentRegisterNotificationWorker extends MonitorWorker
    {
       protected DataspaceAccessor accessor = null;
       protected DataspaceAccessor qAccessor = null;
       
       protected AgentRegisterNotificationWorker(String name, String description, long timeout) throws FabricException, FabricConnectionException, FabricComponentAccessorException 
       {
          super(name, description, timeout);
          accessor = connection.createDataspaceAccessor(DataspaceType.TSPACE, tableSpaceName);
          qAccessor = connection.createDataspaceAccessor(DataspaceType.QSPACE, queueSpaceName);
       }

       protected void doStop()
       {
          super.doStop();

          redSquareAgentRegisterSender.send(AgentState.STOPPED, true);
          
          if (accessor != null)
          {
             accessor.close();
             accessor = null;
          }

          if (qAccessor != null)
          {
             qAccessor.close();
             qAccessor = null;
          }
       }
       
       public void doExecute()
       {
          try
          {
             AgentState state = AgentState.RUNNING;
             SLResponse response;
             try
             {
                response = accessor.invokeLanguageRequest("SELECT * FROM " + escape(PROCESS_CONFIGURATION_TABLE_NAME));
             }
             catch (FabricComponentAccessorException e1)
             {
                Trace.logDebug(RedSquareAgent.class, "Unable to get processTypes");
                return;
             }

             if (printSLResponseIfError("Unable to get processTypes", response))
                return;

             RowSet rowSet = response.getRowSet();
             while (rowSet.next())
             {
                ProcessQueueState queueState = null;

                String procGroup = (String) rowSet.getObject("SERVICE_GROUP");
                String procName = (String) rowSet.getObject("PROCESS_NAME");

                Trace.logDebug(this, "Process name: " + procName);
                ProcessDeploymentState provState = ProcessDeploymentState.valueOf((String) rowSet.getObject("PROCESS_STATE"));

                Trace.logDebug(this, "Status is: " + provState);
                if (provState == ProcessDeploymentState.UNDEPLOYED)
                {
                   processQueueStateChangeSender.send(procName, ProcessQueueState.UNKNOWN);
                   continue;
                }

                Trace.logDebug(this, "Getting Queue State for " + procName + processQueueSuffix);
                try
                {
                   String processState = getProcessState(qAccessor, procName);
                   if (processState == null)
                      queueState = ProcessQueueState.UNKNOWN;
                   else
                      queueState = ProcessQueueState.valueOf(processState);
                }
                catch (FabricComponentAccessorException e)
                {
                   ctx.logError("Unable to get Process State for " + procName);
                }

                if (queueState == ProcessQueueState.SUSPENDED)
                {
                   processQueueStateChangeSender.send(procName, queueState);
                   continue;
                }

                try
                {
                   response = accessor.invokeLanguageRequest("SELECT " + "    SERVICE_NAME, " + "    SERVICE_GROUP "
                         + "FROM " + escape(SERVICE_TYPE_TABLE_NAME) + "WHERE "
                         + "    SERVICE_GROUP='" + procGroup + "' AND " + "    SERVICE_STATE <> 'INITIALIZING' AND "
                         + "    SERVICE_STATE <> 'INITIALIZED' AND " + "    SERVICE_STATE <> 'STARTING' AND "
                         + "    SERVICE_STATE <> 'STARTED'");
                }
                catch (FabricComponentAccessorException e)
                {
                   Trace.logDebug(this, "Unable to get Service state info for " + procGroup);
                   continue;
                }

                if (printSLResponseIfError("Unable to get Service state info for " + procGroup, response))
                   continue;

                RowSet serviceRowSet = response.getRowSet();
                if (!serviceRowSet.first())
                {
                   Trace.logDebug(this, "No services with bad states sending running event!");
                   processQueueStateChangeSender.send(procName, ProcessQueueState.RUNNING);
                }
                else
                {
                   Trace.logDebug(this, "Found some NOK services for " + procGroup);
                   processQueueStateChangeSender.send(procName, ProcessQueueState.SUSPECT);
                   state = AgentState.SUSPECT;
                }
             }

             // checking agents without any configured processes too
             try
             {
                response = accessor.invokeLanguageRequest("SELECT " + "    SERVICE_NAME, " + "    SERVICE_GROUP "
                      + "FROM " + escape(SERVICE_TYPE_TABLE_NAME) + " WHERE "
                      + "    SERVICE_STATE <> 'INITIALIZING' AND " + "    SERVICE_STATE <> 'INITIALIZED' AND "
                      + "    SERVICE_STATE <> 'STARTING' AND " + "    SERVICE_STATE <> 'STARTED'");
             }
             catch (FabricComponentAccessorException e1)
             {
                Trace.logError(this, "Unable to get Service state info.");
                Trace.logException(this, e1, true);
             }

             if (!printSLResponseIfError("Unable to get Service state info.", response))
             {
                RowSet serviceRowSet = response.getRowSet();
                if (serviceRowSet.first())
                   state = AgentState.SUSPECT;
             }

             redSquareAgentRegisterSender.send(state);
             sendProcessInstanceChangeEvent();
          }
          catch (Throwable error)
          {
             Trace.logException(this, error, true);
          }
       }

       private void sendProcessInstanceChangeEvent()
       {
          if (needToNotifyAboutProcessInstanceChanges)
          {
             Trace.logDebug(RedSquareAgent.class, "Sending instance change event!");
             try
             {
                DataEvent event = (DataEvent) datagramFactory.createEvent(PROCESS_INSTANCE_CHANGE_EVENT_ID);
                event.setData(new ProcessListChangeResponse(context.getName()));
                ctx.raiseEvent(event, 0);
             }
             catch (Exception e)
             {
                Trace.logDebug(this, "Unable to send process instance change event!");
                Trace.logException(RedSquareAgent.class, e, true);
             }
             finally
             {
                needToNotifyAboutProcessInstanceChanges = false;
             }
          }
       }
       

       protected String getProcessState(DataspaceAccessor accessor, String processType) throws FabricComponentAccessorException, SQLException
       {
          SLResponse response = accessor.invokeLanguageRequest("SHOW QUEUE STATE FOR " + escape(processType + processQueueSuffix));
          if (printSLResponseIfError("Unable to get process state.", response))
             return null;

          RowSet rowSet = response.getRowSet();
          rowSet.beforeFirst();
          if (rowSet.next())
             return (String)rowSet.getObject(1);

          return null;
       }
    }
    
    ///////////////////////
    
    class RedSquareAgentRegisterSender
    {
       private AgentState previousAgentState = null;
       private long previousAgentStateSendTime = 0;
       private volatile boolean needToResendAgentState = false;
       
       void send(AgentState state)
       {
          send(state, false);
       }
       
       synchronized void send(AgentState state, boolean now)
       {
          if (needToResendAgentState || now)
          {
             needToResendAgentState = true;
             previousAgentState = null;
          }
          
          if (state == previousAgentState && System.currentTimeMillis() - previousAgentStateSendTime < 60000)
             return;
          try
          {
             DataEvent statusEvent = (DataEvent) datagramFactory.createEvent(RED_SQUARE_AGENT_REGISTER_EVENT_ID);
             AgentRegisterEvent event = new AgentRegisterEvent(context.getName());
             event.setNodeName(thisNode);
             event.setState(state);
             statusEvent.setData(event);
             ctx.raiseEvent(statusEvent, 0);
             
             previousAgentStateSendTime = System.currentTimeMillis();
             previousAgentState = state;
          }
          catch (Exception stateEx)
          {
             Trace.logDebug(this, "Unable to send agent state change event!");
             Trace.logException(RedSquareAgent.class, stateEx, true);
             previousAgentState = null;
          }
       }
       
       public void needToResend()
       {
          needToResendAgentState = true;
       }
    }
    
    ///////////////////////

    class ProcessQueueStateChangeSender
    {
       private Map<String, Pair<ProcessQueueState, Long>> states = new ConcurrentHashMap<String, Pair<ProcessQueueState, Long>>();
       
       public void needToResend()
       {
          states.clear();
       }
       
       public void send(String processName, ProcessQueueState queueState)
       {
          send(processName, queueState, null);
       }
       
       public synchronized void send(String processName, ProcessQueueState queueState, String organization)
       {
          Pair<ProcessQueueState, Long> pair = states.get(processName);
          if (pair != null && pair.first == queueState && System.currentTimeMillis() - pair.second < agentStateDuplicateSendInterval)
             return;
          
          try
          {
             DataEvent statusEvent = (DataEvent) datagramFactory.createEvent(PROCESS_STATE_CHANGE_EVENT_ID);
             ProcessStateChangeResponse data = new ProcessStateChangeResponse(processName, queueState, thisNode, ctx.getType() + "." + ctx.getName());
             data.setOrganization(organization);
             statusEvent.setData(data);
             ctx.raiseEvent(statusEvent, 0);
             
             if (queueState != ProcessQueueState.DELETED)
             {             
                pair = new Pair<ProcessQueueState, Long>(queueState, System.currentTimeMillis());
                states.put(processName, pair);
             }
             else
                states.remove(processName);
          }
          catch (Exception stateEx)
          {
             Trace.logDebug(this, "Unable to send process state change event!");
             Trace.logException(RedSquareAgent.class, stateEx, true);
             
             states.remove(processName);
          }
       }
       
    }
    
 }
